<?php

use App\Http\Controllers\AdminUserController;
use App\Http\Controllers\AuthController;
use App\Http\Controllers\BrandController;
use App\Http\Controllers\CategoryController;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\ProductPricingController;
use App\Http\Controllers\VendorController;
use App\Http\Controllers\CustomerProfileController;
use App\Http\Controllers\CustomerAddressController;
use App\Http\Controllers\WishlistController;
use App\Http\Controllers\CartController;
use App\Http\Controllers\CheckoutController;
use App\Http\Controllers\OrderController;
use App\Http\Controllers\AdminOrderController;
use App\Http\Controllers\VendorOrderController;
use App\Http\Controllers\ImageController;
use App\Http\Controllers\SystemController;
use App\Http\Controllers\InvoiceController;
use App\Http\Controllers\PaymentController;
use App\Http\Controllers\SearchController;
use App\Http\Controllers\EmailSubscriptionController;
use App\Http\Controllers\VendorProfileController;
use App\Http\Controllers\VendorAddressController;
use App\Http\Controllers\TestEmailController;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application.
| Routes are loaded by the RouteServiceProvider and assigned to the "api" middleware group.
|
*/

// API version prefix
Route::prefix('v1')->group(function () {
    /*
    |--------------------------------------------------------------------------
    | Public Routes
    |--------------------------------------------------------------------------
    */

    // Universal Search
    Route::get('/search', [SearchController::class, 'search']);

    // Email Subscription
    Route::post('/subscribe', [EmailSubscriptionController::class, 'store']);

    // Test Email Route
    Route::get('/test-email', [TestEmailController::class, 'send']);

    // Public routes that don't require authentication
    // Public Product Routes
Route::prefix('products')->group(function () {
    Route::get('/', [ProductController::class, 'index']);
    Route::get('/homepage-sections', [ProductController::class, 'homepageSections']);
    Route::get('/{id}', [ProductController::class, 'show'])->where('id', '[0-9]+');
    Route::get('/{id}/public', [ProductController::class, 'publicProductView'])->where('id', '[0-9]+');
    Route::get('/public/all/view', [ProductController::class, 'publicProductViewAll']);
    Route::get('/brand/{brandId}', [ProductController::class, 'getProductsByBrand']);
    Route::get('/category/{categoryId}', [ProductController::class, 'getProductsByCategory']);
    Route::get('/category/{categoryId}/sorted', [ProductController::class, 'getSortedProductsByCategory'])->where('categoryId', '[0-9]+');
    Route::get('/vendor/{vendorId}', [ProductController::class, 'getProductsByVendor']);
    Route::get('/vendor-products/{vendorId}', [ProductController::class, 'getProductsByVendor']);
});

// Public Category Routes
Route::prefix('categories')->group(function () {
    Route::get('/', [CategoryController::class, 'index']);
    Route::get('/tree', [CategoryController::class, 'tree']);
    Route::get('/featured', [CategoryController::class, 'featured']);
    Route::get('/{id}', [CategoryController::class, 'show'])->where('id', '[0-9]+');
    Route::get('/{id}/products', [CategoryController::class, 'products'])->where('id', '[0-9]+');
});

// Public Brand Routes
Route::prefix('brands')->group(function () {
    Route::get('/', [BrandController::class, 'index']);
    Route::get('/{id}', [BrandController::class, 'show'])->where('id', '[0-9]+');
});

// Public Vendor Routes
Route::prefix('vendors')->group(function () {
    Route::get('/', [VendorController::class, 'publicIndex']);
    Route::get('/{id}', [VendorController::class, 'publicShow'])->where('id', '[0-9]+');
});

    // Public Payment Intent Info Route
    Route::get('/payments/intent/{paymentIntentId}/info', [PaymentController::class, 'getPaymentIntentInfo']);

    // Public SEO Metadata Routes (root level, not inside any other prefix)
    Route::get('/seometa/products/{id}', [ProductController::class, 'getMeta'])->where('id', '[0-9]+');
    Route::get('/seometa/categories/{id}', [CategoryController::class, 'getMeta'])->where('id', '[0-9]+');
    Route::get('/seometa/brands/{id}', [BrandController::class, 'getMeta'])->where('id', '[0-9]+');

    // Superadmin SEO Metadata Routes
    Route::middleware(['auth:sanctum', 'role:superadmin'])->prefix('superadmin/seometa')->group(function () {
        // GET routes for retrieving meta data
        Route::get('/products/{id}', [ProductController::class, 'getMeta'])->where('id', '[0-9]+');
        Route::get('/categories/{id}', [CategoryController::class, 'getMeta'])->where('id', '[0-9]+');
        Route::get('/brands/{id}', [BrandController::class, 'getMeta'])->where('id', '[0-9]+');

        // PUT routes for updating meta data
        Route::put('/products/{id}', [ProductController::class, 'updateMeta'])->where('id', '[0-9]+');
        Route::put('/categories/{id}', [CategoryController::class, 'updateMeta'])->where('id', '[0-9]+');
        Route::put('/brands/{id}', [BrandController::class, 'updateMeta'])->where('id', '[0-9]+');
    });

    // Supervisor SEO Metadata Routes
    Route::middleware(['auth:sanctum', 'role:supervisor'])->prefix('supervisor/seometa')->group(function () {
        // GET routes for retrieving meta data
        Route::get('/products/{id}', [ProductController::class, 'getMeta'])->where('id', '[0-9]+');
        Route::get('/categories/{id}', [CategoryController::class, 'getMeta'])->where('id', '[0-9]+');
        Route::get('/brands/{id}', [BrandController::class, 'getMeta'])->where('id', '[0-9]+');

        // PUT routes for updating meta data
        Route::put('/products/{id}', [ProductController::class, 'updateMeta'])->where('id', '[0-9]+');
        Route::put('/categories/{id}', [CategoryController::class, 'updateMeta'])->where('id', '[0-9]+');
        Route::put('/brands/{id}', [BrandController::class, 'updateMeta'])->where('id', '[0-9]+');
    });

    /*
    |--------------------------------------------------------------------------
    | Protected Routes (Authentication Required)
    |--------------------------------------------------------------------------
    */

    Route::middleware('auth:sanctum')->group(function () {
        // User Authentication Management
        Route::prefix('auth')->group(function () {
            Route::post('/logout', [AuthController::class, 'logout']);
            Route::get('/me', [AuthController::class, 'me']);
            Route::post('/change-password', [AuthController::class, 'changePassword']);
        });

        // Image Upload Routes
        Route::prefix('images')->group(function () {
            Route::post('/upload', [ImageController::class, 'upload']);
            Route::get('/show', [ImageController::class, 'show']);
            Route::options('/show', [ImageController::class, 'handleCors']);
        });

        /*
        |--------------------------------------------------------------------------
        | Vendor Routes
        |--------------------------------------------------------------------------
        */
        Route::middleware('role:vendor')->prefix('vendor')->group(function () {
            // Vendor Product Management
            Route::prefix('products')->group(function () {
                Route::get('/', [ProductController::class, 'vendorProducts']);
                Route::post('/', [ProductController::class, 'store']);
                Route::get('/{id}', [ProductController::class, 'vendorShow'])->where('id', '[0-9]+');
                Route::put('/{id}', [ProductController::class, 'update'])->where('id', '[0-9]+');
                Route::patch('/{id}', [ProductController::class, 'update'])->where('id', '[0-9]+');
                Route::delete('/{id}', [ProductController::class, 'destroy'])->where('id', '[0-9]+');
                Route::delete('/{id}/permanent', [ProductController::class, 'hardDelete'])->where('id', '[0-9]+');

                // Category management routes
                Route::put('/{id}/categories', [ProductController::class, 'updateProductCategories'])->where('id', '[0-9]+');
                Route::delete('/{id}/categories', [ProductController::class, 'removeProductCategory'])->where('id', '[0-9]+');

                // Product Pricing Management
                Route::get('/{productId}/pricing', [ProductPricingController::class, 'getPricing'])->where('productId', '[0-9]+');
                Route::put('/{productId}/pack-price', [ProductPricingController::class, 'updatePackPrice'])->where('productId', '[0-9]+');
                Route::put('/{productId}/default-margins', [ProductPricingController::class, 'updateDefaultMargins'])->where('productId', '[0-9]+');

                // Bulk Pricing Management
                Route::post('/{productId}/bulk-prices', [ProductPricingController::class, 'createBulkPrice'])->where('productId', '[0-9]+');
                Route::put('/{productId}/bulk-prices', [ProductPricingController::class, 'updateAllBulkPrices'])->where('productId', '[0-9]+');
                Route::put('/{productId}/bulk-prices/order', [ProductPricingController::class, 'updateBulkPriceOrder'])->where('productId', '[0-9]+');
                Route::put('/bulk-prices/{id}', [ProductPricingController::class, 'updateBulkPrice'])->where('id', '[0-9]+');
                Route::delete('/bulk-prices/{id}', [ProductPricingController::class, 'deleteBulkPrice'])->where('id', '[0-9]+');

                // Product Features (Flexible Specs)
                Route::get('/{id}/features', [ProductController::class, 'getFeatures'])->where('id', '[0-9]+');
                Route::put('/{id}/features', [ProductController::class, 'updateFeatures'])->where('id', '[0-9]+');
            });

            // Vendor Address Management
            Route::prefix('addresses')->group(function () {
                Route::get('/', [VendorAddressController::class, 'index']);
                Route::post('/', [VendorAddressController::class, 'store']);
                Route::get('/current', [VendorAddressController::class, 'show']);
                Route::put('/current', [VendorAddressController::class, 'update']);
                Route::delete('/current', [VendorAddressController::class, 'destroy']);
                Route::patch('/current/set-default', [VendorAddressController::class, 'setDefault']);
            });

            // Vendor Profile Route
            Route::get('/profile', [VendorProfileController::class, 'getVendorProfile']);
            Route::put('/profile', [VendorProfileController::class, 'update']);
            Route::put('/address', [VendorController::class, 'updateAddress']);
            Route::get('/orders', [VendorOrderController::class, 'index']);
            Route::get('/orders/{id}', [VendorOrderController::class, 'show']);
            Route::put('/orders/{id}/fulfillment', [VendorOrderController::class, 'updateFulfillmentStatus']);
            Route::post('/orders/{id}/tracking', [VendorOrderController::class, 'addTrackingUpdate']);
        });

        /*
        |--------------------------------------------------------------------------
        | Buyer Routes
        |--------------------------------------------------------------------------
        */
        Route::middleware('role:buyer')->prefix('buyer')->group(function () {
            // Buyer specific routes
            // Add routes for cart, orders, favorites, etc.
        });

        /*
        |--------------------------------------------------------------------------
        | Admin Routes (SuperAdmin, Admin, Supervisor)
        |--------------------------------------------------------------------------
        */
        Route::middleware('role:superadmin,admin,supervisor')->prefix('admin')->group(function () {
            // Product Management
            Route::prefix('products')->group(function () {
                Route::get('/', [ProductController::class, 'index']);
                Route::post('/', [ProductController::class, 'store']);
                Route::get('/{id}', [ProductController::class, 'show'])->where('id', '[0-9]+');
                Route::get('/{id}/any-status', [ProductController::class, 'showAnyStatus'])->where('id', '[0-9]+');
                Route::put('/{id}', [ProductController::class, 'update'])->where('id', '[0-9]+');
                Route::patch('/{id}', [ProductController::class, 'update'])->where('id', '[0-9]+');
                Route::delete('/{id}', [ProductController::class, 'destroy'])->where('id', '[0-9]+');
                Route::delete('/{id}/permanent', [ProductController::class, 'hardDelete'])->where('id', '[0-9]+');
                Route::patch('/{id}/toggle-approval', [ProductController::class, 'toggleApproval'])->where('id', '[0-9]+');
                Route::post('/bulk-action', [ProductController::class, 'bulkAction']);
                Route::patch('/{id}/update-attributes', [ProductController::class, 'updateAnyAttribute'])->where('id', '[0-9]+');
                Route::patch('/category/{categoryId}/sort-order', [ProductController::class, 'updateProductSortOrder'])->where('categoryId', '[0-9]+');

                // Category management routes
                Route::put('/{id}/categories', [ProductController::class, 'updateProductCategories'])->where('id', '[0-9]+');
                Route::delete('/{id}/categories', [ProductController::class, 'removeProductCategory'])->where('id', '[0-9]+');

                // New ranking management routes
                Route::patch('/category/{categoryId}/product/{productId}/rank', [ProductController::class, 'updateProductRanking'])
                    ->where(['categoryId' => '[0-9]+', 'productId' => '[0-9]+']);
                Route::patch('/category/{categoryId}/rankings', [ProductController::class, 'updateProductsRanking'])
                    ->where('categoryId', '[0-9]+');
                Route::put('/bulk-assign-brand', [ProductController::class, 'bulkAssignBrand']);
                Route::delete('/{id}/brand', [ProductController::class, 'removeProductBrand'])->where('id', '[0-9]+');
                // Allow superadmin/admin to update bulk prices
                Route::put('/{productId}/bulk-prices', [ProductPricingController::class, 'updateAllBulkPrices'])->where('productId', '[0-9]+');
            });

            // Brand Management
            Route::prefix('brands')->group(function () {
                Route::get('/', [BrandController::class, 'index']);
                Route::post('/', [BrandController::class, 'store']);
                Route::get('/{id}', [BrandController::class, 'show'])->where('id', '[0-9]+');
                Route::put('/{id}', [BrandController::class, 'update'])->where('id', '[0-9]+');
                Route::patch('/{id}', [BrandController::class, 'update'])->where('id', '[0-9]+');
                Route::delete('/{id}', [BrandController::class, 'destroy'])->where('id', '[0-9]+');
            });
        });

        /*
        |--------------------------------------------------------------------------
        | SuperAdmin & Admin Routes
        |--------------------------------------------------------------------------
        */
        Route::middleware('role:superadmin,admin')->prefix('admin')->group(function () {
            // User Management
            Route::prefix('users')->group(function () {
                Route::get('/', [AdminUserController::class, 'index']);
                Route::post('/', [AdminUserController::class, 'store']);
                Route::get('/{id}', [AdminUserController::class, 'show'])->where('id', '[0-9]+');
                Route::put('/{id}', [AdminUserController::class, 'update'])->where('id', '[0-9]+');
                Route::patch('/{id}', [AdminUserController::class, 'update'])->where('id', '[0-9]+');
                Route::delete('/{id}', [AdminUserController::class, 'destroy'])->where('id', '[0-9]+');
            });

            // Vendor Management
            Route::prefix('vendors')->group(function () {
                Route::get('/', [VendorController::class, 'index']);
                Route::patch('/{id}/approve', [VendorController::class, 'approve'])->where('id', '[0-9]+');
                Route::patch('/{id}/reject', [VendorController::class, 'reject'])->where('id', '[0-9]+');
            });

            // Product Ranking Management (move here)
            Route::prefix('products')->group(function () {
                Route::patch('/category/{categoryId}/product/{productId}/rank', [ProductController::class, 'updateProductRanking'])
                    ->where(['categoryId' => '[0-9]+', 'productId' => '[0-9]+']);
                Route::patch('/category/{categoryId}/rankings', [ProductController::class, 'updateProductsRanking'])
                    ->where('categoryId', '[0-9]+');
            });
        });

        /*
        |--------------------------------------------------------------------------
        | Multiple Role Routes (Available to Multiple User Types)
        |--------------------------------------------------------------------------
        */
        Route::middleware('role:superadmin,admin,supervisor,vendor')->group(function () {
            // Category Management
            Route::prefix('categories')->group(function () {
                Route::post('/', [CategoryController::class, 'store']);
                Route::post('/bulk-create', [CategoryController::class, 'bulkCreate']);
                Route::post('/bulk-update', [CategoryController::class, 'bulkUpdate']);
                Route::put('/{id}', [CategoryController::class, 'update'])->where('id', '[0-9]+');
                Route::patch('/{id}', [CategoryController::class, 'update'])->where('id', '[0-9]+');
                Route::delete('/{id}', [CategoryController::class, 'destroy'])->where('id', '[0-9]+');
                Route::patch('/{id}/status', [CategoryController::class, 'updateStatus'])->where('id', '[0-9]+');
                Route::post('/bulk-action', [CategoryController::class, 'bulkAction']);
                Route::patch('/{id}/sort-order', [CategoryController::class, 'update'])->where('id', '[0-9]+');
                Route::patch('/bulk/sort-order', [CategoryController::class, 'bulkSortOrder']);
            });
        });

        /*
        |--------------------------------------------------------------------------
        | SuperAdmin Only Routes
        |--------------------------------------------------------------------------
        */
        Route::middleware('role:superadmin')->prefix('superadmin')->group(function () {
            // System Maintenance Routes
            Route::prefix('system')->group(function () {
                Route::get('/info', [SystemController::class, 'getSystemInfo']);
                Route::get('/database-status', [SystemController::class, 'getDatabaseStatus']);
                Route::post('/clear-cache', [SystemController::class, 'clearCache']);
                Route::post('/clear-specific-cache', [SystemController::class, 'clearSpecificCache']);
                Route::post('/clear-temp-files', [SystemController::class, 'clearTempFiles']);
                Route::post('/optimize', [SystemController::class, 'optimize']);
            });

            // Email Subscribers Management
            Route::get('/subscribers', [EmailSubscriptionController::class, 'index']);

            // Currency Management
            // Route::apiResource('currencies', CurrencyController::class);
        });

        // Customer Profile Update Route
        Route::put('/customer-profile/update', [CustomerProfileController::class, 'update']);

        // Notification Endpoints
        Route::get('/notifications', [\App\Http\Controllers\NotificationController::class, 'index']);
        Route::get('/notifications/unread', [\App\Http\Controllers\NotificationController::class, 'unread']);
        Route::post('/notifications/{id}/read', [\App\Http\Controllers\NotificationController::class, 'markAsRead']);
        Route::post('/notifications/{id}/unread', [\App\Http\Controllers\NotificationController::class, 'markAsUnread']);
        Route::delete('/notifications/{id}', [\App\Http\Controllers\NotificationController::class, 'destroy']);
    });

    // Customer Routes - Public
    Route::prefix('auth')->group(function () {
        Route::post('/register', [AuthController::class, 'register']);
        Route::post('/login', [AuthController::class, 'login']);
    });

    // Customer Routes - Protected
    Route::middleware('auth:sanctum')->group(function () {
        // Customer Profile Management
        Route::prefix('customer')->group(function () {
            // Profile
            Route::get('/profile', [CustomerProfileController::class, 'show']);
            Route::post('/profile', [CustomerProfileController::class, 'store']);

            // Addresses
            Route::get('/addresses', [CustomerAddressController::class, 'index']);
            Route::post('/addresses', [CustomerAddressController::class, 'store']);
            Route::get('/addresses/{id}', [CustomerAddressController::class, 'show']);
            Route::put('/addresses/{id}', [CustomerAddressController::class, 'update']);
            Route::delete('/addresses/{id}', [CustomerAddressController::class, 'destroy']);
            Route::patch('/addresses/{id}/set-default', [CustomerAddressController::class, 'setDefault']);
            Route::get('/addresses/default', [CustomerAddressController::class, 'getDefault']);

            // Wishlists
            Route::get('/wishlists', [WishlistController::class, 'index']);
            Route::post('/wishlists', [WishlistController::class, 'store']);
            Route::get('/wishlists/{id}', [WishlistController::class, 'show']);
            Route::put('/wishlists/{id}', [WishlistController::class, 'update']);
            Route::delete('/wishlists/{id}', [WishlistController::class, 'destroy']);
            Route::post('/wishlists/{id}/products', [WishlistController::class, 'addProduct']);
            Route::delete('/wishlists/{wishlistId}/products/{productId}', [WishlistController::class, 'removeProduct']);
        });
    });

    // Admin Customer Management Routes
    Route::middleware('role:superadmin,admin,supervisor')->prefix('admin')->group(function () {
        Route::get('/customers', [CustomerProfileController::class, 'index']);
        Route::get('/customers/{id}', [CustomerProfileController::class, 'adminView']);
        Route::patch('/customers/{id}/verify', [CustomerProfileController::class, 'verify'])->middleware('role:superadmin,admin');
    });

    Route::middleware(['auth:sanctum', 'role:superadmin'])->prefix('superadmin')->group(function () {
        Route::get('/products', [ProductController::class, 'superadminProducts']);
        Route::get('/dashboard/metrics', [\App\Http\Controllers\SuperAdminDashboardController::class, 'metrics']);
    });

    /*
    |--------------------------------------------------------------------------
    | Cart Routes (Authenticated)
    |--------------------------------------------------------------------------
    */
    Route::middleware('auth:sanctum')->group(function () {
        Route::post('/carts', [CartController::class, 'create']);
        Route::get('/carts', [CartController::class, 'index']);
        Route::get('/carts/{cart}', [CartController::class, 'show']);
        Route::post('/carts/{cart}/items', [CartController::class, 'addItem']);
        Route::put('/carts/{cart}/items/{item}', [CartController::class, 'updateItem']);
        Route::delete('/carts/{cart}/items/{item}', [CartController::class, 'removeItem']);
        Route::delete('/carts/{cart}', [CartController::class, 'destroy']);
    });

    // Direct Order Processing (for React frontend)
    Route::middleware('auth:sanctum')->post('/checkout/process-order', [CheckoutController::class, 'processOrder']);

    /*
    |--------------------------------------------------------------------------
    | Checkout Routes (Authenticated)
    |--------------------------------------------------------------------------
    */
    Route::middleware('auth:sanctum')->prefix('checkout')->group(function () {
        Route::post('/initialize', [CheckoutController::class, 'initializeCheckout']);
        Route::post('/process', [CheckoutController::class, 'processOrder']);
    });

    /*
    |--------------------------------------------------------------------------
    | Order Routes (Authenticated)
    |--------------------------------------------------------------------------
    */
    Route::middleware('auth:sanctum')->prefix('orders')->group(function () {
        Route::get('/', [OrderController::class, 'index']);
        Route::get('/{id}', [OrderController::class, 'show']);
    });

    /*
    |--------------------------------------------------------------------------
    | Admin Order Management
    |--------------------------------------------------------------------------
    */
    Route::middleware(['auth:sanctum', 'role:superadmin,admin,supervisor'])->prefix('admin/orders')->group(function () {
        Route::get('/', [AdminOrderController::class, 'index']);
        Route::get('/{id}', [AdminOrderController::class, 'show']);
        Route::patch('/{id}/status', [AdminOrderController::class, 'updateStatus']);
        Route::patch('/{id}/payment-status', [AdminOrderController::class, 'updatePaymentStatus']);
        Route::patch('/{id}/shipping-status', [AdminOrderController::class, 'updateShippingStatus']);
    });

    /*
    |--------------------------------------------------------------------------
    | Vendor Order Routes
    |--------------------------------------------------------------------------
    */
    Route::middleware(['auth:sanctum', 'role:vendor'])->prefix('vendor')->group(function () {
        // Vendor Order Management
        Route::prefix('orders')->group(function () {
            Route::get('/', [VendorOrderController::class, 'index']);
            Route::get('/{id}', [VendorOrderController::class, 'show'])->where('id', '[0-9]+');
            Route::patch('/{id}/fulfillment', [VendorOrderController::class, 'updateFulfillmentStatus'])->where('id', '[0-9]+');
            Route::get('/{id}/fulfillment', [VendorOrderController::class, 'getFulfillmentStatus'])->where('id', '[0-9]+');
            Route::post('/{id}/tracking', [VendorOrderController::class, 'addTrackingUpdate'])->where('id', '[0-9]+');
            Route::post('/{id}/delivery-attempt', [VendorOrderController::class, 'recordDeliveryAttempt'])->where('id', '[0-9]+');
        });
    });

    Route::middleware(['auth:sanctum', 'role:vendor'])->prefix('vendor')->group(function () {
        Route::prefix('products')->group(function () {
            // Add this line
            Route::post('/bulk-import', [ProductController::class, 'bulkImport']);
            // Existing routes...
        });
    });

    Route::middleware(['auth:sanctum', 'role:superadmin,admin'])->prefix('superadmin')->group(function () {
        // Routes for product metadata management
        Route::get('/products/{id}/metadata', [ProductController::class, 'getProductMetadata']);
        Route::put('/products/{id}/brand', [ProductController::class, 'updateProductBrand']);
        Route::put('/products/{id}/categories', [ProductController::class, 'updateProductCategories']);
        Route::put('/products/{id}/metadata', [ProductController::class, 'updateProductMetadata']);
        Route::get('/products/{id}/public', [ProductController::class, 'publicProductView']);
        Route::get('/products/public/all', [ProductController::class, 'publicProductViewAll']);
        // Superadmin Get All Customers Route
        Route::get('/customers', [CustomerProfileController::class, 'getAllCustomers']);
    });

    // Place superadmin brand ranking routes here
    Route::middleware(['auth:sanctum', 'role:superadmin,admin'])->prefix('admin')->group(function () {
        Route::prefix('brands')->group(function () {
            Route::patch('/{id}/rank', [BrandController::class, 'updateRank'])->where('id', '[0-9]+');
            Route::patch('/ranks', [BrandController::class, 'bulkUpdateRanks']);
        });
    });
});

// Invoice Routes
Route::middleware(['auth:sanctum'])->group(function () {
    Route::get('/invoices', [InvoiceController::class, 'index']);
    Route::get('/invoices/{invoice}', [InvoiceController::class, 'show']);
    Route::post('/invoices', [InvoiceController::class, 'store']);
    Route::put('/invoices/{invoice}', [InvoiceController::class, 'update']);
    Route::post('/invoices/{invoice}/issue', [InvoiceController::class, 'issue']);
    Route::post('/invoices/{invoice}/mark-as-paid', [InvoiceController::class, 'markAsPaid']);
    Route::post('/invoices/{invoice}/void', [InvoiceController::class, 'void']);
    Route::post('/invoices/{invoice}/cancel', [InvoiceController::class, 'cancel']);
    Route::delete('/invoices/{invoice}', [InvoiceController::class, 'destroy']);
});

// Payment Routes
Route::prefix('payments')->group(function () {
    Route::post('/create-intent', [PaymentController::class, 'createPaymentIntent']);
    Route::get('/intent/{paymentIntentId}', [PaymentController::class, 'retrievePaymentIntent']);
    Route::post('/intent/{paymentIntentId}/confirm', [PaymentController::class, 'confirmPaymentIntent']);
    Route::post('/intent/{paymentIntentId}/cancel', [PaymentController::class, 'cancelPaymentIntent']);
});

// Fallback for undefined routes
Route::fallback(function () {
    return response()->json([
        'success' => false,
        'message' => 'Not Found',
        'details' => 'The requested endpoint does not exist'
    ], 404);
});