<?php

namespace App\Traits;

use Illuminate\Support\Facades\Cache;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Collection;

trait Cacheable
{
    /**
     * Cache duration in seconds
     */
    protected $cacheDuration = 3600; // 1 hour

    /**
     * Get cached data or store it if not exists
     *
     * @param string $key
     * @param \Closure $callback
     * @param int|null $duration
     * @return mixed
     */
    protected function remember(string $key, \Closure $callback, ?int $duration = null)
    {
        return Cache::remember($key, $duration ?? $this->cacheDuration, $callback);
    }

    /**
     * Get cached data for a model
     *
     * @param Model $model
     * @param string $key
     * @param \Closure $callback
     * @param int|null $duration
     * @return mixed
     */
    protected function rememberModel(Model $model, string $key, \Closure $callback, ?int $duration = null)
    {
        $cacheKey = sprintf('%s_%s_%s', class_basename($model), $model->id, $key);
        return $this->remember($cacheKey, $callback, $duration);
    }

    /**
     * Get cached data for a collection
     *
     * @param Collection $collection
     * @param string $key
     * @param \Closure $callback
     * @param int|null $duration
     * @return mixed
     */
    protected function rememberCollection(Collection $collection, string $key, \Closure $callback, ?int $duration = null)
    {
        $cacheKey = sprintf('%s_%s', class_basename($collection->first()), $key);
        return $this->remember($cacheKey, $callback, $duration);
    }

    /**
     * Clear cache for a model
     *
     * @param Model $model
     * @param string|null $key
     * @return void
     */
    protected function clearModelCache(Model $model, ?string $key = null)
    {
        if ($key) {
            Cache::forget(sprintf('%s_%s_%s', class_basename($model), $model->id, $key));
        } else {
            // Clear all caches for this model type
            $pattern = sprintf('%s_%s_*', class_basename($model), $model->id);
            $this->clearCacheByPattern($pattern);
        }
    }

    /**
     * Clear cache by pattern
     *
     * @param string $pattern
     * @return void
     */
    protected function clearCacheByPattern(string $pattern)
    {
        $keys = Cache::getStore()->get($pattern);
        if ($keys) {
            foreach ($keys as $key) {
                Cache::forget($key);
            }
        }
    }

    /**
     * Get public product data (minimal fields for listings)
     *
     * @param Model $product
     * @return array
     */
    protected function getPublicProductData(Model $product)
    {
        return [
            'id' => $product->id,
            'name' => $product->name,
            'slug' => $product->slug,
            'sku' => $product->sku,
            'price' => $product->current_price,
            'special_price' => $product->packPrice?->per_pack_special_price,
            'currency' => $product->currency?->code,
            'image' => $product->images->first()?->image_url,
            'brand' => $product->brand?->name,
            'rating' => $product->rating,
            'review_count' => $product->review_count,
            'is_active' => $product->is_active,
            'approval_status' => $product->approval_status,
            'pack_price' => $product->packPrice ? [
                'number_of_products' => $product->packPrice->number_of_products,
                'per_pack_price' => $product->packPrice->per_pack_price,
                'per_pack_special_price' => $product->packPrice->per_pack_special_price,
                'customer_margin' => $product->packPrice->customer_margin,
                'partner_margin' => $product->packPrice->partner_margin,
                'customer_margin_type' => $product->packPrice->customer_margin_type,
                'partner_margin_type' => $product->packPrice->partner_margin_type,
                'delivery_fee' => $product->packPrice->delivery_fee,
            ] : null,
            'bulk_prices' => $product->bulkPrices->map(fn($price) => [
                'number_of_packs' => $price->number_of_packs,
                'per_pack_price' => $price->per_pack_price,
                'per_pack_special_price' => $price->per_pack_special_price,
                'customer_margin' => $price->customer_margin,
                'partner_margin' => $price->partner_margin,
                'customer_margin_type' => $price->customer_margin_type,
                'partner_margin_type' => $price->partner_margin_type,
                'delivery_fee' => $price->delivery_fee,
                'sort_order' => $price->sort_order
            ]),
        ];
    }

    /**
     * Get detailed product data (for product detail page)
     *
     * @param Model $product
     * @return array
     */
    protected function getDetailedProductData(Model $product)
    {
        return array_merge($this->getPublicProductData($product), [
            'description' => $product->description,
            'specifications' => $product->specifications,
            'images' => $product->images->map(fn($image) => [
                'url' => $image->image_url,
                'alt' => $image->alt_text,
                'is_main' => $image->is_main
            ]),
            'categories' => $product->categories->map(fn($category) => [
                'id' => $category->id,
                'name' => $category->name,
                'slug' => $category->slug
            ]),
            'stock' => [
                'quantity' => $product->quantity,
                'is_in_stock' => $product->quantity > 0,
            ],
            'meta' => [
                'title' => $product->meta_title,
                'description' => $product->meta_description,
                'keywords' => $product->meta_keywords,
            ],
        ]);
    }
} 