<?php

namespace App\Services;

use App\Models\RefreshToken;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class TokenService
{
    /**
     * Create access and refresh tokens for a user during login.
     *
     * @param User $user
     * @param array $deviceInfo
     * @param bool $revokePrevious
     * @return array
     */
    public function createTokens(User $user, array $deviceInfo = [], bool $revokePrevious = true): array
    {
        try {
            // Check if user has too many active tokens
            if (RefreshToken::hasTooManyTokens($user)) {
                $this->revokeOldestTokens($user);
            }

            // Revoke previous tokens if requested
            if ($revokePrevious) {
                RefreshToken::revokeAllForUser($user, $user, 'new_login');
            }

            // Create refresh token
            $refreshToken = RefreshToken::createForUser($user, $deviceInfo);

            // Create access token
            $accessToken = $user->createToken('auth_token')->plainTextToken;

            // Log successful token creation
            Log::info('Tokens created for user', [
                'user_id' => $user->id,
                'email' => $user->email,
                'device_info' => $deviceInfo,
                'ip_address' => request()->ip()
            ]);

            return [
                'access_token' => $accessToken,
                'token' => $accessToken, // Backward compatibility
                'refresh_token' => $refreshToken->token,
                'token_type' => 'Bearer',
                'expires_in' => config('auth_settings.token_expiration.access_token', 60) * 60, // Convert to seconds
                'refresh_expires_in' => config('auth_settings.token_expiration.refresh_token', 1440) * 60, // Convert to seconds
                'user' => $user->load('role')
            ];

        } catch (\Exception $e) {
            Log::error('Failed to create tokens for user', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Refresh access token using refresh token.
     *
     * @param string $refreshToken
     * @param bool $rotateRefreshToken
     * @return array
     * @throws ValidationException
     */
    public function refreshAccessToken(string $refreshToken, bool $rotateRefreshToken = true): array
    {
        try {
            // Find active refresh token
            $token = RefreshToken::findActiveByToken($refreshToken);

            if (!$token) {
                Log::warning('Invalid refresh token attempt', [
                    'token' => substr($refreshToken, 0, 20) . '...',
                    'ip_address' => request()->ip()
                ]);
                throw ValidationException::withMessages([
                    'refresh_token' => ['Invalid or expired refresh token.']
                ]);
            }

            $user = $token->user;

            // Check if user is still active
            if (!$user->is_active) {
                $token->revoke($user, 'user_inactive');
                throw ValidationException::withMessages([
                    'refresh_token' => ['User account is inactive.']
                ]);
            }

            // Check if vendor is still approved
            if ($user->isVendor() && !$user->vendorProfile->is_approved) {
                $token->revoke($user, 'vendor_not_approved');
                throw ValidationException::withMessages([
                    'refresh_token' => ['Vendor account is not approved.']
                ]);
            }

            // Revoke old access tokens
            $user->tokens()->delete();

            // Create new access token
            $newAccessToken = $user->createToken('auth_token')->plainTextToken;

            $response = [
                'access_token' => $newAccessToken,
                'token' => $newAccessToken, // Backward compatibility
                'token_type' => 'Bearer',
                'expires_in' => config('auth_settings.token_expiration.access_token', 60) * 60,
                'user' => $user->load('role')
            ];

            // Rotate refresh token if requested
            if ($rotateRefreshToken) {
                $oldToken = $token;
                
                // Create new refresh token
                $newRefreshToken = RefreshToken::createForUser($user, $oldToken->device_info);
                
                // Revoke old refresh token
                $oldToken->revoke($user, 'token_rotation');
                
                $response['refresh_token'] = $newRefreshToken->token;
                $response['refresh_expires_in'] = config('auth_settings.token_expiration.refresh_token', 1440) * 60;
            }

            // Log successful token refresh
            Log::info('Access token refreshed', [
                'user_id' => $user->id,
                'email' => $user->email,
                'ip_address' => request()->ip(),
                'token_rotated' => $rotateRefreshToken
            ]);

            return $response;

        } catch (ValidationException $e) {
            throw $e;
        } catch (\Exception $e) {
            Log::error('Failed to refresh access token', [
                'error' => $e->getMessage(),
                'ip_address' => request()->ip()
            ]);
            throw ValidationException::withMessages([
                'refresh_token' => ['Failed to refresh token. Please try again.']
            ]);
        }
    }

    /**
     * Revoke tokens for logout.
     *
     * @param User $user
     * @param string|null $refreshToken
     * @param bool $revokeAll
     * @return array
     */
    public function revokeTokens(User $user, ?string $refreshToken = null, bool $revokeAll = false): array
    {
        try {
            $revokedCount = 0;
            $reason = $revokeAll ? 'logout_all_devices' : 'user_logout';

            if ($revokeAll) {
                // Revoke all refresh tokens
                $revokedCount = RefreshToken::revokeAllForUser($user, $user, $reason);
                
                // Revoke all access tokens
                $user->tokens()->delete();
            } elseif ($refreshToken) {
                // Revoke specific refresh token
                $token = RefreshToken::findByToken($refreshToken);
                if ($token && $token->user_id === $user->id) {
                    $token->revoke($user, $reason);
                    $revokedCount = 1;
                }
                
                // Revoke current access token
                $user->currentAccessToken()->delete();
            } else {
                // Revoke current access token only
                $user->currentAccessToken()->delete();
            }

            // Log logout
            Log::info('User logged out', [
                'user_id' => $user->id,
                'email' => $user->email,
                'revoked_count' => $revokedCount,
                'revoke_all' => $revokeAll,
                'ip_address' => request()->ip()
            ]);

            return [
                'message' => $revokeAll ? 'Logged out from all devices' : 'Logged out successfully',
                'revoked_tokens' => $revokedCount
            ];

        } catch (\Exception $e) {
            Log::error('Failed to revoke tokens', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Get user's active sessions.
     *
     * @param User $user
     * @return array
     */
    public function getUserSessions(User $user): array
    {
        try {
            $sessions = $user->activeRefreshTokens()
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($token) {
                    return [
                        'id' => $token->id,
                        'device' => $token->getDeviceInfoString(),
                        'ip_address' => $token->ip_address,
                        'created_at' => $token->created_at->format('Y-m-d H:i:s'),
                        'expires_in' => $token->expires_in,
                        'is_current' => $token->token === request()->header('X-Refresh-Token')
                    ];
                });

            return [
                'sessions' => $sessions,
                'total_sessions' => $sessions->count(),
                'max_sessions' => config('auth_settings.security.max_concurrent_sessions', 5)
            ];

        } catch (\Exception $e) {
            Log::error('Failed to get user sessions', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Revoke a specific session.
     *
     * @param User $user
     * @param int $sessionId
     * @return array
     * @throws ValidationException
     */
    public function revokeSession(User $user, int $sessionId): array
    {
        try {
            $token = RefreshToken::where('id', $sessionId)
                ->where('user_id', $user->id)
                ->whereNull('revoked_at')
                ->first();

            if (!$token) {
                throw ValidationException::withMessages([
                    'session_id' => ['Session not found or already revoked.']
                ]);
            }

            $token->revoke($user, 'session_revoked_by_user');

            Log::info('Session revoked by user', [
                'user_id' => $user->id,
                'session_id' => $sessionId,
                'device' => $token->getDeviceInfoString()
            ]);

            return [
                'message' => 'Session revoked successfully',
                'session_id' => $sessionId
            ];

        } catch (ValidationException $e) {
            throw $e;
        } catch (\Exception $e) {
            Log::error('Failed to revoke session', [
                'user_id' => $user->id,
                'session_id' => $sessionId,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Validate refresh token without refreshing.
     *
     * @param string $refreshToken
     * @return bool
     */
    public function validateRefreshToken(string $refreshToken): bool
    {
        $token = RefreshToken::findActiveByToken($refreshToken);
        return $token !== null;
    }

    /**
     * Get token statistics for a user.
     *
     * @param User $user
     * @return array
     */
    public function getTokenStats(User $user): array
    {
        try {
            $activeCount = RefreshToken::getActiveCountForUser($user);
            $maxTokens = config('auth_settings.security.max_concurrent_sessions', 5);

            return [
                'active_sessions' => $activeCount,
                'max_sessions' => $maxTokens,
                'available_sessions' => max(0, $maxTokens - $activeCount),
                'session_limit_reached' => $activeCount >= $maxTokens
            ];

        } catch (\Exception $e) {
            Log::error('Failed to get token stats', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Clean up expired and old revoked tokens.
     *
     * @return array
     */
    public function cleanupTokens(): array
    {
        try {
            $expiredCount = RefreshToken::cleanupExpired();
            $revokedCount = RefreshToken::cleanupRevoked(30); // 30 days

            Log::info('Token cleanup completed', [
                'expired_removed' => $expiredCount,
                'revoked_removed' => $revokedCount
            ]);

            return [
                'expired_removed' => $expiredCount,
                'revoked_removed' => $revokedCount,
                'total_removed' => $expiredCount + $revokedCount
            ];

        } catch (\Exception $e) {
            Log::error('Failed to cleanup tokens', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Revoke oldest tokens when user has too many.
     *
     * @param User $user
     * @return int
     */
    private function revokeOldestTokens(User $user): int
    {
        $maxTokens = config('auth_settings.security.max_concurrent_sessions', 5);
        $activeTokens = $user->activeRefreshTokens()
            ->orderBy('created_at', 'asc')
            ->get();

        if ($activeTokens->count() >= $maxTokens) {
            $tokensToRevoke = $activeTokens->take($activeTokens->count() - $maxTokens + 1);
            
            foreach ($tokensToRevoke as $token) {
                $token->revoke($user, 'session_limit_exceeded');
            }

            Log::info('Oldest tokens revoked due to session limit', [
                'user_id' => $user->id,
                'revoked_count' => $tokensToRevoke->count()
            ]);

            return $tokensToRevoke->count();
        }

        return 0;
    }

    /**
     * Extract device information from request.
     *
     * @param Request $request
     * @return array
     */
    public function extractDeviceInfo(Request $request): array
    {
        $userAgent = $request->userAgent();
        $deviceInfo = [];

        // Basic device detection (you can enhance this with a proper user agent parser)
        if (strpos($userAgent, 'Mobile') !== false) {
            $deviceInfo['device'] = 'mobile';
        } elseif (strpos($userAgent, 'Tablet') !== false) {
            $deviceInfo['device'] = 'tablet';
        } else {
            $deviceInfo['device'] = 'desktop';
        }

        // Browser detection
        if (strpos($userAgent, 'Chrome') !== false) {
            $deviceInfo['browser'] = 'chrome';
        } elseif (strpos($userAgent, 'Firefox') !== false) {
            $deviceInfo['browser'] = 'firefox';
        } elseif (strpos($userAgent, 'Safari') !== false) {
            $deviceInfo['browser'] = 'safari';
        } elseif (strpos($userAgent, 'Edge') !== false) {
            $deviceInfo['browser'] = 'edge';
        } else {
            $deviceInfo['browser'] = 'unknown';
        }

        // OS detection
        if (strpos($userAgent, 'Windows') !== false) {
            $deviceInfo['os'] = 'Windows';
        } elseif (strpos($userAgent, 'Mac') !== false) {
            $deviceInfo['os'] = 'macOS';
        } elseif (strpos($userAgent, 'Linux') !== false) {
            $deviceInfo['os'] = 'Linux';
        } elseif (strpos($userAgent, 'Android') !== false) {
            $deviceInfo['os'] = 'Android';
        } elseif (strpos($userAgent, 'iOS') !== false) {
            $deviceInfo['os'] = 'iOS';
        } else {
            $deviceInfo['os'] = 'Unknown';
        }

        return $deviceInfo;
    }
} 