<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'order_number',
        'user_id',
        'cart_id',
        'status',
        'payment_status',
        'shipping_status',
        'subtotal',
        'discount_amount',
        'shipping_amount',
        'tax_amount',
        'total_amount',
        'currency_code',
        'notes',
        'billing_address',
        'shipping_address',
        'billing_phone',
        'shipping_phone',
        'billing_email',
        'payment_method',
        'payment_transaction_id',
        'refund_reason',
        'refund_amount',
        'refund_transaction_id',
        'customer_notes',
        'admin_notes',
        'invoice_id'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'subtotal' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'shipping_amount' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'refund_amount' => 'decimal:2',
    ];

    /**
     * Get the user that owns the order.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the cart that was used to create the order.
     */
    public function cart()
    {
        return $this->belongsTo(Cart::class);
    }

    /**
     * Get the items for the order.
     */
    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Get the invoice for the order.
     */
    public function invoice()
    {
        return $this->belongsTo(Invoice::class);
    }

    /**
     * Get the history for the order.
     */
    public function history()
    {
        return $this->hasMany(OrderHistory::class);
    }

    /**
     * Get the shipments for the order.
     */
    public function shipments()
    {
        return $this->hasMany(Shipment::class);
    }

    /**
     * Get the fulfillment orders for the order.
     */
    public function fulfillmentOrders()
    {
        return $this->hasMany(FulfillmentOrder::class);
    }

    /**
     * Get the delivery schedules for the order.
     */
    public function deliverySchedules()
    {
        return $this->hasMany(DeliverySchedule::class);
    }

    /**
     * Format the total amount with currency symbol.
     */
    public function getFormattedTotalAttribute()
    {
        $currency = $this->currency;
        $symbol = $currency ? $currency->symbol : '$';
        
        return $symbol . number_format($this->total_amount, 2);
    }

    /**
     * Get the currency for the order.
     */
    public function currency()
    {
        return $this->belongsTo(Currency::class, 'currency_code', 'code');
    }

    /**
     * Scope a query to only include orders with a specific status.
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope a query to only include orders with a specific payment status.
     */
    public function scopeByPaymentStatus($query, $status)
    {
        return $query->where('payment_status', $status);
    }

    /**
     * Scope a query to only include orders with a specific shipping status.
     */
    public function scopeByShippingStatus($query, $status)
    {
        return $query->where('shipping_status', $status);
    }

    /**
     * Check if the order can be canceled.
     */
    public function canBeCanceled()
    {
        return in_array($this->status, ['pending', 'processing']) && 
               in_array($this->shipping_status, ['pending', 'processing']);
    }

    /**
     * Check if the order can be refunded.
     */
    public function canBeRefunded()
    {
        return in_array($this->status, ['completed']) && 
               in_array($this->payment_status, ['paid']);
    }
}