<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Category extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'slug',
        'description',
        'parent_id',
        'image',
        'is_featured',
        'is_active',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_featured' => 'boolean',
        'is_active' => 'boolean',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        'product_count',
    ];

    /**
     * Boot function from Laravel.
     */
    protected static function boot()
    {
        parent::boot();
        
        // When creating a category, automatically create a slug from the name if not provided
        static::creating(function ($category) {
            if (empty($category->slug)) {
                $category->slug = Str::slug($category->name);
            }
        });
        
        // When updating, regenerate slug if name changes and slug wasn't manually set
        static::updating(function ($category) {
            if ($category->isDirty('name') && !$category->isDirty('slug')) {
                $category->slug = Str::slug($category->name);
            }
        });
    }

    /**
     * Get the parent category.
     */
    public function parent()
    {
        return $this->belongsTo(Category::class, 'parent_id');
    }

    /**
     * Get the subcategories.
     */
    public function children()
    {
        return $this->hasMany(Category::class, 'parent_id');
    }

    /**
     * Get all recursive children (subcategories).
     */
    public function allChildren()
    {
        return $this->children()->with('allChildren');
    }

    /**
     * Get the products for the category.
     */
    public function products()
    {
        return $this->belongsToMany(Product::class, 'product_categories');
    }

    /**
     * Get the active products for the category.
     */
    public function activeProducts()
    {
        return $this->belongsToMany(Product::class, 'product_categories')
            ->where('products.is_active', true)
            ->where('products.is_approved', true);
    }

    /**
     * Get the product count attribute.
     */
    public function getProductCountAttribute()
    {
        return $this->products()->count();
    }

    /**
     * Scope a query to only include active categories.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include featured categories.
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Scope a query to only include root categories (no parent).
     */
    public function scopeRoot($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * Scope a query to include categories with the given parent ID.
     */
    public function scopeByParent($query, $parentId)
    {
        return $query->where('parent_id', $parentId);
    }

    /**
     * Get all parent categories in ascending order.
     */
    public function getParentTree()
    {
        $parents = collect([]);
        
        $parent = $this->parent;
        while ($parent) {
            $parents->prepend($parent);
            $parent = $parent->parent;
        }
        
        return $parents;
    }

    /**
     * Get full breadcrumb for the category.
     */
    public function getBreadcrumb()
    {
        $breadcrumb = $this->getParentTree();
        $breadcrumb->push($this);
        
        return $breadcrumb;
    }
}