<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CartItem extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'cart_id',
        'product_id',
        'quantity',
        'unit_price',
        'subtotal',
        'discount_amount',
        'tax_amount',
        'total_amount',
        'is_bulk_pricing',
        'is_pack_pricing',
        'bulk_price_id',
        'pack_price_id',
        'notes',
        'customer_margin',
        'partner_margin',
        'customer_margin_type',
        'partner_margin_type',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'quantity' => 'integer',
        'unit_price' => 'decimal:2',
        'subtotal' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'is_bulk_pricing' => 'boolean',
        'is_pack_pricing' => 'boolean',
        'customer_margin' => 'decimal:2',
        'partner_margin' => 'decimal:2',
    ];

    /**
     * Get the cart that owns the item.
     */
    public function cart()
    {
        return $this->belongsTo(Cart::class);
    }

    /**
     * Get the product for this cart item.
     */
    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * Get the pack price for this cart item if using pack pricing.
     */
    public function packPrice()
    {
        return $this->belongsTo(ProductPackPrice::class, 'pack_price_id');
    }

    /**
     * Get the bulk price for this cart item if using bulk pricing.
     */
    public function bulkPrice()
    {
        return $this->belongsTo(ProductBulkPrice::class, 'bulk_price_id');
    }

    /**
     * Set quantity and recalculate prices.
     */
    public function setQuantityAndRecalculate($quantity)
    {
        $this->quantity = $quantity;
        
        return $this->calculatePricing();
    }

    /**
     * Calculate pricing based on quantity and pricing type.
     */
    public function calculatePricing()
    {
        $product = $this->product;
        
        if ($this->is_bulk_pricing && $this->bulk_price_id) {
            // Use bulk price
            $bulkPrice = $this->bulkPrice;
            
            if (!$bulkPrice) {
                return $this;
            }
            
            $unitPrice = $bulkPrice->per_pack_special_price ?? $bulkPrice->per_pack_price;
            $regularPrice = $bulkPrice->per_pack_price;
            
            $this->unit_price = $unitPrice;
            $this->subtotal = $regularPrice * $this->quantity;
            $this->discount_amount = ($regularPrice - $unitPrice) * $this->quantity;
            $this->customer_margin = $bulkPrice->customer_margin;
            $this->partner_margin = $bulkPrice->partner_margin;
            $this->customer_margin_type = $bulkPrice->customer_margin_type;
            $this->partner_margin_type = $bulkPrice->partner_margin_type;
        } elseif ($this->is_pack_pricing && $this->pack_price_id) {
            // Use pack price
            $packPrice = $this->packPrice;
            
            if (!$packPrice) {
                return $this;
            }
            
            $unitPrice = $packPrice->per_pack_special_price ?? $packPrice->per_pack_price;
            $regularPrice = $packPrice->per_pack_price;
            
            $this->unit_price = $unitPrice;
            $this->subtotal = $regularPrice * $this->quantity;
            $this->discount_amount = ($regularPrice - $unitPrice) * $this->quantity;
            $this->customer_margin = $packPrice->customer_margin;
            $this->partner_margin = $packPrice->partner_margin;
            $this->customer_margin_type = $packPrice->customer_margin_type;
            $this->partner_margin_type = $packPrice->partner_margin_type;
        } else {
            // Use default pricing from product
            $unitPrice = $product->current_price ?? $product->base_price;
            $regularPrice = $product->base_price;
            
            $this->unit_price = $unitPrice;
            $this->subtotal = $regularPrice * $this->quantity;
            $this->discount_amount = ($regularPrice - $unitPrice) * $this->quantity;
            
            // Get default margins from product
            $customerMargin = $product->getDefaultCustomerMargin();
            $partnerMargin = $product->getDefaultPartnerMargin();
            
            $this->customer_margin = $customerMargin['value'];
            $this->partner_margin = $partnerMargin['value'];
            $this->customer_margin_type = $customerMargin['type'];
            $this->partner_margin_type = $partnerMargin['type'];
        }
        
        // Calculate tax (if applicable)
        $this->tax_amount = 0; // Set to 0 for now, can be updated based on tax rules
        
        // Calculate total
        $this->total_amount = $this->subtotal - $this->discount_amount + $this->tax_amount;
        
        $this->save();
        
        // Update cart totals
        $this->cart->calculateTotals();
        
        return $this;
    }

    /**
     * Calculate the margin amount for customer.
     */
    public function getCustomerMarginAmountAttribute()
    {
        if ($this->customer_margin_type === 'percentage') {
            return round(($this->unit_price * $this->customer_margin) / 100, 2);
        } else {
            return $this->customer_margin;
        }
    }

    /**
     * Calculate the margin amount for partner.
     */
    public function getPartnerMarginAmountAttribute()
    {
        if ($this->partner_margin_type === 'percentage') {
            return round(($this->unit_price * $this->partner_margin) / 100, 2);
        } else {
            return $this->partner_margin;
        }
    }
}