<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Cart extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'session_id',
        'notes',
        'subtotal',
        'discount_amount',
        'tax_amount', 
        'shipping_amount',
        'total_amount',
        'currency_code',
        'is_active',
        'expires_at',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'subtotal' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'shipping_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'is_active' => 'boolean',
        'expires_at' => 'datetime',
    ];

    /**
     * Get the user that owns the cart.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the items in the cart.
     */
    public function items()
    {
        return $this->hasMany(CartItem::class);
    }

    /**
     * Get the currency used for this cart.
     */
    public function currency()
    {
        return $this->belongsTo(Currency::class, 'currency_code', 'code');
    }

    /**
     * Calculate the total of all items in the cart.
     */
    public function calculateTotals()
    {
        $items = $this->items;
        
        $subtotal = $items->sum('subtotal');
        $discountAmount = $items->sum('discount_amount');
        $taxAmount = $items->sum('tax_amount');
        
        // Calculate totals
        $this->subtotal = $subtotal;
        $this->discount_amount = $discountAmount;
        $this->tax_amount = $taxAmount;
        $this->total_amount = $subtotal - $discountAmount + $taxAmount + $this->shipping_amount;
        
        $this->save();
        
        return $this;
    }

    /**
     * Scope a query to only include active carts.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include carts for a specific user.
     */
    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope a query to only include carts that haven't expired.
     */
    public function scopeNotExpired($query)
    {
        return $query->where(function($q) {
            $q->whereNull('expires_at')
              ->orWhere('expires_at', '>', now());
        });
    }

    /**
     * Get active cart for a user, or create one if it doesn't exist.
     */
    public static function getOrCreateForUser($userId, $currencyCode = 'AUD')
    {
        // Find active cart for user
        $cart = self::forUser($userId)->active()->notExpired()->first();
        
        // If no active cart exists, create a new one
        if (!$cart) {
            $cart = self::create([
                'user_id' => $userId,
                'currency_code' => $currencyCode,
                'is_active' => true,
                'subtotal' => 0,
                'discount_amount' => 0,
                'tax_amount' => 0,
                'shipping_amount' => 0,
                'total_amount' => 0
            ]);
        }
        
        return $cart;
    }

    /**
     * Check if the cart has sufficient inventory for all items.
     */
    public function hasSufficientInventory()
    {
        foreach ($this->items as $item) {
            $product = $item->product;
            
            if ($product->quantity < $item->quantity) {
                return false;
            }
        }
        
        return true;
    }

    /**
     * Check if the cart is empty.
     */
    public function isEmpty()
    {
        return $this->items->isEmpty();
    }

    /**
     * Format total amount with currency symbol.
     */
    public function getFormattedTotalAttribute()
    {
        $currency = $this->currency;
        $symbol = $currency ? $currency->symbol : '$';
        
        return $symbol . number_format($this->total_amount, 2);
    }
}