<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ProductPackPrice extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'product_id',
        'number_of_products',
        'per_pack_price',
        'per_pack_special_price',
        'customer_margin',
        'partner_margin',
        'customer_margin_type',
        'partner_margin_type',
        'delivery_fee',
        'tax_rate',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'number_of_products' => 'integer',
        'per_pack_price' => 'decimal:2',
        'per_pack_special_price' => 'decimal:2',
        'customer_margin' => 'decimal:2',
        'partner_margin' => 'decimal:2',
        'customer_margin_type' => 'string',
        'partner_margin_type' => 'string',
        'delivery_fee' => 'decimal:2',
        'tax_rate' => 'decimal:2',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        'discount_percentage',
        'total_price',
        'customer_margin_amount',
        'partner_margin_amount',
    ];

    /**
     * Boot function from Laravel.
     */
    protected static function boot()
    {
        parent::boot();
        
        // When creating a pack price, set default margin types if not provided
        static::creating(function ($packPrice) {
            if (empty($packPrice->customer_margin_type)) {
                $packPrice->customer_margin_type = 'percentage';
            }
            
            if (empty($packPrice->partner_margin_type)) {
                $packPrice->partner_margin_type = 'percentage';
            }
            
            // Set default margins from product if not provided
            if ($packPrice->product_id && (is_null($packPrice->customer_margin) || is_null($packPrice->partner_margin))) {
                $product = Product::find($packPrice->product_id);
                if ($product) {
                    if (is_null($packPrice->customer_margin)) {
                        $customerMargin = $product->getDefaultCustomerMargin();
                        $packPrice->customer_margin = $customerMargin['value'];
                        $packPrice->customer_margin_type = $customerMargin['type'];
                    }
                    
                    if (is_null($packPrice->partner_margin)) {
                        $partnerMargin = $product->getDefaultPartnerMargin();
                        $packPrice->partner_margin = $partnerMargin['value'];
                        $packPrice->partner_margin_type = $partnerMargin['type'];
                    }
                }
            }
        });
    }

    /**
     * Get the product that owns the pack price.
     */
    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * Get the offers for this pack price.
     */
    public function offers()
    {
        return $this->hasMany(ProductOffer::class, 'price_id')
            ->where('price_type', 'pack');
    }

    /**
     * Get the active offers for this pack price.
     */
    public function activeOffers()
    {
        return $this->offers()
            ->where('is_active', true)
            ->where(function ($query) {
                $query->whereNull('start_date')
                    ->orWhere('start_date', '<=', now());
            })
            ->where(function ($query) {
                $query->whereNull('end_date')
                    ->orWhere('end_date', '>=', now());
            });
    }

    /**
     * Get the best active offer for this pack price.
     */
    public function bestOffer()
    {
        return $this->activeOffers()
            ->orderBy('price')
            ->first();
    }

    /**
     * Get the final price after applying the best offer.
     */
    public function getFinalPrice()
    {
        $bestOffer = $this->bestOffer();
        if ($bestOffer) {
            return $bestOffer->price;
        }
        
        return $this->per_pack_special_price ?? $this->per_pack_price;
    }

    /**
     * Get the discount percentage.
     */
    public function getDiscountPercentage()
    {
        if (!$this->per_pack_special_price) {
            return 0;
        }
        
        if ($this->per_pack_price <= 0) {
            return 0;
        }
        
        return round((($this->per_pack_price - $this->per_pack_special_price) / $this->per_pack_price) * 100, 2);
    }
    
    /**
     * Get the discount percentage attribute.
     */
    public function getDiscountPercentageAttribute()
    {
        return $this->getDiscountPercentage();
    }
    
    /**
     * Get the total price (regular price × quantity).
     */
    public function getTotalPriceAttribute()
    {
        return $this->per_pack_price * $this->number_of_products;
    }

    /**
     * Get the tax rate for this pack price (inherits from product if not set)
     */
    public function getTaxRate()
    {
        // Use pack price specific tax rate if set, otherwise inherit from product
        if (!is_null($this->tax_rate)) {
            return $this->tax_rate;
        }
        
        // Fall back to product tax rate
        return $this->product->getTaxRate();
    }

    /**
     * Calculate customer margin amount based on margin type.
     */
    public function getCustomerMarginAmountAttribute()
    {
        $basePrice = $this->per_pack_special_price ?? $this->per_pack_price;
        
        if ($this->customer_margin_type === 'percentage') {
            // Calculate percentage margin
            return round(($basePrice * $this->customer_margin) / 100, 2);
        } else {
            // Return fixed margin
            return $this->customer_margin;
        }
    }
    
    /**
     * Calculate partner margin amount based on margin type.
     */
    public function getPartnerMarginAmountAttribute()
    {
        $basePrice = $this->per_pack_special_price ?? $this->per_pack_price;
        
        if ($this->partner_margin_type === 'percentage') {
            // Calculate percentage margin
            return round(($basePrice * $this->partner_margin) / 100, 2);
        } else {
            // Return fixed margin
            return $this->partner_margin;
        }
    }
    
    /**
     * Format the price with currency symbol.
     */
    public function formatPrice($price)
    {
        if ($this->product) {
            return $this->product->formatPrice($price);
        }
        
        // Fallback if product is not loaded
        return '$' . number_format($price, 2);
    }
}