<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ProductOffer extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'product_id',
        'price_type',
        'price_id',
        'title',
        'price',
        'percent',
        'start_date',
        'end_date',
        'is_active',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'price' => 'decimal:2',
        'percent' => 'decimal:2',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'is_active' => 'boolean',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        'is_current',
        'days_remaining',
    ];

    /**
     * Get the product that owns the offer.
     */
    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * Get the pack price associated with this offer.
     */
    public function packPrice()
    {
        if ($this->price_type === 'pack') {
            return $this->belongsTo(ProductPackPrice::class, 'price_id');
        }
        
        return null;
    }

    /**
     * Get the bulk price associated with this offer.
     */
    public function bulkPrice()
    {
        if ($this->price_type === 'bulk') {
            return $this->belongsTo(ProductBulkPrice::class, 'price_id');
        }
        
        return null;
    }

    /**
     * Check if the offer is currently active.
     */
    public function isCurrentlyActive()
    {
        if (!$this->is_active) {
            return false;
        }
        
        $now = now();
        
        if ($this->start_date && $this->start_date > $now) {
            return false;
        }
        
        if ($this->end_date && $this->end_date < $now) {
            return false;
        }
        
        return true;
    }

    /**
     * Get the is_current attribute.
     */
    public function getIsCurrentAttribute()
    {
        return $this->isCurrentlyActive();
    }

    /**
     * Get days remaining until offer ends.
     */
    public function getDaysRemainingAttribute()
    {
        if (!$this->end_date) {
            return null;
        }
        
        $now = now();
        
        if ($this->end_date < $now) {
            return 0;
        }
        
        return $now->diffInDays($this->end_date);
    }

    /**
     * Get the original price before the offer.
     */
    public function getOriginalPrice()
    {
        if ($this->price_type === 'pack') {
            $packPrice = $this->packPrice;
            return $packPrice ? $packPrice->per_pack_price : null;
        } else if ($this->price_type === 'bulk') {
            $bulkPrice = $this->bulkPrice;
            return $bulkPrice ? $bulkPrice->per_pack_price : null;
        }
        
        return null;
    }

    /**
     * Get the savings amount.
     */
    public function getSavingsAmount()
    {
        $originalPrice = $this->getOriginalPrice();
        
        if (!$originalPrice) {
            return 0;
        }
        
        return $originalPrice - $this->price;
    }

    /**
     * Get the savings percentage.
     */
    public function getSavingsPercentage()
    {
        $originalPrice = $this->getOriginalPrice();
        
        if (!$originalPrice || $originalPrice <= 0) {
            return 0;
        }
        
        return round((($originalPrice - $this->price) / $originalPrice) * 100, 2);
    }

    /**
     * Scope a query to only include active offers.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include current offers (active and within date range).
     */
    public function scopeCurrent($query)
    {
        $now = now();
        
        return $query->where('is_active', true)
            ->where(function ($q) use ($now) {
                $q->whereNull('start_date')
                  ->orWhere('start_date', '<=', $now);
            })
            ->where(function ($q) use ($now) {
                $q->whereNull('end_date')
                  ->orWhere('end_date', '>=', $now);
            });
    }

    /**
     * Scope a query to only include offers for pack prices.
     */
    public function scopePackOffers($query)
    {
        return $query->where('price_type', 'pack');
    }

    /**
     * Scope a query to only include offers for bulk prices.
     */
    public function scopeBulkOffers($query)
    {
        return $query->where('price_type', 'bulk');
    }

    /**
     * Scope a query to only include offers for a specific product.
     */
    public function scopeForProduct($query, $productId)
    {
        return $query->where('product_id', $productId);
    }

    /**
     * Scope a query to order by biggest discount percentage.
     */
    public function scopeOrderByBiggestDiscount($query)
    {
        return $query->orderByRaw('(SELECT 
            CASE price_type 
                WHEN "pack" THEN 
                    (SELECT (ppp.per_pack_price - product_offers.price) / ppp.per_pack_price * 100 
                     FROM product_pack_prices ppp 
                     WHERE ppp.id = product_offers.price_id)
                WHEN "bulk" THEN 
                    (SELECT (pbp.per_pack_price - product_offers.price) / pbp.per_pack_price * 100 
                     FROM product_bulk_prices pbp 
                     WHERE pbp.id = product_offers.price_id)
            END) DESC');
    }
}