<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class OrderItem extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'order_id',
        'product_id',
        'product_name',
        'product_sku',
        'quantity',
        'unit_price',
        'subtotal',
        'discount_amount',
        'tax_amount',
        'total_amount',
        'is_bulk_pricing',
        'is_pack_pricing',
        'bulk_price_id',
        'pack_price_id',
        'customer_margin',
        'partner_margin',
        'customer_margin_type',
        'partner_margin_type',
        'membership_discount_amount',
        'membership_discount_percentage',
        'original_price_before_membership'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'quantity' => 'integer',
        'unit_price' => 'decimal:2',
        'subtotal' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'is_bulk_pricing' => 'boolean',
        'is_pack_pricing' => 'boolean',
        'customer_margin' => 'decimal:2',
        'partner_margin' => 'decimal:2',
        'membership_discount_amount' => 'decimal:2',
        'membership_discount_percentage' => 'decimal:2',
        'original_price_before_membership' => 'decimal:2',
    ];

    /**
     * Get the order that owns the item.
     */
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get the product for this order item.
     */
    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * Get the pack price for this order item.
     */
    public function packPrice()
    {
        return $this->belongsTo(ProductPackPrice::class, 'pack_price_id');
    }

    /**
     * Get the bulk price for this order item.
     */
    public function bulkPrice()
    {
        return $this->belongsTo(ProductBulkPrice::class, 'bulk_price_id');
    }

    /**
     * Get the shipment items for this order item.
     */
    public function shipmentItems()
    {
        return $this->hasMany(ShipmentItem::class);
    }

    /**
     * Get the fulfillment items for this order item.
     */
    public function fulfillmentItems()
    {
        return $this->hasMany(FulfillmentItem::class);
    }

    /**
     * Calculate the margin amount for customer.
     */
    public function getCustomerMarginAmountAttribute()
    {
        if ($this->customer_margin_type === 'percentage') {
            return round(($this->unit_price * $this->customer_margin) / 100, 2);
        } else {
            return $this->customer_margin;
        }
    }

    /**
     * Calculate the margin amount for partner.
     */
    public function getPartnerMarginAmountAttribute()
    {
        if ($this->partner_margin_type === 'percentage') {
            return round(($this->unit_price * $this->partner_margin) / 100, 2);
        } else {
            return $this->partner_margin;
        }
    }

    /**
     * Get the discount percentage.
     */
    public function getDiscountPercentageAttribute()
    {
        if ($this->subtotal <= 0) {
            return 0;
        }
        
        return round(($this->discount_amount / $this->subtotal) * 100, 2);
    }
}