<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Invoice extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'order_id',
        'is_manual_invoice',
        'created_by',
        'created_user_id',
        'invoice_type',
        'invoice_number',
        'invoice_date',
        'due_date',
        'payment_terms',
        'current_status',
        'issued_at',
        'cancelled_at',
        'paid_at',
        'void_at',
        'reference_number',
        'is_retail',
        'is_supplier',
        'company_name',
        'company_registration_number',
        'company_address',
        'company_phone',
        'company_email',
        'company_website',
        'terms',
        'supplier_notes',
        'retailer_notes',
        'internal_notes',
        'footer',
        'meta_version',
        'meta_format',
        'meta_language',
        'currency'
    ];

    protected $casts = [
        'is_manual_invoice' => 'boolean',
        'is_retail' => 'boolean',
        'is_supplier' => 'boolean',
        'invoice_date' => 'date',
        'due_date' => 'date',
        'issued_at' => 'datetime',
        'cancelled_at' => 'datetime',
        'paid_at' => 'datetime',
        'void_at' => 'datetime',
    ];

    // Relationships
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function createdUser()
    {
        return $this->belongsTo(User::class, 'created_user_id');
    }

    // Scopes
    public function scopeDraft($query)
    {
        return $query->where('current_status', 'draft');
    }

    public function scopeIssued($query)
    {
        return $query->where('current_status', 'issued');
    }

    public function scopePaid($query)
    {
        return $query->where('current_status', 'paid');
    }

    public function scopeVoid($query)
    {
        return $query->where('current_status', 'void');
    }

    public function scopeCancelled($query)
    {
        return $query->where('current_status', 'cancelled');
    }

    // Methods
    public function issue()
    {
        $this->update([
            'current_status' => 'issued',
            'issued_at' => now()
        ]);
    }

    public function markAsPaid()
    {
        $this->update([
            'current_status' => 'paid',
            'paid_at' => now()
        ]);
    }

    public function void()
    {
        $this->update([
            'current_status' => 'void',
            'void_at' => now()
        ]);
    }

    public function cancel()
    {
        $this->update([
            'current_status' => 'cancelled',
            'cancelled_at' => now()
        ]);
    }

    public function generateInvoiceNumber()
    {
        $prefix = 'INV';
        $year = date('Y');
        $month = date('m');
        
        $lastInvoice = self::where('invoice_number', 'like', "{$prefix}{$year}{$month}%")
            ->orderBy('invoice_number', 'desc')
            ->first();
        
        if ($lastInvoice) {
            $lastNumber = (int) substr($lastInvoice->invoice_number, -4);
            $newNumber = str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        } else {
            $newNumber = '0001';
        }
        
        return "{$prefix}{$year}{$month}{$newNumber}";
    }
} 