<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CustomerProfile extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'company_name',
        'company_logo',
        'company_website',
        'business_type',
        'business_registration_number',
        'tax_identification_number',
        'industry',
        'annual_revenue',
        'number_of_employees',
        'established_year',
        'discount_tier',
        'default_payment_terms',
        'credit_limit',
        'is_verified',
        'verification_date',
        'verified_by',
        'notes'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'annual_revenue' => 'decimal:2',
        'number_of_employees' => 'integer',
        'established_year' => 'integer',
        'default_payment_terms' => 'integer',
        'credit_limit' => 'decimal:2',
        'is_verified' => 'boolean',
        'verification_date' => 'datetime',
    ];

    /**
     * Get the user that owns the profile.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the admin who verified this profile.
     */
    public function verifier()
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    /**
     * Get the membership for the customer.
     */
    public function membership()
    {
        return $this->belongsTo(CustomerMembership::class, 'membership_id');
    }

    /**
     * Get all addresses for the customer.
     */
    public function addresses()
    {
        return $this->hasMany(CustomerAddress::class, 'user_id', 'user_id');
    }

    /**
     * Get all payment methods for the customer.
     */
    public function paymentMethods()
    {
        return $this->hasMany(CustomerPaymentMethod::class, 'user_id', 'user_id');
    }

    /**
     * Get all documents for the customer.
     */
    public function documents()
    {
        return $this->hasMany(CustomerDocument::class, 'user_id', 'user_id');
    }

    /**
     * Scope a query to only include verified customers.
     */
    public function scopeVerified($query)
    {
        return $query->where('is_verified', true);
    }

    /**
     * Scope a query to filter by discount tier.
     */
    public function scopeByDiscountTier($query, $tier)
    {
        return $query->where('discount_tier', $tier);
    }
}