<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CustomerAddress extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'address_type',
        'is_default',
        'company_name',
        'contact_name',
        'contact_email',
        'contact_phone',
        'address_line1',
        'address_line2',
        'city',
        'state',
        'postal_code',
        'country',
        'delivery_instructions'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_default' => 'boolean',
    ];

    /**
     * Get the user that owns the address.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the customer profile associated with this address.
     */
    public function customerProfile()
    {
        return $this->belongsTo(CustomerProfile::class, 'user_id', 'user_id');
    }

    /**
     * Get payment methods using this address for billing.
     */
    public function paymentMethods()
    {
        return $this->hasMany(CustomerPaymentMethod::class, 'billing_address_id');
    }

    /**
     * Scope a query to only include default addresses.
     */
    public function scopeDefault($query)
    {
        return $query->where('is_default', true);
    }

    /**
     * Scope a query to filter addresses by type.
     */
    public function scopeOfType($query, $type)
    {
        if ($type === 'both') {
            return $query->where('address_type', 'both');
        }
        
        return $query->where(function($q) use ($type) {
            $q->where('address_type', $type)
              ->orWhere('address_type', 'both');
        });
    }

    /**
     * Format the full address as a string.
     */
    public function getFullAddressAttribute()
    {
        $address = $this->address_line1;
        
        if ($this->address_line2) {
            $address .= ', ' . $this->address_line2;
        }
        
        $address .= ', ' . $this->city;
        $address .= ', ' . $this->state;
        $address .= ' ' . $this->postal_code;
        $address .= ', ' . $this->country;
        
        return $address;
    }
}