<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Currency extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'code',
        'name',
        'symbol',
        'is_default',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_default' => 'boolean',
    ];

    /**
     * The primary key for the model.
     *
     * @var string
     */
    protected $primaryKey = 'id';

    /**
     * Get the products that use this currency.
     */
    public function products()
    {
        return $this->hasMany(Product::class, 'currency_code', 'code');
    }

    /**
     * Boot function from Laravel.
     */
    protected static function boot()
    {
        parent::boot();
        
        // When setting a currency as default, unset all others
        static::saving(function($currency) {
            if ($currency->is_default) {
                // Unset default status for all other currencies
                static::where('id', '!=', $currency->id)
                    ->where('is_default', true)
                    ->update(['is_default' => false]);
            }
        });
    }

    /**
     * Scope a query to only include the default currency.
     */
    public function scopeDefault($query)
    {
        return $query->where('is_default', true);
    }

    /**
     * Format the given amount using this currency's symbol.
     *
     * @param float $amount
     * @return string
     */
    public function format($amount)
    {
        return $this->symbol . number_format($amount, 2);
    }

    /**
     * Get the default currency.
     *
     * @return Currency|null
     */
    public static function getDefault()
    {
        return static::where('is_default', true)->first() ?? static::first();
    }

    /**
     * Get the default currency code.
     *
     * @return string
     */
    public static function getDefaultCode()
    {
        $defaultCurrency = static::getDefault();
        return $defaultCurrency ? $defaultCurrency->code : 'AUD';
    }
}