<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CartItem extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'cart_id',
        'product_id',
        'quantity',
        'unit_price',
        'subtotal',
        'discount_amount',
        'tax_amount',
        'total_amount',
        'is_bulk_pricing',
        'is_pack_pricing',
        'bulk_price_id',
        'pack_price_id',
        'notes',
        'customer_margin',
        'partner_margin',
        'customer_margin_type',
        'partner_margin_type',
        'membership_discount_amount',
        'membership_discount_percentage',
        'original_price_before_membership',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'quantity' => 'integer',
        'unit_price' => 'decimal:2',
        'subtotal' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'is_bulk_pricing' => 'boolean',
        'is_pack_pricing' => 'boolean',
        'customer_margin' => 'decimal:2',
        'partner_margin' => 'decimal:2',
        'membership_discount_amount' => 'decimal:2',
        'membership_discount_percentage' => 'decimal:2',
        'original_price_before_membership' => 'decimal:2',
    ];

    /**
     * The attributes that should be hidden for arrays.
     */
    protected $hidden = [
        'product',
        'bulk_price',
        'pack_price',
        'bulk_price_id',
        'pack_price_id',
        'tax_amount',
        'total_amount',
        'base_price_without_margin',
        'base_price_with_margin',
        'special_price_without_margin',
        'special_price_with_margin',
        'formatted_product'
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        'formatted_quantity',
        'formatted_product',
        'base_unit_price_with_margin',
        'special_unit_price_with_margin',
        'subtotal'
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $with = ['product:id,name,title,sku,quantity,stock_status', 'product.images:id,product_id,image_url,alt_text,sort_order,created_at,updated_at'];

    /**
     * Get the cart that owns the item.
     */
    public function cart()
    {
        return $this->belongsTo(Cart::class);
    }

    /**
     * Get the product for this cart item.
     */
    public function product()
    {
        return $this->belongsTo(Product::class)
            ->with(['images' => function ($query) {
                $query->select('id', 'product_id', 'image_url', 'alt_text', 'sort_order', 'created_at', 'updated_at');
            }]);
    }

    /**
     * Get the pack price for this cart item if using pack pricing.
     */
    public function packPrice()
    {
        return $this->belongsTo(ProductPackPrice::class, 'pack_price_id');
    }

    /**
     * Get the bulk price for this cart item if using bulk pricing.
     */
    public function bulkPrice()
    {
        return $this->belongsTo(ProductBulkPrice::class, 'bulk_price_id');
    }

    /**
     * Set quantity and recalculate prices.
     */
    public function setQuantityAndRecalculate($quantity)
    {
        $this->quantity = $quantity;
        
        return $this->calculatePricing();
    }

    /**
     * Select the best bulk price based on quantity.
     */
    public function selectBestBulkPrice()
    {
        $product = $this->product;
        if (!$product) {
            return null;
        }

        // Get all bulk prices for the product, ordered by number_of_packs
        $bulkPrices = $product->bulkPrices()
            ->orderBy('number_of_packs', 'asc')
            ->get();

        if ($bulkPrices->isEmpty()) {
            return null;
        }

        // Find the highest applicable bulk price
        $bestBulkPrice = null;
        foreach ($bulkPrices as $bulkPrice) {
            if ($this->quantity >= $bulkPrice->number_of_packs) {
                $bestBulkPrice = $bulkPrice;
            } else {
                break;
            }
        }

        return $bestBulkPrice;
    }

    /**
     * Get the base price for this item.
     */
    public function getBasePriceAttribute()
    {
        if ($this->is_bulk_pricing && $this->bulkPrice) {
            return $this->bulkPrice->per_pack_price;
        } elseif ($this->is_pack_pricing && $this->packPrice) {
            return $this->packPrice->per_pack_price;
        }
        return $this->product->base_price;
    }

    /**
     * Get the customer price for this item.
     */
    public function getCustomerPriceAttribute()
    {
        $basePrice = $this->base_price;
        
        if ($this->customer_margin_type === 'percentage') {
            return round($basePrice * (1 + ($this->customer_margin / 100)), 2);
        } else {
            return round($basePrice + $this->customer_margin, 2);
        }
    }

    /**
     * Get the partner price for this item.
     */
    public function getPartnerPriceAttribute()
    {
        $basePrice = $this->base_price;
        
        if ($this->partner_margin_type === 'percentage') {
            return round($basePrice * (1 + ($this->partner_margin / 100)), 2);
        } else {
            return round($basePrice + $this->partner_margin, 2);
        }
    }

    /**
     * Calculate pricing based on quantity and pricing type.
     */
    public function calculatePricing()
    {
        // Try to find the best bulk price based on quantity
        $bestBulkPrice = $this->selectBestBulkPrice();
        
        if ($bestBulkPrice) {
            // Use bulk pricing
            $this->is_bulk_pricing = true;
            $this->bulk_price_id = $bestBulkPrice->id;
            $this->is_pack_pricing = false;
            $this->pack_price_id = null;
            
            // Use regular price from bulk price
            $this->unit_price = $bestBulkPrice->per_pack_price;
            $this->subtotal = $this->unit_price * $this->quantity;
            
            // Set margins from bulk price
            $this->customer_margin = $bestBulkPrice->customer_margin;
            $this->partner_margin = $bestBulkPrice->partner_margin;
            $this->customer_margin_type = $bestBulkPrice->customer_margin_type;
            $this->partner_margin_type = $bestBulkPrice->partner_margin_type;
            
            // Set discount amount to 0
            $this->discount_amount = 0;
        } elseif ($this->product->packPrice) {
            // Use pack pricing
            $this->is_pack_pricing = true;
            $this->pack_price_id = $this->product->packPrice->id;
            $this->is_bulk_pricing = false;
            $this->bulk_price_id = null;
            
            // Use regular price from pack price
            $this->unit_price = $this->product->packPrice->per_pack_price;
            $this->subtotal = $this->unit_price * $this->quantity;
            
            // Set margins from pack price
            $this->customer_margin = $this->product->packPrice->customer_margin;
            $this->partner_margin = $this->product->packPrice->partner_margin;
            $this->customer_margin_type = $this->product->packPrice->customer_margin_type;
            $this->partner_margin_type = $this->product->packPrice->partner_margin_type;
            
            // Set discount amount to 0
            $this->discount_amount = 0;
        } else {
            // Use default product pricing
            $this->is_bulk_pricing = false;
            $this->bulk_price_id = null;
            $this->is_pack_pricing = false;
            $this->pack_price_id = null;
            
            // Use base price from product
            $this->unit_price = $this->product->base_price;
            $this->subtotal = $this->unit_price * $this->quantity;
            
            // Set margins from product defaults
            $this->customer_margin = $this->product->default_customer_margin;
            $this->partner_margin = $this->product->default_partner_margin;
            $this->customer_margin_type = $this->product->default_customer_margin_type;
            $this->partner_margin_type = $this->product->default_partner_margin_type;
            
            // Set discount amount to 0
            $this->discount_amount = 0;
        }
        
        // Calculate tax based on product-specific tax rate
        $taxRate = $this->getTaxRate();
        $taxableAmount = $this->subtotal - $this->discount_amount;
        $this->tax_amount = round($taxableAmount * ($taxRate / 100), 2);
        
        // Calculate total amount (subtotal + tax + delivery fee if any)
        $deliveryFee = $this->delivery_fee ?? 0;
        $this->total_amount = $this->subtotal - $this->discount_amount + $this->tax_amount + $deliveryFee;
        
        // Save the changes
        $this->save();
        
        // Note: calculateTotals() should be called by the controller with user parameter
        // to ensure proper membership discount calculations
        
        return $this;
    }

    /**
     * Calculate the margin amount for customer.
     */
    public function getCustomerMarginAmountAttribute()
    {
        if ($this->customer_margin_type === 'percentage') {
            return round(($this->unit_price * $this->customer_margin) / 100, 2);
        } else {
            return $this->customer_margin;
        }
    }

    /**
     * Calculate the margin amount for partner.
     */
    public function getPartnerMarginAmountAttribute()
    {
        if ($this->partner_margin_type === 'percentage') {
            return round(($this->unit_price * $this->partner_margin) / 100, 2);
        } else {
            return $this->partner_margin;
        }
    }

    /**
     * Get the minimum order quantity for this item.
     */
    public function getMinOrderQuantityAttribute()
    {
        return $this->product ? $this->product->min_order_quantity : null;
    }

    /**
     * Get the delivery fee for this item.
     */
    public function getDeliveryFeeAttribute()
    {
        // Always check for bulk price delivery fee if bulk_price_id is set
        if ($this->bulk_price_id && $this->bulkPrice && $this->bulkPrice->delivery_fee !== null) {
            return $this->bulkPrice->delivery_fee;
        }
        if ($this->is_pack_pricing && $this->packPrice && $this->packPrice->delivery_fee !== null) {
            return $this->packPrice->delivery_fee;
        }
        return null;
    }

    /**
     * Get the tax rate for this cart item (inherits from product level)
     */
    public function getTaxRate()
    {
        // Use bulk price specific tax rate if bulk pricing is active and bulk price has tax rate
        if ($this->is_bulk_pricing && $this->bulkPrice && !is_null($this->bulkPrice->tax_rate)) {
            return $this->bulkPrice->tax_rate;
        }
        
        // Use pack price specific tax rate if pack pricing is active and pack price has tax rate
        if ($this->is_pack_pricing && $this->packPrice && !is_null($this->packPrice->tax_rate)) {
            return $this->packPrice->tax_rate;
        }
        
        // Fall back to product tax rate
        return $this->product->getTaxRate();
    }

    /**
     * Get the base price without any margin applied.
     */
    public function getBasePriceWithoutMarginAttribute()
    {
        if ($this->is_bulk_pricing && $this->bulkPrice) {
            return $this->bulkPrice->per_pack_price;
        } elseif ($this->is_pack_pricing && $this->packPrice) {
            return $this->packPrice->per_pack_price;
        }
        return $this->product->base_price;
    }

    /**
     * Helper method to determine which margin to use based on user role
     */
    public function getMarginForUser($user = null)
    {
        if ($user && ($user->hasRole('partner') || $user->hasRole('vendor'))) {
            return [
                'value' => $this->partner_margin,
                'type' => $this->partner_margin_type
            ];
        }
        
        return [
            'value' => $this->customer_margin,
            'type' => $this->customer_margin_type
        ];
    }

    /**
     * Get the base price with margin applied (role-based).
     */
    public function getBasePriceWithMarginAttribute($user = null)
    {
        $basePrice = $this->unit_price;
        $marginData = $this->getMarginForUser($user);
        $margin = $marginData['value'];
        $marginType = $marginData['type'];

        if ($marginType === 'percentage') {
            $priceWithMargin = $basePrice * (1 + ($margin / 100));
        } else {
            $priceWithMargin = $basePrice + $margin;
        }

        return round($priceWithMargin * $this->quantity, 2);
    }

    /**
     * Get the special price without any margin applied.
     */
    public function getSpecialPriceWithoutMarginAttribute()
    {
        if ($this->is_bulk_pricing && $this->bulkPrice) {
            return $this->bulkPrice->per_pack_special_price;
        } elseif ($this->is_pack_pricing && $this->packPrice) {
            return $this->packPrice->per_pack_special_price;
        }
        return $this->product->current_price;
    }

    /**
     * Get the special price with margin (role-based).
     */
    public function getSpecialPriceWithMarginAttribute($user = null)
    {
        if ($this->is_bulk_pricing) {
            $price = $this->bulkPrice?->per_pack_special_price ?? 0;
        } else {
            $price = $this->packPrice?->per_pack_special_price ?? 0;
        }

        if (!$price) {
            return null;
        }

        $marginData = $this->getMarginForUser($user);
        $margin = $marginData['value'] ?? 0;
        $marginType = $marginData['type'] ?? 'percentage';

        if ($marginType === 'percentage') {
            return round($price * (1 + ($margin / 100)) * $this->quantity, 2);
        }

        return round(($price + $margin) * $this->quantity, 2);
    }

    /**
     * Get the total base price for the quantity.
     */
    public function getTotalBasePriceAttribute()
    {
        return round($this->base_price_without_margin * $this->quantity, 2);
    }

    /**
     * Get the total special price for the quantity.
     */
    public function getTotalSpecialPriceAttribute()
    {
        return round($this->special_price_without_margin * $this->quantity, 2);
    }

    /**
     * Get the formatted quantity.
     */
    public function getFormattedQuantityAttribute()
    {
        return [
            'value' => $this->quantity,
            'unit' => 'pcs',
            'display' => $this->quantity . ' pcs'
        ];
    }

    /**
     * Get formatted product data.
     */
    public function getFormattedProductAttribute()
    {
        return [
            'id' => $this->product->id,
            'name' => $this->product->name,
            'title' => $this->product->title,
            'sku' => $this->product->sku,
            'quantity' => $this->product->quantity,
            'stock_status' => $this->product->stock_status,
            'images' => $this->product->images->map(function ($image) {
                return [
                    'id' => $image->id,
                    'product_id' => $image->product_id,
                    'image_url' => $image->image_url,
                    'alt_text' => $image->alt_text,
                    'sort_order' => $image->sort_order,
                    'created_at' => $image->created_at,
                    'updated_at' => $image->updated_at,
                    'full_image_url' => $image->full_image_url
                ];
            })
        ];
    }

    /**
     * Get the cart items.
     */
    public function items()
    {
        return $this->hasMany(CartItem::class)
            ->with(['product:id,name,title,slug,description,sku,quantity,stock_status,currency_code'])
            ->select([
                'id', 'cart_id', 'product_id', 'quantity', 'unit_price',
                'is_bulk_pricing', 'is_pack_pricing', 'notes',
                'customer_margin', 'partner_margin', 'customer_margin_type',
                'partner_margin_type', 'created_at', 'updated_at'
            ]);
    }

    public function getBaseUnitPriceWithMarginAttribute()
    {
        if ($this->customer_margin_type === 'percentage') {
            return round($this->base_price_without_margin * (1 + ($this->customer_margin / 100)), 2);
        }
        return round($this->base_price_without_margin + $this->customer_margin, 2);
    }

    public function getSpecialUnitPriceWithMarginAttribute()
    {
        if (!$this->special_price_without_margin) {
            return null;
        }

        if ($this->customer_margin_type === 'percentage') {
            return round($this->special_price_without_margin * (1 + ($this->customer_margin / 100)), 2);
        }
        return round($this->special_price_without_margin + $this->customer_margin, 2);
    }

    public function getSubtotalAttribute()
    {
        if ($this->special_unit_price_with_margin !== null) {
            return round($this->special_unit_price_with_margin * $this->quantity, 2);
        }
        return round($this->base_unit_price_with_margin * $this->quantity, 2);
    }
}