<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Cart extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'session_id',
        'notes',
        'subtotal',
        'discount_amount',
        'tax_amount', 
        'shipping_amount',
        'total_amount',
        'currency_code',
        'is_active',
        'expires_at',
        'membership_discount_amount',
        'membership_discount_percentage',
        'applied_membership_id',
        'membership_discount_description',
        'membership_discount_applied_at',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'subtotal' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'shipping_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'membership_discount_amount' => 'decimal:2',
        'membership_discount_percentage' => 'decimal:2',
        'is_active' => 'boolean',
        'expires_at' => 'datetime',
        'membership_discount_applied_at' => 'datetime',
    ];

    /**
     * The attributes that should be appended to arrays.
     */
    protected $appends = [
        'total_amount',
        'total_amount_with_tax',
        'total_delivery_fees',
        'final_price'
    ];

    /**
     * The attributes that should be hidden from arrays.
     */
    protected $hidden = [
        'subtotal',
        'shipping_amount',
        'formatted_total'
    ];

    /**
     * Get the user that owns the cart.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the cart items.
     */
    public function items()
    {
        return $this->hasMany(CartItem::class)
            ->with([
                'product:id,name,title,sku,quantity,stock_status',
                'product.images:id,product_id,image_url,alt_text,sort_order,created_at,updated_at'
            ])
            ->select([
                'id', 'cart_id', 'product_id', 'quantity', 'unit_price', 'subtotal',
                'discount_amount', 'is_bulk_pricing', 'is_pack_pricing', 'notes',
                'customer_margin', 'partner_margin', 'customer_margin_type', 'partner_margin_type',
                'created_at', 'updated_at'
            ]);
    }

    /**
     * Get the currency used for this cart.
     */
    public function currency()
    {
        return $this->belongsTo(Currency::class, 'currency_code', 'code');
    }

    /**
     * Get the applied membership for this cart.
     */
    public function appliedMembership()
    {
        return $this->belongsTo(CustomerMembership::class, 'applied_membership_id');
    }

    /**
     * Get the user's active membership.
     */
    public function getUserMembership()
    {
        $user = $this->user;
        if (!$user) return null;
        
        return $user->customerProfile?->membership;
    }

    /**
     * Calculate cart totals.
     */
    public function calculateTotals($user = null)
    {
        $subtotal = 0;
        $taxAmount = 0;
        $discountAmount = 0;
        $shippingAmount = 0;

        foreach ($this->items as $item) {
            // Determine which margin to use based on user role
            if ($user && ($user->hasRole('partner') || $user->hasRole('vendor'))) {
                $margin = $item->partner_margin;
                $marginType = $item->partner_margin_type;
            } else {
                $margin = $item->customer_margin;
                $marginType = $item->customer_margin_type;
            }
            
            // Calculate price with appropriate margin
            $basePrice = $item->unit_price;
            if ($marginType === 'percentage') {
                $priceWithMargin = $basePrice * (1 + ($margin / 100));
            } else {
                $priceWithMargin = $basePrice + $margin;
            }
            
            $itemSubtotal = $priceWithMargin * $item->quantity;
            
            $subtotal += $itemSubtotal;
            
            // Calculate tax for this item based on product-specific tax rate
            $taxRate = $item->getTaxRate();
            $taxableAmount = $itemSubtotal - $item->discount_amount;
            $taxAmount += round($taxableAmount * ($taxRate / 100), 2);
            
            $discountAmount += $item->discount_amount;
            $shippingAmount += $item->delivery_fee ?? 0;
        }

        // Calculate membership discount
        $membershipDiscount = $this->calculateMembershipDiscount($user);
        $this->membership_discount_amount = $membershipDiscount['amount'];
        $this->membership_discount_percentage = $membershipDiscount['percentage'];
        
        // Update applied membership info if user has membership
        if ($user && $user->hasMembership()) {
            $membership = $user->getActiveMembership();
            $this->applied_membership_id = $membership->id;
            $this->membership_discount_description = "{$membership->name} discount ({$membership->extra_discount}%)";
            $this->membership_discount_applied_at = now();
        } else {
            $this->applied_membership_id = null;
            $this->membership_discount_description = null;
            $this->membership_discount_applied_at = null;
        }

        // Recalculate tax after membership discount (proportional to each item's tax rate)
        $taxAmountAfterMembershipDiscount = 0;
        foreach ($this->items as $item) {
            $taxRate = $item->getTaxRate();
            
            // Calculate item subtotal with appropriate margin based on user role
            if ($user && ($user->hasRole('partner') || $user->hasRole('vendor'))) {
                $margin = $item->partner_margin;
                $marginType = $item->partner_margin_type;
            } else {
                $margin = $item->customer_margin;
                $marginType = $item->customer_margin_type;
            }
            
            // Calculate price with appropriate margin
            $basePrice = $item->unit_price;
            if ($marginType === 'percentage') {
                $priceWithMargin = $basePrice * (1 + ($margin / 100));
            } else {
                $priceWithMargin = $basePrice + $margin;
            }
            
            $itemSubtotal = $priceWithMargin * $item->quantity;
            $itemMembershipDiscount = $this->calculateItemMembershipDiscount($itemSubtotal, $user)['amount'];
            $taxableAmountAfterDiscount = $itemSubtotal - $itemMembershipDiscount;
            $taxAmountAfterMembershipDiscount += round($taxableAmountAfterDiscount * ($taxRate / 100), 2);
        }

        $this->subtotal = $subtotal;
        $this->tax_amount = $taxAmountAfterMembershipDiscount;
        $this->discount_amount = $discountAmount;
        $this->shipping_amount = $shippingAmount;
        $this->total_amount = $subtotal + $taxAmountAfterMembershipDiscount + $shippingAmount - $discountAmount - $this->membership_discount_amount;
        $this->save();
    }

    /**
     * Calculate membership discount for the cart.
     */
    public function calculateMembershipDiscount($user = null)
    {
        if (!$user) return ['amount' => 0, 'percentage' => 0];
        
        $membership = $user->getActiveMembership();
        if (!$membership) return ['amount' => 0, 'percentage' => 0];
        
        // Calculate subtotal before membership discount
        $subtotal = 0;
        foreach ($this->items as $item) {
            // Determine which margin to use based on user role
            if ($user && ($user->hasRole('partner') || $user->hasRole('vendor'))) {
                $margin = $item->partner_margin;
                $marginType = $item->partner_margin_type;
            } else {
                $margin = $item->customer_margin;
                $marginType = $item->customer_margin_type;
            }
            
            // Calculate price with appropriate margin
            $basePrice = $item->unit_price;
            if ($marginType === 'percentage') {
                $priceWithMargin = $basePrice * (1 + ($margin / 100));
            } else {
                $priceWithMargin = $basePrice + $margin;
            }
            
            $itemSubtotal = $priceWithMargin * $item->quantity;
            $subtotal += $itemSubtotal;
        }
        
        // Apply membership extra_discount to total_amount (subtotal only)
        $discountPercentage = $membership->extra_discount;
        $discountAmount = ($subtotal * $discountPercentage) / 100;
        
        return [
            'amount' => round($discountAmount, 2),
            'percentage' => $discountPercentage
        ];
    }

    /**
     * Calculate membership discount for a specific item.
     */
    private function calculateItemMembershipDiscount($itemSubtotal, $user)
    {
        if (!$user) return ['amount' => 0, 'percentage' => 0];
        
        $membership = $user->getActiveMembership();
        if (!$membership) return ['amount' => 0, 'percentage' => 0];
        
        $discountPercentage = $membership->extra_discount;
        $discountAmount = ($itemSubtotal * $discountPercentage) / 100;
        
        return [
            'amount' => round($discountAmount, 2),
            'percentage' => $discountPercentage
        ];
    }

    /**
     * Scope a query to only include active carts.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include carts for a specific user.
     */
    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope a query to only include carts that haven't expired.
     */
    public function scopeNotExpired($query)
    {
        return $query->where(function($q) {
            $q->whereNull('expires_at')
              ->orWhere('expires_at', '>', now());
        });
    }

    /**
     * Get active cart for a user, or create one if it doesn't exist.
     */
    public static function getOrCreateForUser($userId, $currencyCode = 'AUD')
    {
        // Find active cart for user
        $cart = self::forUser($userId)->active()->notExpired()->first();
        
        // If no active cart exists, create a new one
        if (!$cart) {
            $cart = self::create([
                'user_id' => $userId,
                'currency_code' => $currencyCode,
                'is_active' => true,
                'subtotal' => 0,
                'discount_amount' => 0,
                'tax_amount' => 0,
                'shipping_amount' => 0,
                'total_amount' => 0
            ]);
        }
        
        return $cart;
    }

    /**
     * Check if the cart has sufficient inventory for all items.
     */
    public function hasSufficientInventory()
    {
        foreach ($this->items as $item) {
            $product = $item->product;
            
            if ($product->quantity < $item->quantity) {
                return false;
            }
        }
        
        return true;
    }

    /**
     * Check if the cart is empty.
     */
    public function isEmpty()
    {
        return $this->items->isEmpty();
    }

    /**
     * Get the total delivery fees.
     */
    public function getTotalDeliveryFeesAttribute()
    {
        return $this->items->sum(function ($item) {
            if ($item->is_pack_pricing && $item->pack_price) {
                return $item->pack_price->delivery_fee ?? 0;
            } elseif ($item->is_bulk_pricing && $item->bulk_price) {
                return $item->bulk_price->delivery_fee ?? 0;
            }
            return 0;
        });
    }

    /**
     * Get the formatted total amount.
     */
    public function getFormattedTotalAttribute()
    {
        return $this->currency->symbol . number_format($this->total_amount, 2);
    }

    public function getTotalAmountAttribute()
    {
        $subtotal = (float) $this->items->sum('subtotal');
        $membershipDiscount = (float) ($this->membership_discount_amount ?? 0);
        $afterDiscount = round($subtotal - $membershipDiscount, 2);
        return $afterDiscount < 0 ? 0 : $afterDiscount;
    }

    public function getTotalAmountWithTaxAttribute()
    {
        // Recompute tax after membership discount proportionally per item using stored percentage
        $membershipPercentage = (float) ($this->membership_discount_percentage ?? 0);
        $taxAfterMembership = 0.0;
        foreach ($this->items as $item) {
            $taxRate = method_exists($item, 'getTaxRate') ? (float) $item->getTaxRate() : 0.0;
            $itemSubtotal = (float) $item->subtotal;
            $itemDiscount = (float) $item->discount_amount;
            $itemMembershipDiscount = round($itemSubtotal * ($membershipPercentage / 100), 2);
            $taxable = $itemSubtotal - $itemDiscount - $itemMembershipDiscount;
            if ($taxable < 0) { $taxable = 0; }
            $taxAfterMembership += round($taxable * ($taxRate / 100), 2);
        }
        $totalAfterMembership = (float) $this->total_amount; // already post-membership
        return round($totalAfterMembership + $taxAfterMembership, 2);
    }

    public function getFinalPriceAttribute()
    {
        $totalDeliveryFees = $this->total_delivery_fees;
        return round($this->total_amount_with_tax + $totalDeliveryFees, 2);
    }
}