<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class VendorProfileController extends Controller
{
    /**
     * Get vendor profile details.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getVendorProfile(Request $request)
    {
        $user = $request->user();

        if (!$user->isVendor()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
                'details' => 'Only vendors can access this endpoint'
            ], 403);
        }

        $vendorProfile = $user->vendorProfile;

        if (!$vendorProfile) {
            return response()->json([
                'success' => false,
                'message' => 'Not Found',
                'details' => 'Vendor profile not found'
            ], 404);
        }

        // Load the vendor profile with all necessary relationships
        $vendorProfile->load(['user']);

        return response()->json([
            'success' => true,
            'data' => [
                'vendor_profile' => [
                    'id' => $vendorProfile->id,
                    'user_id' => $vendorProfile->user_id,
                    'company_name' => $vendorProfile->company_name,
                    'company_website' => $vendorProfile->company_website,
                    'business_type' => $vendorProfile->business_type,
                    'business_registration_number' => $vendorProfile->business_registration_number,
                    'tax_identification_number' => $vendorProfile->tax_identification_number,
                    'industry' => $vendorProfile->industry,
                    'annual_revenue' => $vendorProfile->annual_revenue,
                    'number_of_employees' => $vendorProfile->number_of_employees,
                    'established_year' => $vendorProfile->established_year,
                    'notes' => $vendorProfile->notes,
                    'is_approved' => $vendorProfile->is_approved,
                    'created_at' => $vendorProfile->created_at,
                    'updated_at' => $vendorProfile->updated_at,
                    'user' => [
                        'id' => $vendorProfile->user->id,
                        'name' => $vendorProfile->user->name,
                        'email' => $vendorProfile->user->email,
                        'phone' => $vendorProfile->user->phone,
                        'address' => $vendorProfile->user->address
                    ]
                ]
            ]
        ]);
    }

    /**
     * Update vendor profile.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $user = $request->user();

        if (!$user->isVendor()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
                'details' => 'Only vendors can access this endpoint'
            ], 403);
        }

        $vendorProfile = $user->vendorProfile;

        if (!$vendorProfile) {
            return response()->json([
                'success' => false,
                'message' => 'Not Found',
                'details' => 'Vendor profile not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'company_name' => 'nullable|string|max:255',
            'company_website' => 'nullable|url|max:255',
            'business_type' => 'nullable|string|max:100',
            'business_registration_number' => 'nullable|string|max:100',
            'tax_identification_number' => 'nullable|string|max:100',
            'industry' => 'nullable|string|max:100',
            'annual_revenue' => 'nullable|numeric',
            'number_of_employees' => 'nullable|integer',
            'established_year' => 'nullable|integer|min:1900|max:' . (date('Y') + 1),
            'notes' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $vendorProfile->update($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Vendor profile updated successfully',
                'data' => [
                    'vendor_profile' => $vendorProfile
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating vendor profile',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Assign or update a vendor's membership.
     * Accessible by admin/superadmin only.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $vendorProfileId
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateMembership(Request $request, $vendorProfileId)
    {
        $request->validate([
            'membership_id' => 'required|exists:vendor_memberships,id',
        ]);

        $vendorProfile = \App\Models\VendorProfile::findOrFail($vendorProfileId);
        $vendorProfile->membership_id = $request->membership_id;
        $vendorProfile->save();

        // Optionally load membership info
        $vendorProfile->load('membership');

        return response()->json([
            'success' => true,
            'message' => 'Vendor membership updated.',
            'data' => $vendorProfile
        ]);
    }
} 