<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\FulfillmentOrder;
use App\Models\FulfillmentItem;
use App\Models\Shipment;
use App\Models\ShipmentItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class VendorOrderController extends Controller
{
    /**
     * Display a paginated listing of orders containing the vendor's products.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $vendorId = $request->user()->id;
            
            // Query orders that contain the vendor's products
            $query = Order::whereHas('items.product', function($query) use ($vendorId) {
                $query->where('vendor_id', $vendorId);
            })
            ->with([
                'items' => function($query) use ($vendorId) {
                    $query->whereHas('product', function($q) use ($vendorId) {
                        $q->where('vendor_id', $vendorId);
                    })->with('product');
                },
                'history',
                'user:id,name,email'
            ])
            ->orderBy('created_at', 'desc');
            
            // Apply filters if provided
            if ($request->has('status')) {
                $query->where('status', $request->status);
            }
            
            if ($request->has('date_from')) {
                $query->where('created_at', '>=', $request->date_from);
            }
            
            if ($request->has('date_to')) {
                $query->where('created_at', '<=', $request->date_to . ' 23:59:59');
            }
            
            // Search by order number (partial match)
            if ($request->has('order_number')) {
                $orderNumber = $request->order_number;
                $query->where('order_number', 'like', "%{$orderNumber}%");
            }
            
            // Paginate the results
            $orders = $query->paginate(10);
            
            // Calculate vendor-specific totals for each order
            $orders->getCollection()->transform(function ($order) use ($vendorId) {
                $vendorItems = $order->items;
                
                $order->vendor_subtotal = $vendorItems->sum('subtotal');
                $order->vendor_discount = $vendorItems->sum('discount_amount');
                $order->vendor_tax = $vendorItems->sum('tax_amount');
                $order->vendor_total = $vendorItems->sum('total_amount');
                
                // Get latest fulfillment status
                $fulfillmentOrders = $order->fulfillmentOrders()
                    ->whereHas('items.orderItem.product', function($query) use ($vendorId) {
                        $query->where('vendor_id', $vendorId);
                    })->get();
                
                $order->fulfillment_status = $fulfillmentOrders->isNotEmpty() 
                    ? $fulfillmentOrders->first()->status 
                    : 'pending';
                
                // Get latest shipment status
                $shipments = $order->shipments()
                    ->whereHas('items.orderItem.product', function($query) use ($vendorId) {
                        $query->where('vendor_id', $vendorId);
                    })->get();
                
                $order->vendor_shipping_status = $shipments->isNotEmpty() 
                    ? $shipments->first()->status 
                    : null;
                
                return $order;
            });
            
            return response()->json([
                'success' => true,
                'data' => [
                    'orders' => $orders
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error retrieving vendor orders', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving orders',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }

    /**
     * Display the specified order with the vendor's items.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Request $request, $id)
    {
        try {
            $vendorId = $request->user()->id;
            
            // Find the order with vendor's items
            $order = Order::where('id', $id)
                ->whereHas('items.product', function($query) use ($vendorId) {
                    $query->where('vendor_id', $vendorId);
                })
                ->with([
                    'items' => function($query) use ($vendorId) {
                        // Only load items that belong to this vendor
                        $query->whereHas('product', function($q) use ($vendorId) {
                            $q->where('vendor_id', $vendorId);
                        })->with('product');
                    },
                    'history',
                    'user:id,name,email'
                ])
                ->firstOrFail();
            
            // Calculate vendor-specific totals
            $vendorItems = $order->items;
            
            $order->vendor_subtotal = $vendorItems->sum('subtotal');
            $order->vendor_discount = $vendorItems->sum('discount_amount');
            $order->vendor_tax = $vendorItems->sum('tax_amount');
            $order->vendor_total = $vendorItems->sum('total_amount');
            
            // Get fulfillment orders where the assigned items belong to this vendor
            // This assumes FulfillmentOrder doesn't have vendor_id but has a relationship to items
            $fulfillmentOrders = \App\Models\FulfillmentOrder::whereHas('items.orderItem.product', function($query) use ($vendorId) {
                $query->where('vendor_id', $vendorId);
            })->where('order_id', $id)->with(['items' => function($query) use ($vendorId) {
                $query->whereHas('orderItem.product', function($q) use ($vendorId) {
                    $q->where('vendor_id', $vendorId);
                })->with('orderItem.product');
            }])->get();
            
            $order->vendor_fulfillment_orders = $fulfillmentOrders;
            
            // Get shipments where the order items belong to this vendor
            // This assumes Shipment doesn't have vendor_id but has a relationship to items
            $shipments = \App\Models\Shipment::whereHas('items.orderItem.product', function($query) use ($vendorId) {
                $query->where('vendor_id', $vendorId);
            })->where('order_id', $id)->with(['items' => function($query) use ($vendorId) {
                $query->whereHas('orderItem.product', function($q) use ($vendorId) {
                    $q->where('vendor_id', $vendorId);
                })->with('orderItem.product');
            }, 'tracking', 'deliveryAttempts'])->get();
            
            $order->vendor_shipments = $shipments;
            
            // Get latest fulfillment status
            $fulfillmentStatus = 'pending';
            if ($fulfillmentOrders->isNotEmpty()) {
                $fulfillmentStatus = $fulfillmentOrders->first()->status;
            }
            $order->fulfillment_status = $fulfillmentStatus;
            
            // Get latest shipment status
            $shippingStatus = null;
            if ($shipments->isNotEmpty()) {
                $shippingStatus = $shipments->first()->status;
            }
            $order->vendor_shipping_status = $shippingStatus;
            
            return response()->json([
                'success' => true,
                'data' => [
                    'order' => $order
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found or does not contain your products',
                'error' => 'Not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error retrieving vendor order', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $id,
                'vendor_id' => $request->user()->id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving order',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }

    /**
     * Update fulfillment status for vendor's items in an order.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateFulfillmentStatus(Request $request, $id)
    {
        try {
            $vendorId = $request->user()->id;
            
            // Validate the request
            $validator = Validator::make($request->all(), [
                'status' => 'required|in:pending,processing,ready_to_ship,shipped,delivered,cancelled',
                'notes' => 'nullable|string|max:500',
                'tracking_number' => 'nullable|string|max:255',
                'tracking_url' => 'nullable|url|max:255',
                'carrier' => 'nullable|string|max:100',
                'shipping_date' => 'nullable|date',
                'estimated_delivery_date' => 'nullable|date',
                'warehouse_id' => 'nullable|exists:warehouses,id', // Optional warehouse ID
                'use_warehouse' => 'nullable|boolean', // Flag to indicate if warehouse should be used
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }
            
            // Find the order with vendor's items
            $order = Order::whereHas('items.product', function($query) use ($vendorId) {
                $query->where('vendor_id', $vendorId);
            })->findOrFail($id);
            
            DB::beginTransaction();
            
            // Get vendor items for this order
            $vendorItems = OrderItem::whereHas('product', function($query) use ($vendorId) {
                $query->where('vendor_id', $vendorId);
            })->where('order_id', $id)->get();
            
            if ($vendorItems->isEmpty()) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'No items found for this vendor in the order',
                    'meta' => [
                        'code' => 404,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 404);
            }
            
            // Determine if we should use a warehouse
            $useWarehouse = $request->boolean('use_warehouse', false);
            $warehouseId = null;
            
            if ($useWarehouse) {
                // If warehouse_id is provided in the request, use it
                if ($request->has('warehouse_id')) {
                    $warehouseId = $request->warehouse_id;
                } else {
                    // Try to find a warehouse associated with this vendor
                    $vendorWarehouse = \App\Models\Warehouse::where('vendor_id', $vendorId)->first();
                    
                    if ($vendorWarehouse) {
                        $warehouseId = $vendorWarehouse->id;
                    } else {
                        // Get the first available warehouse (as fallback)
                        $firstWarehouse = \App\Models\Warehouse::where('is_active', true)->first();
                        
                        if ($firstWarehouse) {
                            $warehouseId = $firstWarehouse->id;
                        } else {
                            // If warehouse usage is requested but none found, return an error
                            DB::rollBack();
                            return response()->json([
                                'success' => false,
                                'message' => 'No warehouse available',
                                'details' => 'You requested to use a warehouse but no warehouse is available. Please provide a warehouse_id or contact support.',
                                'meta' => [
                                    'code' => 422,
                                    'timestamp' => now()->toIso8601String(),
                                ],
                            ], 422);
                        }
                    }
                }
            }
            
            // Create or update a fulfillment order for this vendor's items
            // First, let's see if there's already a fulfillment order for these items
            $fulfillmentOrderItems = \App\Models\FulfillmentItem::whereIn('order_item_id', $vendorItems->pluck('id'))
                ->get();
            
            $existingFulfillmentOrderIds = $fulfillmentOrderItems->pluck('fulfillment_order_id')->unique();
            
            $fulfillmentOrder = null;
            
            // If we found existing fulfillment items, get their parent fulfillment order
            if ($existingFulfillmentOrderIds->isNotEmpty()) {
                $fulfillmentOrder = \App\Models\FulfillmentOrder::find($existingFulfillmentOrderIds->first());
            }
            
            // If no existing fulfillment order, create a new one
            if (!$fulfillmentOrder) {
                // First let's modify the FulfillmentOrder model to make warehouse_id nullable
                // This is a temporary fix for testing - in production you may want to adjust your database schema
                
                // Option 1: Use createWithoutEvents to bypass validation temporarily
                // $fulfillmentOrder = \App\Models\FulfillmentOrder::withoutEvents(function() use ($id, $request, $vendorId, $warehouseId) {
                //     return \App\Models\FulfillmentOrder::create([
                //         'order_id' => $id,
                //         'status' => $request->status,
                //         'notes' => $request->notes,
                //         'priority' => 'normal',
                //         'assigned_to' => $vendorId,
                //         'warehouse_id' => $warehouseId,
                //     ]);
                // });
                
                // Option 2: Use raw query for testing (not recommended for production)
                if ($useWarehouse) {
                    // Create with warehouse
                    $fulfillmentOrder = \App\Models\FulfillmentOrder::create([
                        'order_id' => $id,
                        'status' => $request->status,
                        'notes' => $request->notes,
                        'priority' => 'normal',
                        'assigned_to' => $vendorId,
                        'warehouse_id' => $warehouseId,
                    ]);
                } else {
                    // Create without warehouse - using query builder to bypass validation
                    $fulfillmentOrderId = DB::table('fulfillment_orders')->insertGetId([
                        'order_id' => $id,
                        'status' => $request->status,
                        'notes' => $request->notes,
                        'priority' => 'normal',
                        'assigned_to' => $vendorId,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                    
                    $fulfillmentOrder = \App\Models\FulfillmentOrder::find($fulfillmentOrderId);
                }
                
            } else {
                // Update existing fulfillment order
                $fulfillmentOrder->status = $request->status;
                $fulfillmentOrder->notes = $request->notes ?? $fulfillmentOrder->notes;
                
                // Update warehouse if requested
                if ($useWarehouse && $warehouseId) {
                    $fulfillmentOrder->warehouse_id = $warehouseId;
                }
                
                $fulfillmentOrder->save();
            }
            
            // Create/update fulfillment items for each vendor item
            foreach ($vendorItems as $orderItem) {
                $fulfillmentItem = \App\Models\FulfillmentItem::firstOrNew([
                    'fulfillment_order_id' => $fulfillmentOrder->id,
                    'order_item_id' => $orderItem->id,
                ]);
                
                $fulfillmentItem->quantity_requested = $orderItem->quantity;
                
                if ($request->status === 'shipped' || $request->status === 'delivered') {
                    $fulfillmentItem->quantity_fulfilled = $orderItem->quantity;
                }
                
                $fulfillmentItem->status = $request->status;
                $fulfillmentItem->save();
            }
            
            // Handle shipping creation/updates if status is shipped or delivered
            $shipment = null;
            if (in_array($request->status, ['shipped', 'delivered'])) {
                // Check if we already have a shipment for these items
                $shipmentIds = \App\Models\ShipmentItem::whereIn('order_item_id', $vendorItems->pluck('id'))
                    ->pluck('shipment_id')
                    ->unique();
                    
                if ($shipmentIds->isNotEmpty()) {
                    $shipment = \App\Models\Shipment::find($shipmentIds->first());
                }
                
                $newShipment = false;
                if (!$shipment) {
                    // Create a new shipment
                    $shipment = new \App\Models\Shipment();
                    $shipment->order_id = $id;
                    $shipment->shipment_number = 'SHP-' . strtoupper(Str::random(8));
                    $shipment->created_by = $vendorId;
                    $newShipment = true;
                }
                
                // Update shipment details
                $shipment->carrier = $request->carrier ?? $shipment->carrier;
                $shipment->tracking_number = $request->tracking_number ?? $shipment->tracking_number;
                $shipment->tracking_url = $request->tracking_url ?? $shipment->tracking_url;
                $shipment->status = $request->status;
                
                if ($request->status === 'shipped' && $request->has('shipping_date')) {
                    $shipment->shipped_date = $request->shipping_date;
                } elseif ($request->status === 'shipped' && !$shipment->shipped_date) {
                    $shipment->shipped_date = now();
                }
                
                if ($request->has('estimated_delivery_date')) {
                    $shipment->estimated_delivery_date = $request->estimated_delivery_date;
                }
                
                if ($request->status === 'delivered' && !$shipment->delivery_date) {
                    $shipment->delivery_date = now();
                }
                
                $shipment->save();
                
                // Create shipment items if this is a new shipment
                if ($newShipment) {
                    foreach ($vendorItems as $orderItem) {
                        \App\Models\ShipmentItem::create([
                            'shipment_id' => $shipment->id,
                            'order_item_id' => $orderItem->id,
                            'quantity' => $orderItem->quantity
                        ]);
                    }
                }
                
                // Add tracking record
                $shipment->tracking()->create([
                    'status' => $request->status,
                    'location' => $request->status === 'shipped' ? 'Warehouse' : 'Destination',
                    'description' => $request->status === 'shipped' 
                        ? 'Package has been shipped by vendor' 
                        : 'Package has been delivered',
                    'timestamp' => now()
                ]);
                
                // Add delivery attempt record if delivered
                if ($request->status === 'delivered') {
                    $shipment->deliveryAttempts()->create([
                        'attempt_date' => now(),
                        'status' => 'delivered',
                        'notes' => 'Package successfully delivered',
                        'delivery_person' => 'Vendor delivery'
                    ]);
                }
            }
            
            // Record in order history
            $order->history()->create([
                'status' => 'vendor_fulfillment_updated',
                'comment' => 'Vendor updated fulfillment status to: ' . $request->status,
                'user_id' => $vendorId
            ]);
            
            DB::commit();
            
            // Return the updated fulfillment order with items
            $fulfillmentOrder->load('items');
            
            return response()->json([
                'success' => true,
                'message' => 'Fulfillment status updated successfully',
                'data' => [
                    'fulfillment_order' => $fulfillmentOrder,
                    'shipment' => $shipment,
                    'using_warehouse' => $useWarehouse
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Order not found or does not contain your products',
                'error' => 'Not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating fulfillment status', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $id,
                'vendor_id' => $request->user()->id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating fulfillment status',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
    
    /**
     * Get delivery and fulfillment status for a vendor's order items
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function getFulfillmentStatus(Request $request, $id)
    {
        try {
            $vendorId = $request->user()->id;
            
            // Find the order with vendor's items
            $order = Order::whereHas('items.product', function($query) use ($vendorId) {
                $query->where('vendor_id', $vendorId);
            })->findOrFail($id);
            
            // Get fulfillment orders for this vendor
            // Change 'vendor_id' to 'assigned_to' which is the correct column name
            $fulfillmentOrder = FulfillmentOrder::with(['items.orderItem.product'])
                ->where('order_id', $id)
                ->where('assigned_to', $vendorId)
                ->first();
                
            // Need to also update this query to use the proper relationship
            // Since there's no direct 'vendor_id' in shipments table either,
            // we need to check if the shipment was created by this vendor or contains items from this vendor
            $shipment = Shipment::with(['items.orderItem.product', 'tracking', 'deliveryAttempts'])
                ->where('order_id', $id)
                ->where(function($query) use ($vendorId) {
                    $query->where('created_by', $vendorId)
                          ->orWhereHas('items.orderItem.product', function($q) use ($vendorId) {
                              $q->where('vendor_id', $vendorId);
                          });
                })
                ->first();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'order_id' => $id,
                    'order_number' => $order->order_number,
                    'fulfillment_order' => $fulfillmentOrder,
                    'shipment' => $shipment
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found or does not contain your products',
                'error' => 'Not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error retrieving fulfillment status', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $id,
                'vendor_id' => $request->user()->id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving fulfillment status',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
    
    /**
     * Add a tracking update to a shipment
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function addTrackingUpdate(Request $request, $id)
    {
        try {
            $vendorId = $request->user()->id;
            
            // Validate the request
            $validator = Validator::make($request->all(), [
                'status' => 'required|string|max:100',
                'location' => 'required|string|max:255',
                'description' => 'required|string|max:500',
                'timestamp' => 'nullable|date',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }
            
            // Find the shipment for this vendor
            $shipment = Shipment::where('order_id', $id)
                ->where('vendor_id', $vendorId)
                ->firstOrFail();
            
            // Add tracking update
            $tracking = $shipment->tracking()->create([
                'status' => $request->status,
                'location' => $request->location,
                'description' => $request->description,
                'timestamp' => $request->timestamp ?? now()
            ]);
            
            // If status is "delivered", update shipment and fulfillment order
            if (strtolower($request->status) === 'delivered') {
                $shipment->status = 'delivered';
                $shipment->delivery_date = $request->timestamp ?? now();
                $shipment->save();
                
                // Update fulfillment order
                $fulfillmentOrder = FulfillmentOrder::where('order_id', $id)
                    ->where('vendor_id', $vendorId)
                    ->first();
                    
                if ($fulfillmentOrder) {
                    $fulfillmentOrder->status = 'delivered';
                    $fulfillmentOrder->save();
                    
                    // Update fulfillment items
                    FulfillmentItem::where('fulfillment_order_id', $fulfillmentOrder->id)
                        ->update(['status' => 'delivered']);
                }
                
                // Add delivery attempt
                $shipment->deliveryAttempts()->create([
                    'attempt_date' => $request->timestamp ?? now(),
                    'status' => 'delivered',
                    'notes' => $request->description,
                    'delivery_person' => 'Vendor delivery service'
                ]);
                
                // Add order history
                Order::find($id)->history()->create([
                    'status' => 'vendor_delivery_completed',
                    'comment' => 'Vendor marked delivery as complete: ' . $request->description,
                    'user_id' => $vendorId
                ]);
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Tracking update added successfully',
                'data' => [
                    'tracking' => $tracking,
                    'shipment' => $shipment->fresh(['tracking', 'deliveryAttempts'])
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Shipment not found for this order',
                'error' => 'Not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error adding tracking update', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $id,
                'vendor_id' => $request->user()->id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error adding tracking update',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
    
    /**
     * Record a delivery attempt
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function recordDeliveryAttempt(Request $request, $id)
    {
        try {
            $vendorId = $request->user()->id;
            
            // Validate the request
            $validator = Validator::make($request->all(), [
                'status' => 'required|in:attempted,delivered,failed,rescheduled',
                'attempt_date' => 'nullable|date',
                'reason' => 'nullable|string|max:255',
                'notes' => 'nullable|string|max:500',
                'delivery_person' => 'nullable|string|max:255',
                'signature_image' => 'nullable|string',
                'proof_of_delivery_image' => 'nullable|string',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }
            
            // Find the shipment for this vendor
            $shipment = Shipment::where('order_id', $id)
                ->where('vendor_id', $vendorId)
                ->firstOrFail();
            
            // Record delivery attempt
            $deliveryAttempt = $shipment->deliveryAttempts()->create([
                'attempt_date' => $request->attempt_date ?? now(),
                'status' => $request->status,
                'reason' => $request->reason,
                'notes' => $request->notes,
                'signature_image' => $request->signature_image,
                'proof_of_delivery_image' => $request->proof_of_delivery_image,
                'delivery_person' => $request->delivery_person ?? 'Vendor delivery service'
            ]);
            
            // If status is "delivered", update shipment and fulfillment order
            if ($request->status === 'delivered') {
                $shipment->status = 'delivered';
                $shipment->delivery_date = $request->attempt_date ?? now();
                $shipment->save();
                
                // Update fulfillment order
                $fulfillmentOrder = FulfillmentOrder::where('order_id', $id)
                    ->where('vendor_id', $vendorId)
                    ->first();
                    
                if ($fulfillmentOrder) {
                    $fulfillmentOrder->status = 'delivered';
                    $fulfillmentOrder->save();
                    
                    // Update fulfillment items
                    FulfillmentItem::where('fulfillment_order_id', $fulfillmentOrder->id)
                        ->update(['status' => 'delivered']);
                }
                
                // Add order history
                Order::find($id)->history()->create([
                    'status' => 'vendor_delivery_completed',
                    'comment' => 'Vendor marked delivery as complete' . ($request->notes ? ': ' . $request->notes : ''),
                    'user_id' => $vendorId
                ]);
            } else {
                // Add tracking update for attempted delivery
                $shipment->tracking()->create([
                    'status' => 'delivery_' . $request->status,
                    'location' => 'Destination',
                    'description' => ($request->status === 'attempted' ? 'Delivery attempted' : 'Delivery ' . $request->status) . 
                                    ($request->reason ? ' - ' . $request->reason : ''),
                    'timestamp' => $request->attempt_date ?? now()
                ]);
                
                // Add order history
                Order::find($id)->history()->create([
                    'status' => 'vendor_delivery_' . $request->status,
                    'comment' => 'Vendor ' . $request->status . ' delivery' . ($request->reason ? ': ' . $request->reason : ''),
                    'user_id' => $vendorId
                ]);
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Delivery attempt recorded successfully',
                'data' => [
                    'delivery_attempt' => $deliveryAttempt,
                    'shipment' => $shipment->fresh(['tracking', 'deliveryAttempts'])
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Shipment not found for this order',
                'error' => 'Not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error recording delivery attempt', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $id,
                'vendor_id' => $request->user()->id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error recording delivery attempt',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
}