<?php

namespace App\Http\Controllers;

use App\Models\Role;
use App\Models\User;
use App\Models\VendorProfile;
use App\Models\VendorAddress;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class VendorController extends Controller
{
    /**
     * Approve a vendor account.
     *
     * This endpoint allows superadmins and admins to approve vendor accounts,
     * changing their status from pending to approved.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function approve(Request $request, $id)
    {
        try {
            // Find the user
            $user = User::with('vendorProfile', 'role')->find($id);
            
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found',
                    'details' => "No user exists with ID: {$id}"
                ], 404);
            }
            
            // Check if user is a vendor
            if (!$user->hasRole('vendor')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role mismatch',
                    'details' => 'This user is not a vendor'
                ], 400);
            }
            
            // Check if vendor profile exists
            if (!$user->vendorProfile) {
                return response()->json([
                    'success' => false,
                    'message' => 'Vendor profile not found',
                    'details' => 'This vendor does not have a profile'
                ], 404);
            }
            
            // Update vendor approval status
            $user->vendorProfile->is_approved = true;
            $user->vendorProfile->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Vendor approved successfully',
                'data' => [
                    'vendor' => $user->fresh(['vendorProfile', 'role'])
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Vendor approval failed',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reject a vendor account.
     *
     * This endpoint allows superadmins and admins to reject vendor accounts,
     * with an optional reason for rejection.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function reject(Request $request, $id)
    {
        try {
            // Validate request
            $request->validate([
                'rejection_reason' => 'nullable|string|max:1000'
            ]);
            
            // Find the user
            $user = User::with('vendorProfile', 'role')->find($id);
            
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found',
                    'details' => "No user exists with ID: {$id}"
                ], 404);
            }
            
            // Check if user is a vendor
            if (!$user->hasRole('vendor')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role mismatch',
                    'details' => 'This user is not a vendor'
                ], 400);
            }
            
            // Check if vendor profile exists
            if (!$user->vendorProfile) {
                return response()->json([
                    'success' => false,
                    'message' => 'Vendor profile not found',
                    'details' => 'This vendor does not have a profile'
                ], 404);
            }
            
            // Update vendor approval status
            $user->vendorProfile->is_approved = false;
            
            // Add rejection reason if provided
            if ($request->has('rejection_reason')) {
                $user->vendorProfile->rejection_reason = $request->rejection_reason;
            }
            
            $user->vendorProfile->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Vendor rejected successfully',
                'data' => [
                    'vendor' => $user->fresh(['vendorProfile', 'role'])
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Vendor rejection failed',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * List all vendors with their approval status.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            // Get approval status filter
            $approvalStatus = $request->query('status');
            
            // Get vendors
            $vendorRole = Role::where('slug', 'vendor')->first();
            
            if (!$vendorRole) {
                return response()->json([
                    'success' => false,
                    'message' => 'Vendor role not found',
                    'details' => 'The vendor role does not exist in the system'
                ], 404);
            }
            
            $query = User::with(['role', 'vendorProfile'])
                ->where('role_id', $vendorRole->id);
            
            // Filter by approval status if specified
            if ($approvalStatus === 'approved') {
                $query->whereHas('vendorProfile', function ($q) {
                    $q->where('is_approved', true);
                });
            } elseif ($approvalStatus === 'pending') {
                $query->whereHas('vendorProfile', function ($q) {
                    $q->where('is_approved', false);
                });
            }
            
            $vendors = $query->paginate(15);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'vendors' => $vendors
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving vendors',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the vendor profile.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            // Find the user
            $user = User::with('vendorProfile')->find($id);
            
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found',
                    'details' => "No user exists with ID: {$id}"
                ], 404);
            }
            
            // Check if user is a vendor
            if (!$user->hasRole('vendor')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role mismatch',
                    'details' => 'This user is not a vendor'
                ], 400);
            }
            
            // Validate the request
            $request->validate([
                'company_name' => 'required|string|max:255',
                'company_logo' => 'nullable|image|max:2048',
                'company_website' => 'nullable|url|max:255',
                'business_type' => 'nullable|string|max:100',
                'business_registration_number' => 'nullable|string|max:100',
                'tax_identification_number' => 'nullable|string|max:100',
                'industry' => 'nullable|string|max:100',
                'annual_revenue' => 'nullable|numeric|min:0',
                'number_of_employees' => 'nullable|integer|min:1',
                'established_year' => 'nullable|integer|min:1900|max:' . date('Y'),
                'notes' => 'nullable|string',
            ]);
            
            // Handle logo upload if provided
            $logoPath = null;
            if ($request->hasFile('company_logo')) {
                $logoPath = $request->file('company_logo')->store('vendor-logos', 'public');
            }
            
            // Update the profile
            $user->vendorProfile->update(array_merge(
                $request->except('company_logo'),
                $logoPath ? ['company_logo' => Storage::url($logoPath)] : []
            ));
            
            return response()->json([
                'success' => true,
                'message' => 'Vendor profile updated successfully',
                'data' => [
                    'vendor' => $user->fresh(['vendorProfile'])
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating vendor profile',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Add or update the vendor's address.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateAddress(Request $request)
    {
        try {
            $user = $request->user();

            if (!$user->isVendor()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized',
                    'details' => 'Only vendors can access this endpoint'
                ], 403);
            }

            // Validate the request
            $validator = Validator::make($request->all(), [
                'address_line1' => 'required|string|max:255',
                'address_line2' => 'nullable|string|max:255',
                'city' => 'required|string|max:100',
                'state' => 'required|string|max:100',
                'postal_code' => 'required|string|max:20',
                'country' => 'required|string|max:100',
                'phone' => 'nullable|string|max:20'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Format the full address
            $address = $request->address_line1;
            if ($request->address_line2) {
                $address .= ', ' . $request->address_line2;
            }
            $address .= ', ' . $request->city;
            $address .= ', ' . $request->state;
            $address .= ' ' . $request->postal_code;
            $address .= ', ' . $request->country;

            // Update user's address and phone
            $user->update([
                'address' => $address,
                'phone' => $request->phone
            ]);

            $message = $user->wasRecentlyCreated ? 'Vendor address added successfully' : 'Vendor address updated successfully';

            return response()->json([
                'success' => true,
                'message' => $message,
                'data' => [
                    'vendor' => $user->fresh()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating vendor address',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * List all vendors for public access.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function publicIndex(Request $request)
    {
        try {
            // Get vendors
            $vendorRole = Role::where('slug', 'vendor')->first();
            
            if (!$vendorRole) {
                return response()->json([
                    'success' => false,
                    'message' => 'Vendor role not found',
                    'details' => 'The vendor role does not exist in the system'
                ], 404);
            }
            
            $query = User::with(['vendorProfile' => function($q) {
                    $q->select('id', 'user_id', 'company_name', 'company_logo', 'company_description', 'is_approved');
                }])
                ->where('role_id', $vendorRole->id)
                ->select('id', 'name', 'email', 'phone', 'address');
            
            // Add search functionality
            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhereHas('vendorProfile', function($q) use ($search) {
                          $q->where('company_name', 'like', "%{$search}%");
                      });
                });
            }
            
            $vendors = $query->paginate(15);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'vendors' => $vendors
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving vendors',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get a specific vendor's public profile.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function publicShow(Request $request, $id)
    {
        try {
            $vendor = User::with(['vendorProfile' => function($q) {
                    $q->select('id', 'user_id', 'company_name', 'company_logo', 'company_description', 'is_approved');
                }])
                ->select('id', 'name', 'email', 'phone', 'address')
                ->find($id);
            
            if (!$vendor) {
                return response()->json([
                    'success' => false,
                    'message' => 'Vendor not found',
                    'details' => 'The requested vendor does not exist'
                ], 404);
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'vendor' => $vendor
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving vendor',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all addresses for the authenticated vendor.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getAddresses(Request $request)
    {
        try {
            $vendor = $request->user();
            
            if (!$vendor->isVendor()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized',
                    'details' => 'Only vendors can access this endpoint'
                ], 403);
            }

            $addresses = VendorAddress::where('vendor_id', $vendor->id)
                ->orderBy('is_default', 'desc')
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'data' => [
                    'addresses' => $addresses
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving addresses',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Add a new address for the vendor.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addAddress(Request $request)
    {
        try {
            $vendor = $request->user();

            if (!$vendor->isVendor()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized',
                    'details' => 'Only vendors can access this endpoint'
                ], 403);
            }

            // Validate the request
            $validator = Validator::make($request->all(), [
                'address_line1' => 'required|string|max:255',
                'address_line2' => 'nullable|string|max:255',
                'city' => 'required|string|max:100',
                'state' => 'required|string|max:100',
                'postal_code' => 'required|string|max:20',
                'country' => 'required|string|max:100',
                'phone' => 'nullable|string|max:20',
                'is_default' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            DB::beginTransaction();

            // If setting as default, update other addresses
            if ($request->boolean('is_default')) {
                VendorAddress::where('vendor_id', $vendor->id)
                    ->update(['is_default' => false]);
            }

            // Create the address
            $address = VendorAddress::create(array_merge(
                $request->all(),
                ['vendor_id' => $vendor->id]
            ));

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Address added successfully',
                'data' => [
                    'address' => $address
                ]
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error adding address',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get a specific address.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function getAddress(Request $request, $id)
    {
        try {
            $vendor = $request->user();

            if (!$vendor->isVendor()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized',
                    'details' => 'Only vendors can access this endpoint'
                ], 403);
            }

            $address = VendorAddress::where('id', $id)
                ->where('vendor_id', $vendor->id)
                ->firstOrFail();

            return response()->json([
                'success' => true,
                'data' => [
                    'address' => $address
                ]
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Address not found',
                'details' => 'The requested address does not exist or you do not have access to it'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving address',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update an existing vendor address.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateVendorAddress(Request $request, $id)
    {
        try {
            $vendor = $request->user();

            if (!$vendor->isVendor()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized',
                    'details' => 'Only vendors can access this endpoint'
                ], 403);
            }

            // Find the address
            $address = VendorAddress::where('id', $id)
                ->where('vendor_id', $vendor->id)
                ->firstOrFail();

            // Validate the request
            $validator = Validator::make($request->all(), [
                'address_line1' => 'required|string|max:255',
                'address_line2' => 'nullable|string|max:255',
                'city' => 'required|string|max:100',
                'state' => 'required|string|max:100',
                'postal_code' => 'required|string|max:20',
                'country' => 'required|string|max:100',
                'phone' => 'nullable|string|max:20',
                'is_default' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            DB::beginTransaction();

            // If setting as default, update other addresses
            if ($request->boolean('is_default')) {
                VendorAddress::where('vendor_id', $vendor->id)
                    ->where('id', '!=', $id)
                    ->update(['is_default' => false]);
            }

            // Update the address
            $address->update($request->all());

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Address updated successfully',
                'data' => [
                    'address' => $address->fresh()
                ]
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Address not found',
                'details' => 'The requested address does not exist or you do not have access to it'
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error updating address',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete an address.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function deleteAddress(Request $request, $id)
    {
        try {
            $vendor = $request->user();

            if (!$vendor->isVendor()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized',
                    'details' => 'Only vendors can access this endpoint'
                ], 403);
            }

            $address = VendorAddress::where('id', $id)
                ->where('vendor_id', $vendor->id)
                ->firstOrFail();

            // Don't allow deletion of default address if it's the only address
            if ($address->is_default && VendorAddress::where('vendor_id', $vendor->id)->count() === 1) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete address',
                    'details' => 'Cannot delete the default address when it is the only address'
                ], 400);
            }

            $address->delete();

            return response()->json([
                'success' => true,
                'message' => 'Address deleted successfully'
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Address not found',
                'details' => 'The requested address does not exist or you do not have access to it'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting address',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Set an address as default.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function setDefaultAddress(Request $request, $id)
    {
        try {
            $vendor = $request->user();

            if (!$vendor->isVendor()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized',
                    'details' => 'Only vendors can access this endpoint'
                ], 403);
            }

            $address = VendorAddress::where('id', $id)
                ->where('vendor_id', $vendor->id)
                ->firstOrFail();

            DB::beginTransaction();

            // Update all addresses to not default
            VendorAddress::where('vendor_id', $vendor->id)
                ->update(['is_default' => false]);

            // Set this address as default
            $address->is_default = true;
            $address->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Default address updated successfully',
                'data' => [
                    'address' => $address->fresh()
                ]
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Address not found',
                'details' => 'The requested address does not exist or you do not have access to it'
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error updating default address',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}