<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class SystemController extends Controller
{
    /**
     * Clear all application cache
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function clearCache()
    {
        try {
            Artisan::call('cache:clear');
            Artisan::call('config:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');
            
            return response()->json([
                'success' => true,
                'message' => 'All cache cleared successfully',
                'data' => [
                    'cache' => 'cleared',
                    'config' => 'cleared',
                    'route' => 'cleared',
                    'view' => 'cleared'
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error clearing cache', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error clearing cache',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Clear specific cache
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function clearSpecificCache(Request $request)
    {
        try {
            $validated = $request->validate([
                'type' => 'required|in:cache,config,route,view,all'
            ]);

            switch ($validated['type']) {
                case 'cache':
                    Artisan::call('cache:clear');
                    break;
                case 'config':
                    Artisan::call('config:clear');
                    break;
                case 'route':
                    Artisan::call('route:clear');
                    break;
                case 'view':
                    Artisan::call('view:clear');
                    break;
                case 'all':
                    return $this->clearCache();
            }

            return response()->json([
                'success' => true,
                'message' => ucfirst($validated['type']) . ' cache cleared successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error clearing specific cache', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error clearing cache',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Clear temporary files
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function clearTempFiles()
    {
        try {
            $tempPath = storage_path('app/temp');
            if (file_exists($tempPath)) {
                $files = glob($tempPath . '/*');
                foreach ($files as $file) {
                    if (is_file($file)) {
                        unlink($file);
                    }
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Temporary files cleared successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error clearing temp files', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error clearing temporary files',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get system information
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getSystemInfo()
    {
        try {
            $info = [
                'php_version' => PHP_VERSION,
                'laravel_version' => app()->version(),
                'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
                'database_connection' => config('database.default'),
                'cache_driver' => config('cache.default'),
                'queue_connection' => config('queue.default'),
                'storage_disk' => config('filesystems.default'),
                'memory_usage' => memory_get_usage(true),
                'disk_free_space' => disk_free_space(storage_path()),
                'upload_max_filesize' => ini_get('upload_max_filesize'),
                'post_max_size' => ini_get('post_max_size'),
                'max_execution_time' => ini_get('max_execution_time'),
            ];

            return response()->json([
                'success' => true,
                'data' => $info
            ]);
        } catch (\Exception $e) {
            Log::error('Error getting system info', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error getting system information',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Optimize the application
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function optimize()
    {
        try {
            Artisan::call('optimize');
            Artisan::call('config:cache');
            Artisan::call('route:cache');
            Artisan::call('view:cache');

            return response()->json([
                'success' => true,
                'message' => 'Application optimized successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error optimizing application', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error optimizing application',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get database status
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDatabaseStatus()
    {
        try {
            $status = [
                'connection' => config('database.default'),
                'tables' => DB::select('SHOW TABLES'),
                'size' => $this->getDatabaseSize(),
                'last_backup' => $this->getLastBackupTime(),
            ];

            return response()->json([
                'success' => true,
                'data' => $status
            ]);
        } catch (\Exception $e) {
            Log::error('Error getting database status', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error getting database status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get database size
     *
     * @return string
     */
    private function getDatabaseSize()
    {
        $tables = DB::select('SHOW TABLES');
        $size = 0;
        foreach ($tables as $table) {
            $tableName = reset($table);
            $tableSize = DB::select("SELECT data_length + index_length as size FROM information_schema.TABLES WHERE table_schema = DATABASE() AND table_name = ?", [$tableName]);
            $size += $tableSize[0]->size;
        }
        return $this->formatBytes($size);
    }

    /**
     * Get last backup time
     *
     * @return string|null
     */
    private function getLastBackupTime()
    {
        $backupPath = storage_path('app/backups');
        if (!file_exists($backupPath)) {
            return null;
        }
        $files = glob($backupPath . '/*.sql');
        if (empty($files)) {
            return null;
        }
        return date('Y-m-d H:i:s', filemtime(end($files)));
    }

    /**
     * Format bytes to human readable format
     *
     * @param int $bytes
     * @return string
     */
    private function formatBytes($bytes)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);
        return round($bytes, 2) . ' ' . $units[$pow];
    }
} 