<?php

namespace App\Http\Controllers;

use App\Services\SearchService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class SearchController extends Controller
{
    protected $searchService;

    public function __construct(SearchService $searchService)
    {
        $this->searchService = $searchService;
    }

    /**
     * Perform a universal search
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function search(Request $request)
    {
        try {
            $query = $request->input('query');
            $filters = $request->only([
                'brand_name',
                'category_name',
                'min_price',
                'max_price',
                'in_stock',
                'per_page',
                'page'
            ]);

            // Log the search parameters
            Log::info('Search parameters', [
                'query' => $query,
                'filters' => $filters
            ]);

            if (empty($query)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Search query is required'
                ], 400);
            }

            $results = $this->searchService->search($query, $filters);

            return response()->json([
                'success' => true,
                'data' => $results
            ]);

        } catch (\Exception $e) {
            // Enhanced error logging
            Log::error('Search error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'line' => $e->getLine(),
                'file' => $e->getFile(),
                'query' => $query ?? null,
                'filters' => $filters ?? []
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while performing the search',
                'error' => $e->getMessage() // Include the actual error message in development
            ], 500);
        }
    }
} 