<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\ProductBulkPrice;
use App\Models\ProductPackPrice;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class ProductPricingController extends Controller
{
    /**
     * Get all pricing information for a product.
     *
     * @param int $productId
     * @return JsonResponse
     */
    public function getPricing(int $productId): JsonResponse
    {
        try {
            // Find the product with pricing
            $product = Product::with(['packPrice', 'bulkPrices', 'currency'])
                ->findOrFail($productId);
            
            // Check if the product belongs to the vendor
            if ($product->vendor_id !== request()->user()->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to view this product pricing',
                    'meta' => [
                        'code' => 403,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 403);
            }
            
            // Get default margins for displaying in UI
            $defaultMargins = [
                'customer_margin' => $product->getDefaultCustomerMargin(),
                'partner_margin' => $product->getDefaultPartnerMargin()
            ];
            
            return response()->json([
                'success' => true,
                'data' => [
                    'product_id' => $product->id,
                    'product_name' => $product->name,
                    'currency' => $product->currency,
                    'pack_price' => $product->packPrice,
                    'bulk_prices' => $product->bulkPrices,
                    'default_margins' => $defaultMargins
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 200);
            
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error retrieving product pricing', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving product pricing',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }

    /**
 * Update pack price for a product.
 *
 * @param  \Illuminate\Http\Request  $request
 * @param  int  $productId
 * @return \Illuminate\Http\JsonResponse
 */
public function updatePackPrice(Request $request, int $productId): JsonResponse
{
    try {
        Log::info('Request received for updating pack price', [
            'product_id' => $productId,
            'data' => $request->all()
        ]);
        
        // Find the product
        $product = Product::findOrFail($productId);
        
        // Check if the product belongs to the vendor
        if ($product->vendor_id !== $request->user()->id) {
            return response()->json([
                'success' => false,
                'message' => 'You do not have permission to update this product pricing',
                'meta' => [
                    'code' => 403,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 403);
        }
        
        // Validate request
        try {
            $validated = $request->validate([
                'number_of_products' => 'sometimes|required|integer|min:1',
                'per_pack_price' => 'required|numeric|min:0',
                'per_pack_special_price' => 'nullable|numeric|min:0',
                'customer_margin' => 'nullable|numeric|min:0|max:100',
                'partner_margin' => 'nullable|numeric|min:0|max:100',
                'customer_margin_type' => 'nullable|in:percentage,fixed',
                'partner_margin_type' => 'nullable|in:percentage,fixed',
                'delivery_fee' => 'nullable|numeric|min:0',
            ]);
            
            Log::info('Validation passed for pack price update');
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Validation failed for pack price update', [
                'errors' => $e->errors()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(),
                'meta' => [
                    'code' => 422,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 422);
        }
        
        // Set default margin types if not provided
        if (!isset($validated['customer_margin_type']) && isset($validated['customer_margin'])) {
            $validated['customer_margin_type'] = 'percentage';
        }
        
        if (!isset($validated['partner_margin_type']) && isset($validated['partner_margin'])) {
            $validated['partner_margin_type'] = 'percentage';
        }
        
        // Find or create pack price
        $packPrice = $product->packPrice;
        if ($packPrice) {
            // Make sure to include delivery_fee in the update
            $packPrice->update([
                'number_of_products' => $validated['number_of_products'] ?? $packPrice->number_of_products,
                'per_pack_price' => $validated['per_pack_price'],
                'per_pack_special_price' => $validated['per_pack_special_price'] ?? null,
                'customer_margin' => $validated['customer_margin'] ?? $packPrice->customer_margin,
                'partner_margin' => $validated['partner_margin'] ?? $packPrice->partner_margin,
                'customer_margin_type' => $validated['customer_margin_type'] ?? $packPrice->customer_margin_type,
                'partner_margin_type' => $validated['partner_margin_type'] ?? $packPrice->partner_margin_type,
                'delivery_fee' => $validated['delivery_fee'] ?? $packPrice->delivery_fee,
            ]);
            $message = 'Pack price updated successfully';
        } else {
            // Include delivery_fee when creating a new pack price
            $packPrice = ProductPackPrice::create([
                'product_id' => $productId,
                'number_of_products' => $validated['number_of_products'] ?? 1,
                'per_pack_price' => $validated['per_pack_price'],
                'per_pack_special_price' => $validated['per_pack_special_price'] ?? null,
                'customer_margin' => $validated['customer_margin'] ?? null,
                'partner_margin' => $validated['partner_margin'] ?? null,
                'customer_margin_type' => $validated['customer_margin_type'] ?? 'percentage',
                'partner_margin_type' => $validated['partner_margin_type'] ?? 'percentage',
                'delivery_fee' => $validated['delivery_fee'] ?? null,
            ]);
            $message = 'Pack price created successfully';
        }
        
        Log::info($message, [
            'pack_price_id' => $packPrice->id
        ]);
        
        // Return response
        return response()->json([
            'success' => true,
            'message' => $message,
            'data' => [
                'pack_price' => $packPrice->fresh()
            ],
            'meta' => [
                'code' => 200,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 200);
    } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
        Log::error('Product not found', [
            'product_id' => $productId
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Product not found',
            'meta' => [
                'code' => 404,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 404);
    } catch (\Exception $e) {
        Log::error('Error updating pack price', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Error updating pack price',
            'error' => $e->getMessage(),
            'meta' => [
                'code' => 500,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 500);
    }
}

/**
 * Create a new bulk price for a product.
 *
 * @param  \Illuminate\Http\Request  $request
 * @param  int  $productId
 * @return \Illuminate\Http\JsonResponse
 */
public function createBulkPrice(Request $request, int $productId): JsonResponse
{
    try {
        Log::info('Request received for creating bulk price', [
            'product_id' => $productId,
            'data' => $request->all()
        ]);
        
        // Find the product
        $product = Product::findOrFail($productId);
        
        // Check if the product belongs to the vendor
        if ($product->vendor_id !== $request->user()->id) {
            return response()->json([
                'success' => false,
                'message' => 'You do not have permission to add bulk prices to this product',
                'meta' => [
                    'code' => 403,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 403);
        }
        
        // Validate request
        try {
            $validated = $request->validate([
                'number_of_packs' => 'required|integer|min:1',
                'per_pack_price' => 'required|numeric|min:0',
                'per_pack_special_price' => 'nullable|numeric|min:0',
                'customer_margin' => 'nullable|numeric|min:0|max:100',
                'partner_margin' => 'nullable|numeric|min:0|max:100',
                'customer_margin_type' => 'nullable|in:percentage,fixed',
                'partner_margin_type' => 'nullable|in:percentage,fixed',
                'delivery_fee' => 'nullable|numeric|min:0',
            ]);
            
            Log::info('Validation passed for bulk price creation');
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Validation failed for bulk price creation', [
                'errors' => $e->errors()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(),
                'meta' => [
                    'code' => 422,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 422);
        }
        
        // Check if a bulk price with this number_of_packs already exists
        $existingBulkPrice = ProductBulkPrice::where('product_id', $productId)
            ->where('number_of_packs', $validated['number_of_packs'])
            ->first();
            
        if ($existingBulkPrice) {
            return response()->json([
                'success' => false,
                'message' => 'A bulk price for this number of packs already exists',
                'meta' => [
                    'code' => 409,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 409);
        }
        
        // Set default margin types if not provided
        if (!isset($validated['customer_margin_type']) && isset($validated['customer_margin'])) {
            $validated['customer_margin_type'] = 'percentage';
        }
        
        if (!isset($validated['partner_margin_type']) && isset($validated['partner_margin'])) {
            $validated['partner_margin_type'] = 'percentage';
        }
        
        // Get the next sort order
        $maxSortOrder = ProductBulkPrice::where('product_id', $productId)
            ->max('sort_order');
        
        // Create the bulk price with delivery_fee
        $bulkPrice = ProductBulkPrice::create([
            'product_id' => $productId,
            'number_of_packs' => $validated['number_of_packs'],
            'per_pack_price' => $validated['per_pack_price'],
            'per_pack_special_price' => $validated['per_pack_special_price'] ?? null,
            'customer_margin' => $validated['customer_margin'] ?? null,
            'partner_margin' => $validated['partner_margin'] ?? null,
            'customer_margin_type' => $validated['customer_margin_type'] ?? 'percentage',
            'partner_margin_type' => $validated['partner_margin_type'] ?? 'percentage',
            'sort_order' => $maxSortOrder !== null ? $maxSortOrder + 1 : 0,
            'delivery_fee' => $validated['delivery_fee'] ?? null,
        ]);
        
        Log::info('Bulk price created successfully', [
            'bulk_price_id' => $bulkPrice->id
        ]);
        
        // Return response
        return response()->json([
            'success' => true,
            'message' => 'Bulk price created successfully',
            'data' => [
                'bulk_price' => $bulkPrice
            ],
            'meta' => [
                'code' => 201,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 201);
    } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
        Log::error('Product not found', [
            'product_id' => $productId
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Product not found',
            'meta' => [
                'code' => 404,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 404);
    } catch (\Exception $e) {
        Log::error('Error creating bulk price', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Error creating bulk price',
            'error' => $e->getMessage(),
            'meta' => [
                'code' => 500,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 500);
    }
}
    
    /**
     * Update an existing bulk price.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function updateBulkPrice(Request $request, int $id): JsonResponse
    {
        try {
            Log::info('Request received for updating bulk price', [
                'bulk_price_id' => $id,
                'data' => $request->all()
            ]);
            
            // Find the bulk price
            $bulkPrice = ProductBulkPrice::findOrFail($id);
            
            // Check if the bulk price belongs to the vendor
            $product = $bulkPrice->product;
            if ($product->vendor_id !== $request->user()->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to update this bulk price',
                    'meta' => [
                        'code' => 403,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 403);
            }
            
            // Validate request
            try {
                $validated = $request->validate([
                    'number_of_packs' => 'sometimes|required|integer|min:1',
                    'per_pack_price' => 'required|numeric|min:0',
                    'per_pack_special_price' => 'nullable|numeric|min:0',
                    'customer_margin' => 'nullable|numeric|min:0|max:100',
                    'partner_margin' => 'nullable|numeric|min:0|max:100',
                    'customer_margin_type' => 'nullable|in:percentage,fixed',
                    'partner_margin_type' => 'nullable|in:percentage,fixed',
                ]);
                
                Log::info('Validation passed for bulk price update');
                
            } catch (\Illuminate\Validation\ValidationException $e) {
                Log::error('Validation failed for bulk price update', [
                    'errors' => $e->errors()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $e->errors(),
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }
            
            // If number_of_packs is changed, check for duplicates
            if (isset($validated['number_of_packs']) && $validated['number_of_packs'] !== $bulkPrice->number_of_packs) {
                $existingBulkPrice = ProductBulkPrice::where('product_id', $bulkPrice->product_id)
                    ->where('number_of_packs', $validated['number_of_packs'])
                    ->where('id', '!=', $id)
                    ->first();
                    
                if ($existingBulkPrice) {
                    return response()->json([
                        'success' => false,
                        'message' => 'A bulk price for this number of packs already exists',
                        'meta' => [
                            'code' => 409,
                            'timestamp' => now()->toIso8601String(),
                        ],
                    ], 409);
                }
            }
            
            // Set default margin types if not provided but margin values are
            if (!isset($validated['customer_margin_type']) && isset($validated['customer_margin'])) {
                $validated['customer_margin_type'] = $bulkPrice->customer_margin_type ?? 'percentage';
            }
            
            if (!isset($validated['partner_margin_type']) && isset($validated['partner_margin'])) {
                $validated['partner_margin_type'] = $bulkPrice->partner_margin_type ?? 'percentage';
            }
            
            // Update the bulk price
            $bulkPrice->update($validated);
            
            Log::info('Bulk price updated successfully', [
                'bulk_price_id' => $bulkPrice->id
            ]);
            
            // Return response
            return response()->json([
                'success' => true,
                'message' => 'Bulk price updated successfully',
                'data' => [
                    'bulk_price' => $bulkPrice->fresh()
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            Log::error('Bulk price not found', [
                'bulk_price_id' => $id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Bulk price not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error updating bulk price', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating bulk price',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
    
    /**
     * Delete a bulk price.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function deleteBulkPrice(int $id): JsonResponse
    {
        try {
            Log::info('Request received for deleting bulk price', [
                'bulk_price_id' => $id
            ]);
            
            // Find the bulk price
            $bulkPrice = ProductBulkPrice::findOrFail($id);
            
            // Check if the bulk price belongs to the vendor
            $product = $bulkPrice->product;
            if ($product->vendor_id !== request()->user()->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to delete this bulk price',
                    'meta' => [
                        'code' => 403,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 403);
            }
            
            // Delete the bulk price
            $bulkPrice->delete();
            
            // Reorder the remaining bulk prices
            $this->reorderBulkPrices($product->id);
            
            Log::info('Bulk price deleted successfully', [
                'bulk_price_id' => $id
            ]);
            
            // Return response
            return response()->json([
                'success' => true,
                'message' => 'Bulk price deleted successfully',
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            Log::error('Bulk price not found', [
                'bulk_price_id' => $id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Bulk price not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error deleting bulk price', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error deleting bulk price',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
    
    /**
     * Update the sort order of bulk prices.
     *
     * @param Request $request
     * @param int $productId
     * @return JsonResponse
     */
    public function updateBulkPriceOrder(Request $request, int $productId): JsonResponse
    {
        try {
            Log::info('Request received for updating bulk price order', [
                'product_id' => $productId,
                'data' => $request->all()
            ]);
            
            // Find the product
            $product = Product::findOrFail($productId);
            
            // Check if the product belongs to the vendor
            if ($product->vendor_id !== $request->user()->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to update this product pricing',
                    'meta' => [
                        'code' => 403,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 403);
            }
            
            // Validate request
            try {
                $validated = $request->validate([
                    'bulk_price_ids' => 'required|array',
                    'bulk_price_ids.*' => 'exists:product_bulk_prices,id',
                ]);
                
                Log::info('Validation passed for bulk price order update');
                
            } catch (\Illuminate\Validation\ValidationException $e) {
                Log::error('Validation failed for bulk price order update', [
                    'errors' => $e->errors()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $e->errors(),
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }
            
            // Verify all bulk prices belong to this product
            $bulkPriceIds = $validated['bulk_price_ids'];
            $bulkPrices = ProductBulkPrice::whereIn('id', $bulkPriceIds)
                ->where('product_id', $productId)
                ->get();
            
            if (count($bulkPrices) !== count($bulkPriceIds)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Some bulk prices do not belong to this product',
                    'meta' => [
                        'code' => 400,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 400);
            }
            
            // Update sort order
            DB::beginTransaction();
            
            try {
                foreach ($bulkPriceIds as $index => $bulkPriceId) {
                    ProductBulkPrice::where('id', $bulkPriceId)
                        ->update(['sort_order' => $index]);
                }
                
                DB::commit();
                
                Log::info('Bulk price order updated successfully');
                
                // Return response with updated bulk prices
                $updatedBulkPrices = ProductBulkPrice::where('product_id', $productId)
                    ->orderBy('sort_order')
                    ->get();
                
                return response()->json([
                    'success' => true,
                    'message' => 'Bulk price order updated successfully',
                    'data' => [
                        'bulk_prices' => $updatedBulkPrices
                    ],
                    'meta' => [
                        'code' => 200,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 200);
            } catch (\Exception $e) {
                DB::rollBack();
                throw $e;
            }
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            Log::error('Product not found', [
                'product_id' => $productId
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Product not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error updating bulk price order', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating bulk price order',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
    
    /**
     * Update multiple bulk prices for a product at once.
     *
     * @param Request $request
     * @param int $productId
     * @return JsonResponse
     */
    public function updateAllBulkPrices(Request $request, int $productId): JsonResponse
    {
        try {
            Log::info('Request received for updating all bulk prices', [
                'product_id' => $productId,
                'data' => $request->all()
            ]);
            
            // Find the product
            $product = Product::findOrFail($productId);
            
            // Check if the product belongs to the vendor
            if ($product->vendor_id !== $request->user()->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to update this product pricing',
                    'meta' => [
                        'code' => 403,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 403);
            }
            
            // Validate request
            try {
                $validated = $request->validate([
                    'bulk_prices' => 'required|array',
                    'bulk_prices.*.id' => 'sometimes|exists:product_bulk_prices,id',
                    'bulk_prices.*.number_of_packs' => 'required|integer|min:1',
                    'bulk_prices.*.per_pack_price' => 'required|numeric|min:0',
                    'bulk_prices.*.per_pack_special_price' => 'nullable|numeric|min:0',
                    'bulk_prices.*.customer_margin' => 'nullable|numeric|min:0|max:100',
                    'bulk_prices.*.partner_margin' => 'nullable|numeric|min:0|max:100',
                    'bulk_prices.*.customer_margin_type' => 'nullable|in:percentage,fixed',
                    'bulk_prices.*.partner_margin_type' => 'nullable|in:percentage,fixed',
                ]);
                
                Log::info('Validation passed for bulk price updates');
                
            } catch (\Illuminate\Validation\ValidationException $e) {
                Log::error('Validation failed for bulk price updates', [
                    'errors' => $e->errors()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $e->errors(),
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }
            
            // Process bulk prices
            DB::beginTransaction();
            
            try {
                // Delete all existing bulk prices
                ProductBulkPrice::where('product_id', $productId)->delete();
                
                // Create new bulk prices
                $bulkPrices = [];
                foreach ($validated['bulk_prices'] as $index => $bulkPriceData) {
                    // Set default margin types if not provided
                    if (!isset($bulkPriceData['customer_margin_type']) && isset($bulkPriceData['customer_margin'])) {
                        $bulkPriceData['customer_margin_type'] = 'percentage';
                    }
                    
                    if (!isset($bulkPriceData['partner_margin_type']) && isset($bulkPriceData['partner_margin'])) {
                        $bulkPriceData['partner_margin_type'] = 'percentage';
                    }
                    
                    $bulkPriceData['product_id'] = $productId;
                    $bulkPriceData['sort_order'] = $index;
                    
                    $bulkPrice = ProductBulkPrice::create($bulkPriceData);
                    $bulkPrices[] = $bulkPrice;
                }
                
                DB::commit();
                
                Log::info('All bulk prices updated successfully', [
                    'count' => count($bulkPrices)
                ]);
                
                return response()->json([
                    'success' => true,
                    'message' => 'All bulk prices updated successfully',
                    'data' => [
                        'bulk_prices' => $bulkPrices
                    ],
                    'meta' => [
                        'code' => 200,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 200);
            } catch (\Exception $e) {
                DB::rollBack();
                throw $e;
            }
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            Log::error('Product not found', [
                'product_id' => $productId
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Product not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error updating all bulk prices', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating all bulk prices',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
    
    /**
     * Update product default margins.
     *
     * @param Request $request
     * @param int $productId
     * @return JsonResponse
     */
    public function updateDefaultMargins(Request $request, int $productId): JsonResponse
    {
        try {
            Log::info('Request received for updating product default margins', [
                'product_id' => $productId,
                'data' => $request->all()
            ]);
            
            // Find the product
            $product = Product::findOrFail($productId);
            
            // Check if the product belongs to the vendor
            if ($product->vendor_id !== $request->user()->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to update this product margins',
                    'meta' => [
                        'code' => 403,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 403);
            }
            
            // Validate request
            try {
                $validated = $request->validate([
                    'default_customer_margin' => 'required|numeric|min:0|max:100',
                    'default_partner_margin' => 'required|numeric|min:0|max:100',
                    'default_customer_margin_type' => 'required|in:percentage,fixed',
                    'default_partner_margin_type' => 'required|in:percentage,fixed',
                ]);
                
                Log::info('Validation passed for default margins update');
                
            } catch (\Illuminate\Validation\ValidationException $e) {
                Log::error('Validation failed for default margins update', [
                    'errors' => $e->errors()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $e->errors(),
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }
            
            // Update the product's default margins
            $product->update($validated);
            
            Log::info('Product default margins updated successfully', [
                'product_id' => $product->id
            ]);
            
            // Return response
            return response()->json([
                'success' => true,
                'message' => 'Default margins updated successfully',
                'data' => [
                    'default_margins' => [
                        'customer_margin' => $product->getDefaultCustomerMargin(),
                        'partner_margin' => $product->getDefaultPartnerMargin()
                    ]
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            Log::error('Product not found', [
                'product_id' => $productId
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Product not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error updating default margins', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating default margins',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
    
    /**
     * Helper method to reorder bulk prices after deletion.
     *
     * @param int $productId
     * @return void
     */
    private function reorderBulkPrices(int $productId): void
    {
        $bulkPrices = ProductBulkPrice::where('product_id', $productId)
            ->orderBy('number_of_packs')
            ->get();
            
        foreach ($bulkPrices as $index => $bulkPrice) {
            $bulkPrice->update(['sort_order' => $index]);
        }
    }
}