<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\ProductImage;
use App\Models\ProductPackPrice;
use App\Models\ProductBulkPrice;
use App\Models\Category;
use App\Http\Controllers\AdminUserController;
use App\Models\Brand; 
use App\Models\Currency;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Builder;
use App\Traits\Cacheable;
use Illuminate\Support\Facades\Cache;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class ProductController extends Controller
{
    use Cacheable;

    /**
     * Import multiple products in bulk.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function bulkImport(Request $request): JsonResponse
    {
        try {
            Log::info('Request received for bulk product import', [
                'vendor_id' => $request->user()->id
            ]);

            // Check that the user is a vendor
            if (!$request->user()->hasRole('vendor')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Only vendors can import products',
                    'meta' => [
                        'code' => 403,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 403);
            }

            // Validate request structure
            $validator = Validator::make($request->all(), [
                'products' => 'required|array|min:1|max:500',
                'products.*' => 'required|array',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }

            // Start transaction
            DB::beginTransaction();

            $results = [
                'success_count' => 0,
                'failed_count' => 0,
                'failed_items' => [],
                'imported_products' => []
            ];

            // Process each product in the array
            foreach ($request->products as $index => $productData) {
                try {
                    // Validate each product - using same validation rules as single product creation
                    $productValidator = Validator::make($productData, [
                        'name' => 'required|string|max:255',
                        'sku' => 'required|string|max:100|unique:products,sku',
                        'brand_id' => 'required|exists:brands,id',
                        'category_id' => 'required|exists:categories,id',
                        'description' => 'required|string',
                        'currency_code' => 'nullable|exists:currencies,code',
                        
                        // Pack pricing
                        'number_of_products' => 'required|integer|min:1',
                        'per_pack_price' => 'required|numeric|min:0',
                        'per_pack_special_price' => 'nullable|numeric|min:0',
                        'customer_margin' => 'nullable|numeric|min:0|max:100',
                        'partner_margin' => 'nullable|numeric|min:0|max:100',
                        'customer_margin_type' => 'nullable|in:percentage,fixed',
                        'partner_margin_type' => 'nullable|in:percentage,fixed',
                        'delivery_fee' => 'nullable|numeric|min:0',
                        
                        // Order thresholds and pallet info
                        'min_order_quantity' => 'nullable|integer|min:1',
                        'min_order_value' => 'nullable|numeric|min:0',
                        'free_shipping_threshold' => 'nullable|integer|min:1',
                        'products_per_pallet' => 'nullable|integer|min:1',
                        'pallet_delivery_fee' => 'nullable|numeric|min:0',
                        
                        // Bulk prices
                        'add_bulk_prices' => 'nullable|boolean',
                        'bulk_prices' => 'required_if:add_bulk_prices,1|nullable|array',
                        'bulk_prices.*.number_of_packs' => 'required_with:bulk_prices|integer|min:1',
                        'bulk_prices.*.per_pack_price' => 'required_with:bulk_prices|numeric|min:0',
                        'bulk_prices.*.per_pack_special_price' => 'nullable|numeric|min:0',
                        'bulk_prices.*.customer_margin' => 'nullable|numeric|min:0|max:100',
                        'bulk_prices.*.partner_margin' => 'nullable|numeric|min:0|max:100',
                        'bulk_prices.*.customer_margin_type' => 'nullable|in:percentage,fixed',
                        'bulk_prices.*.partner_margin_type' => 'nullable|in:percentage,fixed',
                        'bulk_prices.*.delivery_fee' => 'nullable|numeric|min:0',
                        
                        // Default margins
                        'default_customer_margin' => 'nullable|numeric|min:0|max:100',
                        'default_partner_margin' => 'nullable|numeric|min:0|max:100',
                        'default_customer_margin_type' => 'nullable|in:percentage,fixed',
                        'default_partner_margin_type' => 'nullable|in:percentage,fixed',
                        
                        // Stock information
                        'quantity' => 'nullable|integer|min:0',
                        
                        // SEO information
                        'meta_title' => 'nullable|string|max:70',
                        'meta_description' => 'nullable|string|max:160',
                        'meta_keywords' => 'nullable|string|max:255',
                        
                        // Images
                        'images' => 'nullable|array|max:10',
                        'images.*' => 'nullable|url|max:255',
                    ]);

                    if ($productValidator->fails()) {
                        throw new \Illuminate\Validation\ValidationException($productValidator);
                    }

                    // Generate slug from name
                    $slug = Str::slug($productData['name']);
                    $slugCount = Product::where('slug', 'LIKE', $slug . '%')->count();
                    if ($slugCount > 0) {
                        $slug = $slug . '-' . ($slugCount + 1);
                    }

                    // Get default currency code if not provided
                    $currencyCode = $productData['currency_code'] ?? Currency::getDefaultCode();

                    // Create the product
                    $product = Product::create([
                        'name' => $productData['name'],
                        'title' => $productData['name'],
                        'slug' => $slug,
                        'sku' => $productData['sku'],
                        'brand_id' => $productData['brand_id'],
                        'vendor_id' => $request->user()->id,
                        'created_by' => $request->user()->id,
                        'description' => $productData['description'],
                        'quantity' => $productData['quantity'] ?? 0,
                        'stock_status' => ($productData['quantity'] ?? 0) > 0 ? 
                            (($productData['quantity'] ?? 0) <= 5 ? 'low_stock' : 'in_stock') : 'out_of_stock',
                        'meta_title' => $productData['meta_title'] ?? $productData['name'],
                        'meta_description' => $productData['meta_description'] ?? null,
                        'meta_keywords' => isset($productData['meta_keywords']) ? 
                            json_encode(explode(',', $productData['meta_keywords'])) : null,
                        'is_approved' => false,
                        'approval_status' => 'pending',
                        'is_active' => true,
                        'currency_code' => $currencyCode,
                        'default_customer_margin' => $productData['default_customer_margin'] ?? null,
                        'default_partner_margin' => $productData['default_partner_margin'] ?? null,
                        'default_customer_margin_type' => $productData['default_customer_margin_type'] ?? null,
                        'default_partner_margin_type' => $productData['default_partner_margin_type'] ?? null,
                        
                        // Order thresholds and pallet info
                        'min_order_quantity' => $productData['min_order_quantity'] ?? null,
                        'min_order_value' => $productData['min_order_value'] ?? null,
                        'free_shipping_threshold' => $productData['free_shipping_threshold'] ?? null,
                        'products_per_pallet' => $productData['products_per_pallet'] ?? null,
                        'pallet_delivery_fee' => $productData['pallet_delivery_fee'] ?? null,
                    ]);

                    // Create pack price
                    ProductPackPrice::create([
                        'product_id' => $product->id,
                        'number_of_products' => $productData['number_of_products'] ?? 1,
                        'per_pack_price' => $productData['per_pack_price'],
                        'per_pack_special_price' => $productData['per_pack_special_price'] ?? null,
                        'customer_margin' => $productData['customer_margin'] ?? null,
                        'partner_margin' => $productData['partner_margin'] ?? null,
                        'customer_margin_type' => $productData['customer_margin_type'] ?? 'percentage',
                        'partner_margin_type' => $productData['partner_margin_type'] ?? 'percentage',
                        'delivery_fee' => $productData['delivery_fee'] ?? null,
                    ]);

                    // Create bulk prices if provided
                    if (isset($productData['add_bulk_prices']) && $productData['add_bulk_prices'] && 
                        isset($productData['bulk_prices']) && is_array($productData['bulk_prices'])) {
                        
                        foreach ($productData['bulk_prices'] as $index => $bulkPrice) {
                            ProductBulkPrice::create([
                                'product_id' => $product->id,
                                'number_of_packs' => $bulkPrice['number_of_packs'],
                                'per_pack_price' => $bulkPrice['per_pack_price'],
                                'per_pack_special_price' => $bulkPrice['per_pack_special_price'] ?? null,
                                'customer_margin' => $bulkPrice['customer_margin'] ?? null,
                                'partner_margin' => $bulkPrice['partner_margin'] ?? null,
                                'customer_margin_type' => $bulkPrice['customer_margin_type'] ?? 'percentage',
                                'partner_margin_type' => $bulkPrice['partner_margin_type'] ?? 'percentage',
                                'delivery_fee' => $bulkPrice['delivery_fee'] ?? null,
                                'sort_order' => $index,
                            ]);
                        }
                    }

                    // Create category association
                    DB::table('product_categories')->insert([
                        'product_id' => $product->id,
                        'category_id' => $productData['category_id'],
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);

                    // Create product images if provided
                    if (isset($productData['images']) && is_array($productData['images'])) {
                        foreach ($productData['images'] as $index => $imageUrl) {
                            if ($imageUrl) {
                                ProductImage::create([
                                    'product_id' => $product->id,
                                    'image_url' => $imageUrl,
                                    'alt_text' => $productData['name'],
                                    'sort_order' => $index,
                                ]);
                            }
                        }
                    }

                    // Increment success count and add to imported products list
                    $results['success_count']++;
                    $results['imported_products'][] = [
                        'id' => $product->id,
                        'name' => $product->name,
                        'sku' => $product->sku
                    ];

                } catch (\Illuminate\Validation\ValidationException $e) {
                    // Validation error for this product
                    $results['failed_count']++;
                    $results['failed_items'][] = [
                        'index' => $index,
                        'name' => $productData['name'] ?? 'Unknown',
                        'sku' => $productData['sku'] ?? 'Unknown',
                        'reason' => 'Validation failed',
                        'errors' => $e->errors()
                    ];
                    
                    // Continue to next product
                    continue;
                    
                } catch (\Exception $e) {
                    // Other error for this product
                    $results['failed_count']++;
                    $results['failed_items'][] = [
                        'index' => $index,
                        'name' => $productData['name'] ?? 'Unknown',
                        'sku' => $productData['sku'] ?? 'Unknown',
                        'reason' => 'Processing error',
                        'error' => $e->getMessage()
                    ];
                    
                    // Continue to next product
                    continue;
                }
            }

            // If some failed but some succeeded, still commit the transaction
            if ($results['success_count'] > 0) {
                DB::commit();
                
                $statusCode = $results['failed_count'] > 0 ? 207 : 201; // 207 Multi-Status
                $message = $results['failed_count'] > 0 ? 
                    'Bulk import completed with some failures' : 
                    'Bulk import completed successfully';
                    
                return response()->json([
                    'success' => true,
                    'message' => $message,
                    'data' => $results,
                    'meta' => [
                        'code' => $statusCode,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], $statusCode);
            } else {
                // If all failed, roll back
                DB::rollBack();
                
                return response()->json([
                    'success' => false,
                    'message' => 'Bulk import failed - no products were imported',
                    'data' => $results,
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }
        } catch (\Exception $e) {
            // Catch any other exceptions
            DB::rollBack();
            
            Log::error('Error in bulk product import', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error processing bulk import',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
   
    /**
     * Store a newly created product in storage.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function store(Request $request): JsonResponse
    {
        Log::info('Request received for creating product', $request->all());

        // Validation with try/catch
        try {
            Log::info('Attempting validation for product creation', ['request_data' => $request->all()]);
            
            $validated = $request->validate([
                // Basic product information
                'name' => 'required|string|max:255|unique:products,name',
                'sku' => 'required|string|max:100|unique:products,sku',
                'brand_id' => 'required|exists:brands,id',
                'category_id' => 'required|exists:categories,id',
                'description' => 'required|string',
                'currency_code' => 'nullable|exists:currencies,code',
                
                // Optional order thresholds
                'min_order_quantity' => 'nullable|integer|min:1',
                'min_order_value' => 'nullable|numeric|min:0',
                'free_shipping_threshold' => 'nullable|integer|min:1',
                
                // Optional pallet information
                'products_per_pallet' => 'nullable|integer|min:1',
                'pallet_delivery_fee' => 'nullable|numeric|min:0',
                
                // Pack pricing
                'number_of_products' => 'required|integer|min:1',
                'per_pack_price' => 'required|numeric|min:0',
                'per_pack_special_price' => 'nullable|numeric|min:0',
                'customer_margin' => 'nullable|numeric|min:0|max:100',
                'partner_margin' => 'nullable|numeric|min:0|max:100',
                'customer_margin_type' => 'nullable|in:percentage,fixed',
                'partner_margin_type' => 'nullable|in:percentage,fixed',
                'pack_delivery_fee' => 'nullable|numeric|min:0',
                
                // Bulk pricing
                'add_bulk_prices' => 'nullable|boolean',
                'bulk_prices' => 'required_if:add_bulk_prices,1|nullable|array',
                'bulk_prices.*.number_of_packs' => 'required_with:bulk_prices|integer|min:1',
                'bulk_prices.*.per_pack_price' => 'required_with:bulk_prices|numeric|min:0',
                'bulk_prices.*.per_pack_special_price' => 'nullable|numeric|min:0',
                'bulk_prices.*.customer_margin' => 'nullable|numeric|min:0|max:100',
                'bulk_prices.*.partner_margin' => 'nullable|numeric|min:0|max:100',
                'bulk_prices.*.customer_margin_type' => 'nullable|in:percentage,fixed',
                'bulk_prices.*.partner_margin_type' => 'nullable|in:percentage,fixed',
                'bulk_prices.*.delivery_fee' => 'nullable|numeric|min:0',
                
                // Default margins for product
                'default_customer_margin' => 'nullable|numeric|min:0|max:100',
                'default_partner_margin' => 'nullable|numeric|min:0|max:100',
                'default_customer_margin_type' => 'nullable|in:percentage,fixed',
                'default_partner_margin_type' => 'nullable|in:percentage,fixed',
                
                // Stock information
                'show_stock_price' => 'nullable|boolean',
                'quantity' => 'nullable|integer|min:0',
                
                // SEO information
                'meta_title' => 'nullable|string|max:70',
                'meta_description' => 'nullable|string|max:160',
                'meta_keywords' => 'nullable|string|max:255',
                
                // Product images
                'images' => 'nullable|array|max:10',
                'images.*' => 'nullable|url|max:255',
            ]);
            
            Log::info('Validation passed for product creation');
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Validation failed for product creation', [
                'errors' => $e->errors(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Product validation failed',
                'errors' => $e->errors(),
                'meta' => [
                    'code' => 422,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 422);
        } catch (\Exception $e) {
            Log::error('Unexpected error during product validation', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'An unexpected error occurred during validation',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }

        // Database operations with transaction
        DB::beginTransaction();
        
        try {
            // Get default currency if not provided
            $currencyCode = $validated['currency_code'] ?? Currency::getDefaultCode();
            
            // Generate slug from name
            $slug = Str::slug($validated['name']);
            
            // Check if slug exists, if so, append a unique identifier
            $slugCount = Product::where('slug', 'LIKE', $slug . '%')->count();
            if ($slugCount > 0) {
                $slug = $slug . '-' . ($slugCount + 1);
            }

            // Create the product
            $product = Product::create([
                'name' => $validated['name'],
                'title' => $validated['name'], // Using name as title for simplicity
                'slug' => $slug,
                'sku' => $validated['sku'],
                'brand_id' => $validated['brand_id'],
                'vendor_id' => $request->user()->id, // Assuming vendor is the logged-in user
                'created_by' => $request->user()->id,
                'description' => $validated['description'],
                'quantity' => $validated['quantity'] ?? 0,
                'stock_status' => ($validated['quantity'] ?? 0) > 0 ? 'in_stock' : 'out_of_stock',
                'meta_title' => $validated['meta_title'] ?? $validated['name'],
                'meta_description' => $validated['meta_description'] ?? null,
                'meta_keywords' => $validated['meta_keywords'] ? json_encode(explode(',', $validated['meta_keywords'])) : null,
                'is_approved' => $request->user()->hasRole(['superadmin', 'admin']), // Auto-approve for admins
                'approval_status' => $request->user()->hasRole(['superadmin', 'admin']) ? 'approved' : 'pending',
                'is_active' => true,
                'currency_code' => $currencyCode,
                'default_customer_margin' => $validated['default_customer_margin'] ?? null,
                'default_partner_margin' => $validated['default_partner_margin'] ?? null,
                'default_customer_margin_type' => $validated['default_customer_margin_type'] ?? 'percentage',
                'default_partner_margin_type' => $validated['default_partner_margin_type'] ?? 'percentage',
                'min_order_quantity' => $validated['min_order_quantity'] ?? null,
                'min_order_value' => $validated['min_order_value'] ?? null,
                'free_shipping_threshold' => $validated['free_shipping_threshold'] ?? null,
                'products_per_pallet' => $validated['products_per_pallet'] ?? null,
                'pallet_delivery_fee' => $validated['pallet_delivery_fee'] ?? null,
            ]);

            Log::info('Product created successfully', ['product_id' => $product->id]);

            // Handle pack pricing with delivery fee
            ProductPackPrice::create([
                'product_id' => $product->id,
                'number_of_products' => $validated['number_of_products'],
                'per_pack_price' => $validated['per_pack_price'],
                'per_pack_special_price' => $validated['per_pack_special_price'] ?? null,
                'customer_margin' => $validated['customer_margin'] ?? null,
                'partner_margin' => $validated['partner_margin'] ?? null,
                'customer_margin_type' => $validated['customer_margin_type'] ?? 'percentage',
                'partner_margin_type' => $validated['partner_margin_type'] ?? 'percentage',
                'delivery_fee' => $validated['pack_delivery_fee'] ?? null, // New delivery fee field
            ]);

            Log::info('Product pack price created');

            // Handle bulk pricing if enabled with delivery fees
            if (isset($validated['add_bulk_prices']) && $validated['add_bulk_prices'] && 
                isset($validated['bulk_prices']) && is_array($validated['bulk_prices'])) {
                
                foreach ($validated['bulk_prices'] as $index => $bulkPrice) {
                    ProductBulkPrice::create([
                        'product_id' => $product->id,
                        'number_of_packs' => $bulkPrice['number_of_packs'],
                        'per_pack_price' => $bulkPrice['per_pack_price'],
                        'per_pack_special_price' => $bulkPrice['per_pack_special_price'] ?? null,
                        'customer_margin' => $bulkPrice['customer_margin'] ?? null,
                        'partner_margin' => $bulkPrice['partner_margin'] ?? null,
                        'customer_margin_type' => $bulkPrice['customer_margin_type'] ?? 'percentage',
                        'partner_margin_type' => $bulkPrice['partner_margin_type'] ?? 'percentage',
                        'delivery_fee' => $bulkPrice['delivery_fee'] ?? null, // New delivery fee field
                        'sort_order' => $index,
                    ]);
                }
                
                Log::info('Product bulk prices created', ['count' => count($validated['bulk_prices'])]);
            }

            // Handle product categories (assuming a single category from the form)
            if ($validated['category_id']) {
                DB::table('product_categories')->insert([
                    'product_id' => $product->id,
                    'category_id' => $validated['category_id'],
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
                
                Log::info('Product category association created');
            }

            // Handle product images
            if (isset($validated['images']) && is_array($validated['images'])) {
                foreach ($validated['images'] as $index => $imageUrl) {
                    if ($imageUrl) {
                        ProductImage::create([
                            'product_id' => $product->id,
                            'image_url' => $imageUrl,
                            'alt_text' => $validated['name'],
                            'sort_order' => $index,
                        ]);
                    }
                }
                
                Log::info('Product images created', ['count' => count(array_filter($validated['images']))]);
            }

            // Commit the transaction
            DB::commit();

            // Load related data for response
            $product->load(['brand', 'packPrice', 'bulkPrices', 'images', 'currency', 'categories']);

            // Clear product caches
            $this->clearModelCache($product);
            Cache::forget("products_list_*"); // Clear all product list caches

            return response()->json([
                'success' => true,
                'message' => 'Product created successfully',
                'data' => [
                    'product' => $product
                ],
                'meta' => [
                    'code' => 201,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 201);
            
        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollBack();
            
            Log::error('Error creating product', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error creating product',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
    
    /**
     * Display a listing of the products with pagination.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $perPage = $request->input('per_page', 12);
            $sortBy = $request->input('sort_by', 'created_at');
            $sortDirection = $request->input('sort_direction', 'desc');

            $query = Product::with([
                'brand',
                'categories',
                'images',
                'packPrice',
                'bulkPrices',
                'currency',
                'variantStocks',
                'variantStocks.variantAttributes'
            ])
            ->where('approval_status', 'approved')
            ->where('is_active', true);

            // Apply filters
            if ($request->has('search')) {
                $query->search($request->input('search'));
            }

            if ($request->has('brand_id')) {
                $query->where('brand_id', $request->input('brand_id'));
            }

            if ($request->has('min_price') && $request->has('max_price')) {
                $query->byPriceRange($request->input('min_price'), $request->input('max_price'));
            }

            // Apply sorting
            $query->orderBy($sortBy, $sortDirection);

            $products = $query->paginate($perPage);

            // Transform the data to include only necessary fields
            $products->getCollection()->transform(function ($product) {
                return $this->getPublicProductData($product);
            });

            return response()->json([
                'success' => true,
                'data' => [
                    'products' => $products
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching products', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error fetching products',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified product.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function show(int $id): JsonResponse
    {
        try {
            // Get the authenticated user if any
            $user = auth()->user();
            
            // Log the request details
            Log::info('Retrieving product', [
                'product_id' => $id,
                'user_id' => $user ? $user->id : null,
                'user_role' => $user ? ($user->role ? $user->role->slug : null) : null
            ]);

            // Get the product with all necessary relationships
            $product = Product::with([
                'brand',
                'categories',
                'images',
                'packPrice',
                'bulkPrices',
                'currency',
                'variantStocks',
                'variantStocks.variantAttributes'
            ])->find($id);

            if (!$product) {
                Log::warning('Product not found', ['product_id' => $id]);
                return response()->json([
                    'success' => false,
                    'message' => 'Product not found'
                ], 404);
            }

            // Check if user has permission to view this product
            if (!$user || !in_array($user->role?->slug, ['superadmin', 'admin', 'supervisor'])) {
                if ($product->approval_status !== 'approved' || !$product->is_active) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Product not found'
                    ], 404);
                }
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'product' => [
                        'id' => $product->id,
                        'name' => $product->name,
                        'slug' => $product->slug,
                        'description' => $product->description,
                        'sku' => $product->sku,
                        'quantity' => $product->quantity,
                        'stock_status' => $product->stock_status,
                        'meta' => $product->meta ?? [
                            'title' => $product->meta_title,
                            'description' => $product->meta_description,
                            'keywords' => $product->meta_keywords,
                        ],
                        'features' => $product->features,
                        'min_order_quantity' => $product->min_order_quantity,
                        'min_order_value' => $product->min_order_value,
                        'free_shipping_threshold' => $product->free_shipping_threshold,
                        'tax_rate' => $product->getTaxRate(),
                        'brand' => $product->brand ? [
                            'id' => $product->brand->id,
                            'name' => $product->brand->name,
                            'slug' => $product->brand->slug,
                            'description' => $product->brand->description,
                            'logo' => $product->brand->logo,
                        ] : null,
                        'categories' => $product->categories->map(function ($category) {
                            return [
                                'id' => $category->id,
                                'name' => $category->name,
                                'slug' => $category->slug,
                                'description' => $category->description,
                            ];
                        }),
                        'images' => $product->images->map(function ($image) {
                            return [
                                'id' => $image->id,
                                'url' => $image->full_image_url, // Use full URL for consistency
                                'alt' => $image->alt_text,
                                'sort_order' => $image->sort_order
                            ];
                        }),
                        'pack_price' => $product->packPrice ? [
                            'number_of_products' => $product->packPrice->number_of_products,
                            'per_pack_price' => $product->packPrice->per_pack_price,
                            'per_pack_special_price' => $product->packPrice->per_pack_special_price,
                            'customer_margin' => $product->packPrice->customer_margin,
                            'partner_margin' => $product->packPrice->partner_margin,
                            'customer_margin_type' => $product->packPrice->customer_margin_type,
                            'partner_margin_type' => $product->packPrice->partner_margin_type,
                            'delivery_fee' => $product->packPrice->delivery_fee,
                        ] : null,
                        'bulk_prices' => $product->bulkPrices->map(function ($price) {
                            return [
                                'number_of_packs' => $price->number_of_packs,
                                'per_pack_price' => $price->per_pack_price,
                                'per_pack_special_price' => $price->per_pack_special_price,
                                'customer_margin' => $price->customer_margin,
                                'partner_margin' => $price->partner_margin,
                                'customer_margin_type' => $price->customer_margin_type,
                                'partner_margin_type' => $price->partner_margin_type,
                                'delivery_fee' => $price->delivery_fee,
                                'sort_order' => $price->sort_order,
                                'tax_rate' => $price->getTaxRate(),
                            ];
                        }),
                        'currency' => $product->currency ? [
                            'id' => $product->currency->id,
                            'code' => $product->currency->code,
                            'name' => $product->currency->name,
                            'symbol' => $product->currency->symbol,
                            'exchange_rate' => $product->currency->exchange_rate,
                        ] : null,
                    ]
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching product', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error fetching product',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Format product data for response
     *
     * @param Product $product
     * @return array
     */
    private function formatProductData($product)
    {
        return [
            'id' => $product->id,
            'name' => $product->name,
            'title' => $product->title,
            'slug' => $product->slug,
            'sku' => $product->sku,
            'description' => $product->description,
            'quantity' => $product->quantity,
            'stock_status' => $product->stock_status,
            'is_approved' => $product->is_approved,
            'approval_status' => $product->approval_status,
            'is_active' => $product->is_active,
            'is_featured' => $product->is_featured,
            'is_bestseller' => $product->is_bestseller,
            'is_trending' => $product->is_trending,
            'created_at' => $product->created_at,
            'updated_at' => $product->updated_at,
            
            // Brand information
            'brand' => $product->brand ? [
                'id' => $product->brand->id,
                'name' => $product->brand->name,
                'slug' => $product->brand->slug
            ] : null,
            
            // Categories
            'categories' => $product->categories->map(function ($category) {
                return [
                    'id' => $category->id,
                    'name' => $category->name,
                    'slug' => $category->slug
                ];
            }),
            
            // Images
            'images' => $product->images->map(function ($image) {
                return [
                    'id' => $image->id,
                    'url' => $image->full_image_url, // Use full URL for consistency
                    'alt' => $image->alt_text,
                    'sort_order' => $image->sort_order
                ];
            }),
            
            // Pack price
            'pack_price' => $product->packPrice ? [
                'id' => $product->packPrice->id,
                'number_of_products' => $product->packPrice->number_of_products,
                'per_pack_price' => $product->packPrice->per_pack_price,
                'per_pack_special_price' => $product->packPrice->per_pack_special_price,
                'customer_margin' => $product->packPrice->customer_margin,
                'partner_margin' => $product->packPrice->partner_margin,
                'customer_margin_type' => $product->packPrice->customer_margin_type,
                'partner_margin_type' => $product->packPrice->partner_margin_type,
                'delivery_fee' => $product->packPrice->delivery_fee
            ] : null,
            
            // Currency
            'currency' => $product->currency ? [
                'id' => $product->currency->id,
                'code' => $product->currency->code,
                'name' => $product->currency->name,
                'symbol' => $product->currency->symbol
            ] : null,
            
            // Variant stocks
            'variant_stocks' => $product->variantStocks->map(function ($stock) {
                return [
                    'id' => $stock->id,
                    'variant_attributes' => $stock->variant_attributes,
                    'quantity' => $stock->quantity,
                    'stock_status' => $stock->stock_status
                ];
            }),
            
            // Order thresholds
            'min_order_quantity' => $product->min_order_quantity,
            'min_order_value' => $product->min_order_value,
            'free_shipping_threshold' => $product->free_shipping_threshold,
            
            // Pallet information
            'products_per_pallet' => $product->products_per_pallet,
            'pallet_delivery_fee' => $product->pallet_delivery_fee,
            
            // Default margins
            'default_customer_margin' => $product->default_customer_margin,
            'default_partner_margin' => $product->default_partner_margin,
            'default_customer_margin_type' => $product->default_customer_margin_type,
            'default_partner_margin_type' => $product->default_partner_margin_type
        ];
    }

    /**
     * Toggle the approval status of a product.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function toggleApproval(Request $request, int $id): JsonResponse
    {
        try {
            $product = Product::findOrFail($id);
            
            $validated = $request->validate([
                'final_commission' => 'nullable|numeric|min:0|max:100',
                'status' => 'nullable|in:pending,approved,rejected',
                'rejection_reason' => 'required_if:status,rejected|nullable|string',
            ]);

            // Determine the new status
            if (isset($validated['status'])) {
                // Explicit status provided
                $newStatus = $validated['status'];
            } else {
                // Toggle between approved and pending
                $newStatus = $product->approval_status === 'approved' ? 'pending' : 'approved';
            }

            // Update both approval fields
            $product->approval_status = $newStatus;
            $product->is_approved = ($newStatus === 'approved');
            
            // Store rejection reason if applicable
            if ($newStatus === 'rejected' && isset($validated['rejection_reason'])) {
                $product->rejection_reason = $validated['rejection_reason'];
            }
            
            // Set approval metadata
            if ($newStatus === 'approved') {
                $product->approved_by = $request->user()->id;
                $product->approval_date = now();
            } else {
                // Clear approval metadata if status is not approved
                $product->approved_by = null;
                $product->approval_date = null;
            }
            
            if (isset($validated['final_commission'])) {
                $product->final_commission = $validated['final_commission'];
            }
            
            $product->save();
            
            // Clear all product-related caches
            $this->clearModelCache($product);
            Cache::forget('product_' . $product->id);
            Cache::forget('products_list_*');
            Cache::forget('public_products_list_*');
            Cache::forget('vendor_products_list_*');
            Cache::forget('admin_products_list_*');
            Cache::forget('superadmin_products_list_*');
            
            // Prepare appropriate message
            $messages = [
                'approved' => 'Product approved successfully',
                'pending' => 'Product approval revoked successfully',
                'rejected' => 'Product rejected successfully'
            ];
            
            return response()->json([
                'success' => true,
                'message' => $messages[$newStatus] ?? 'Product status updated successfully',
                'data' => ['product' => $product->fresh()->load(['brand', 'packPrice', 'bulkPrices', 'images', 'currency', 'categories'])]
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error toggling product approval', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'product_id' => $id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating product approval status',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }



    /**
     * Perform bulk actions on products (delete, activate, deactivate).
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function bulkAction(Request $request): JsonResponse
    {
        Log::info('Request received for bulk action on products', $request->all());

        $validated = $request->validate([
            'action' => 'required|in:delete,activate,deactivate',
            'product_ids' => 'required|array',
            'product_ids.*' => 'exists:products,id',
        ]);

        $action = $validated['action'];
        $productIds = $validated['product_ids'];

        DB::beginTransaction();
        
        try {
            switch ($action) {
                case 'delete':
                    // Delete associated data
                    $products = Product::whereIn('id', $productIds)->get();
                    foreach ($products as $product) {
                        // Delete images
                        $product->images()->delete();
                        
                        // Delete pack price
                        $product->packPrice()->delete();
                        
                        // Delete bulk prices
                        $product->bulkPrices()->delete();
                        
                        // Delete product category relationships
                        DB::table('product_categories')->where('product_id', $product->id)->delete();
                    }
                    
                    // Delete products
                    Product::whereIn('id', $productIds)->delete();
                    $message = 'Products deleted successfully';
                    break;
                    
                case 'activate':
                    Product::whereIn('id', $productIds)->update(['is_active' => true]);
                    $message = 'Products activated successfully';
                    break;
                    
                case 'deactivate':
                    Product::whereIn('id', $productIds)->update(['is_active' => false]);
                    $message = 'Products deactivated successfully';
                    break;
                    
                default:
                    DB::rollBack();
                    return response()->json([
                        'success' => false,
                        'message' => 'Invalid action',
                        'meta' => [
                            'code' => 400,
                            'timestamp' => now()->toIso8601String(),
                        ],
                    ], 400);
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => $message,
                'data' => [],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Error performing bulk action on products', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error performing bulk action',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }

    /**
    * Display the specified product for public view (cached, minimal fields).
    *
    * @param int $id
    * @return JsonResponse
    */
    public function publicProductView(int $id): JsonResponse
    {
        try {
            $product = Product::with([
                'brand:id,name,slug,description,logo',
                'categories:id,name,slug,description',
                'images:id,product_id,image_url,alt_text,sort_order',
                'packPrice:id,product_id,number_of_products,per_pack_price,per_pack_special_price,customer_margin,partner_margin,customer_margin_type,partner_margin_type,delivery_fee',
                'bulkPrices:id,product_id,number_of_packs,per_pack_price,per_pack_special_price,customer_margin,partner_margin,customer_margin_type,partner_margin_type,delivery_fee,sort_order,tax_rate',
                'currency:id,code,name,symbol,exchange_rate',
                'variantStocks:id,product_id,variant_attributes,quantity,stock_status'
            ])->findOrFail($id);

            if (!$product->is_approved || !$product->is_active) {
                return response()->json([
                    'success' => false,
                    'message' => 'Product not available',
                    'details' => 'This product is either not approved or inactive'
                ], 404);
            }

            // Transform the product data to include all necessary information
            $productData = [
                'id' => $product->id,
                'name' => $product->name,
                'slug' => $product->slug,
                'description' => $product->description,
                'sku' => $product->sku,
                'quantity' => $product->quantity,
                'stock_status' => $product->stock_status,
                'meta' => $product->meta ?? [
                    'title' => $product->meta_title,
                    'description' => $product->meta_description,
                    'keywords' => $product->meta_keywords,
                ],
                'features' => $product->features,
                'min_order_quantity' => $product->min_order_quantity,
                'min_order_value' => $product->min_order_value,
                'free_shipping_threshold' => $product->free_shipping_threshold,
                'brand' => $product->brand ? [
                    'id' => $product->brand->id,
                    'name' => $product->brand->name,
                    'slug' => $product->brand->slug,
                    'description' => $product->brand->description,
                    'logo' => $product->brand->logo,
                ] : null,
                'categories' => $product->categories->map(function ($category) {
                    return [
                        'id' => $category->id,
                        'name' => $category->name,
                        'slug' => $category->slug,
                        'description' => $category->description,
                    ];
                }),
                'images' => $product->images->map(function ($image) {
                    return [
                        'id' => $image->id,
                        'url' => $image->image_url,
                        'alt_text' => $image->alt_text,
                        'sort_order' => $image->sort_order,
                    ];
                }),
                'pack_price' => $product->packPrice ? [
                    'number_of_products' => $product->packPrice->number_of_products,
                    'per_pack_price' => $product->packPrice->per_pack_price,
                    'per_pack_special_price' => $product->packPrice->per_pack_special_price,
                    'customer_margin' => $product->packPrice->customer_margin,
                    'partner_margin' => $product->packPrice->partner_margin,
                    'customer_margin_type' => $product->packPrice->customer_margin_type,
                    'partner_margin_type' => $product->packPrice->partner_margin_type,
                    'delivery_fee' => $product->packPrice->delivery_fee,
                    'tax_rate' => $product->packPrice->getTaxRate()
                ] : null,
                'bulk_prices' => $product->bulkPrices->map(function ($price) {
                    return [
                        'number_of_packs' => $price->number_of_packs,
                        'per_pack_price' => $price->per_pack_price,
                        'per_pack_special_price' => $price->per_pack_special_price,
                        'customer_margin' => $price->customer_margin,
                        'partner_margin' => $price->partner_margin,
                        'customer_margin_type' => $price->customer_margin_type,
                        'partner_margin_type' => $price->partner_margin_type,
                        'delivery_fee' => $price->delivery_fee,
                        'sort_order' => $price->sort_order,
                        'tax_rate' => $price->getTaxRate()
                    ];
                }),
                'currency' => $product->currency ? [
                    'id' => $product->currency->id,
                    'code' => $product->currency->code,
                    'name' => $product->currency->name,
                    'symbol' => $product->currency->symbol,
                    'exchange_rate' => $product->currency->exchange_rate,
                ] : null,
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'product' => $productData
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error in publicProductView: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving product',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
 * Update the specified product in storage.
 *
 * @param Request $request
 * @param int $id
 * @return JsonResponse
 */
public function update(Request $request, int $id): JsonResponse
{
    try {
        $product = Product::findOrFail($id);
        
        // Check if user is authorized to update this product
        if ($request->user()->id !== $product->vendor_id && 
            !$request->user()->hasRole(['superadmin', 'admin', 'supervisor'])) {
            return response()->json([
                'success' => false,
                'message' => 'You are not authorized to update this product',
                'meta' => [
                    'code' => 403,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 403);
        }

        try {
            $validated = $request->validate([
                'name' => 'sometimes|string|max:255|unique:products,name,' . $id,
                'brand_id' => 'sometimes|exists:brands,id',
                'category_id' => 'sometimes|exists:categories,id',
                'description' => 'sometimes|string',
                'sku' => 'sometimes|string|max:100|unique:products,sku,' . $id,
                'currency_code' => 'sometimes|exists:currencies,code',
                
                // Default margins
                'default_customer_margin' => 'nullable|numeric|min:0|max:100',
                'default_partner_margin' => 'nullable|numeric|min:0|max:100',
                'default_customer_margin_type' => 'nullable|in:percentage,fixed',
                'default_partner_margin_type' => 'nullable|in:percentage,fixed',
                
                // Additional fields for order thresholds
                'min_order_quantity' => 'nullable|integer|min:1',
                'min_order_value' => 'nullable|numeric|min:0',
                'free_shipping_threshold' => 'nullable|integer|min:1',
                'products_per_pallet' => 'nullable|integer|min:1',
                'pallet_delivery_fee' => 'nullable|numeric|min:0',
                
                // Pack pricing
                'pack_price' => 'sometimes|array',
                'pack_price.number_of_products' => 'required_with:pack_price|integer|min:1',
                'pack_price.per_pack_price' => 'required_with:pack_price|numeric|min:0',
                'pack_price.per_pack_special_price' => 'nullable|numeric|min:0',
                'pack_price.customer_margin' => 'nullable|numeric|min:0|max:100',
                'pack_price.partner_margin' => 'nullable|numeric|min:0|max:100',
                'pack_price.customer_margin_type' => 'nullable|in:percentage,fixed',
                'pack_price.partner_margin_type' => 'nullable|in:percentage,fixed',
                'pack_price.delivery_fee' => 'nullable|numeric|min:0',
                
                // Bulk prices
                'bulk_prices' => 'sometimes|array',
                'bulk_prices.*.id' => 'sometimes|exists:product_bulk_prices,id',
                'bulk_prices.*.number_of_packs' => 'required_with:bulk_prices|integer|min:1',
                'bulk_prices.*.per_pack_price' => 'required_with:bulk_prices|numeric|min:0',
                'bulk_prices.*.per_pack_special_price' => 'nullable|numeric|min:0',
                'bulk_prices.*.customer_margin' => 'nullable|numeric|min:0|max:100',
                'bulk_prices.*.partner_margin' => 'nullable|numeric|min:0|max:100',
                'bulk_prices.*.customer_margin_type' => 'nullable|in:percentage,fixed',
                'bulk_prices.*.partner_margin_type' => 'nullable|in:percentage,fixed',
                'bulk_prices.*.delivery_fee' => 'nullable|numeric|min:0',
                
                // Stock information
                'quantity' => 'sometimes|nullable|integer|min:0',
                
                // SEO information
                'meta_title' => 'sometimes|nullable|string|max:70',
                'meta_description' => 'sometimes|nullable|string|max:160',
                'meta_keywords' => 'sometimes|nullable|string|max:255',
                
                // Images
                'images' => 'sometimes|nullable|array|max:10',
                'images.*' => 'nullable|url|max:255',
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(),
                'meta' => [
                    'code' => 422,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 422);
        }

        DB::beginTransaction();
        
        try {
            // Build update data array - only include fields that were provided
            $updateData = [];
            
            if (isset($validated['name'])) {
                $updateData['name'] = $validated['name'];
                $updateData['title'] = $validated['name'];
                
                // Generate new slug if name changed
                if ($validated['name'] !== $product->name) {
                    $slug = Str::slug($validated['name']);
                    $slugCount = Product::where('slug', 'LIKE', $slug . '%')
                        ->where('id', '!=', $id)
                        ->count();
                    if ($slugCount > 0) {
                        $slug = $slug . '-' . ($slugCount + 1);
                    }
                    $updateData['slug'] = $slug;
                }
            }
            
            // Other basic fields
            if (isset($validated['brand_id'])) $updateData['brand_id'] = $validated['brand_id'];
            if (isset($validated['description'])) $updateData['description'] = $validated['description'];
            if (isset($validated['sku'])) $updateData['sku'] = $validated['sku'];
            if (isset($validated['currency_code'])) $updateData['currency_code'] = $validated['currency_code'];
            
            // Default margins
            if (isset($validated['default_customer_margin'])) $updateData['default_customer_margin'] = $validated['default_customer_margin'];
            if (isset($validated['default_partner_margin'])) $updateData['default_partner_margin'] = $validated['default_partner_margin'];
            if (isset($validated['default_customer_margin_type'])) $updateData['default_customer_margin_type'] = $validated['default_customer_margin_type'];
            if (isset($validated['default_partner_margin_type'])) $updateData['default_partner_margin_type'] = $validated['default_partner_margin_type'];
            
            // Order thresholds and pallet info
            if (isset($validated['min_order_quantity'])) $updateData['min_order_quantity'] = $validated['min_order_quantity'];
            if (isset($validated['min_order_value'])) $updateData['min_order_value'] = $validated['min_order_value'];
            if (isset($validated['free_shipping_threshold'])) $updateData['free_shipping_threshold'] = $validated['free_shipping_threshold'];
            if (isset($validated['products_per_pallet'])) $updateData['products_per_pallet'] = $validated['products_per_pallet'];
            if (isset($validated['pallet_delivery_fee'])) $updateData['pallet_delivery_fee'] = $validated['pallet_delivery_fee'];
            
            // Stock
            if (isset($validated['quantity'])) {
                $updateData['quantity'] = $validated['quantity'];
                $updateData['stock_status'] = $validated['quantity'] > 0 ? 
                    ($validated['quantity'] <= 5 ? 'low_stock' : 'in_stock') : 'out_of_stock';
            }
            
            // SEO
            if (isset($validated['meta_title'])) $updateData['meta_title'] = $validated['meta_title'];
            if (isset($validated['meta_description'])) $updateData['meta_description'] = $validated['meta_description'];
            if (isset($validated['meta_keywords'])) {
                $updateData['meta_keywords'] = json_encode(explode(',', $validated['meta_keywords']));
            }
            
            // Update the product
            $product->update($updateData);
            
            // Update pack price if provided
            if (isset($validated['pack_price'])) {
                $packPrice = $product->packPrice;
                
                if ($packPrice) {
                    $packPrice->update([
                        'number_of_products' => $validated['pack_price']['number_of_products'],
                        'per_pack_price' => $validated['pack_price']['per_pack_price'],
                        'per_pack_special_price' => $validated['pack_price']['per_pack_special_price'] ?? null,
                        'customer_margin' => $validated['pack_price']['customer_margin'] ?? null,
                        'partner_margin' => $validated['pack_price']['partner_margin'] ?? null,
                        'customer_margin_type' => $validated['pack_price']['customer_margin_type'] ?? 'percentage',
                        'partner_margin_type' => $validated['pack_price']['partner_margin_type'] ?? 'percentage',
                        'delivery_fee' => $validated['pack_price']['delivery_fee'] ?? null,
                    ]);
                } else {
                    // Create pack price if it doesn't exist
                    ProductPackPrice::create(array_merge(
                        $validated['pack_price'],
                        ['product_id' => $product->id]
                    ));
                }
            }
            
            // Update bulk prices if provided
            if (isset($validated['bulk_prices']) && is_array($validated['bulk_prices'])) {
                // Delete existing bulk prices - safer approach to avoid orphaned data
                $product->bulkPrices()->delete();
                
                // Create new bulk prices
                foreach ($validated['bulk_prices'] as $index => $bulkPrice) {
                    ProductBulkPrice::create(array_merge(
                        $bulkPrice,
                        [
                            'product_id' => $product->id,
                            'sort_order' => $index,
                        ]
                    ));
                }
            }
            
            // Update category if provided
            if (isset($validated['category_id'])) {
                // Remove existing category associations
                DB::table('product_categories')->where('product_id', $product->id)->delete();
                
                // Add new category association
                DB::table('product_categories')->insert([
                    'product_id' => $product->id,
                    'category_id' => $validated['category_id'],
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
            
            // Update images if provided
            if (isset($validated['images']) && is_array($validated['images'])) {
                // Delete existing images
                $product->images()->delete();
                
                // Add new images
                foreach ($validated['images'] as $index => $imageUrl) {
                    if ($imageUrl) {
                        ProductImage::create([
                            'product_id' => $product->id,
                            'image_url' => $imageUrl,
                            'alt_text' => $product->name,
                            'sort_order' => $index,
                        ]);
                    }
                }
            }
            
            DB::commit();
            
            // Load relationships for response
            $product->refresh();
            $product->load(['brand', 'packPrice', 'bulkPrices', 'images', 'currency', 'categories']);
            
            // Clear product caches
            $this->clearModelCache($product);
            Cache::forget("products_list_*"); // Clear all product list caches
            
            return response()->json([
                'success' => true,
                'message' => 'Product updated successfully',
                'data' => ['product' => $product],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 200);
            
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
        
    } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Product not found',
            'meta' => [
                'code' => 404,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 404);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error updating product',
            'error' => $e->getMessage(),
            'meta' => [
                'code' => 500,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 500);
    }
}
    /**
     * Remove the specified product from storage.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function destroy(int $id): JsonResponse
    {
        Log::info('Request received for deleting product', ['product_id' => $id]);

        try {
            $product = Product::findOrFail($id);
            $currentUserId = auth()->id();
            $currentUser = auth()->user();

            // Null checks for user and role
            if (!$currentUser) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not authenticated',
                    'meta' => [
                        'code' => 401,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 401);
            }
            $userRoleSlug = $currentUser->role ? $currentUser->role->slug : null;
            $userHasAdminRole = $currentUser->hasRole(['superadmin', 'admin', 'supervisor']);
            $isOwner = ((int)$currentUserId == (int)$product->vendor_id) && $currentUser->hasRole('vendor');
            $authorizationPassed = $isOwner || $userHasAdminRole;

            // Debug logging
            Log::info('Product deletion authorization check', [
                'product_id' => $id,
                'product_vendor_id' => $product->vendor_id,
                'current_user_id' => $currentUserId,
                'current_user_role' => $userRoleSlug,
                'user_has_admin_role' => $userHasAdminRole,
                'is_owner' => $isOwner,
                'authorization_passed' => $authorizationPassed
            ]);

            // Check if user is authorized to delete this product
            if (!$authorizationPassed) {
                Log::warning('Product deletion authorization failed', [
                    'product_id' => $id,
                    'product_vendor_id' => $product->vendor_id,
                    'current_user_id' => $currentUserId,
                    'current_user_role' => $userRoleSlug
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'You are not authorized to delete this product',
                    'debug_info' => [
                        'product_vendor_id' => $product->vendor_id,
                        'current_user_id' => $currentUserId,
                        'is_owner' => $isOwner,
                        'user_role' => $userRoleSlug
                    ],
                    'meta' => [
                        'code' => 403,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 403);
            }
            
            DB::beginTransaction();
    
            // Delete related data
            $product->images()->delete();
            $product->packPrice()->delete();
            $product->bulkPrices()->delete();
            
            // Delete product category relationships
            DB::table('product_categories')->where('product_id', $product->id)->delete();
            
            // Delete the product
            $product->delete();
            
            DB::commit();
    
            Log::info('Product deleted successfully', [
                'product_id' => $id,
                'deleted_by_user_id' => $currentUserId
            ]);
    
            return response()->json([
                'success' => true,
                'message' => 'Product deleted successfully',
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Error deleting product', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error deleting product',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }

    /**
     * Display homepage sections (best sellers, new arrivals, trending) with pagination.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function homepageSections(Request $request): JsonResponse
    {
        Log::info('Request received for retrieving homepage sections', $request->all());

        $perPage = $request->input('per_page', 10);
        $page = $request->input('page', 1);
        $currencyCode = $request->input('currency_code');

        // Base query with currency filter if provided
        $baseQuery = Product::query()
            ->where('approval_status', 'approved')
            ->where('is_active', true);
            
        if ($currencyCode) {
            $baseQuery->where('currency_code', $currencyCode);
        }

        // Best sellers query
        $bestSellersQuery = clone $baseQuery;
        $bestSellersQuery->where('is_bestseller', true)
            ->with(['categories', 'brand', 'images', 'packPrice', 'currency']);
        $bestSellers = $bestSellersQuery->paginate($perPage, ['*'], 'best_sellers_page', $page);

        // New arrivals query
        $newArrivalsQuery = clone $baseQuery;
        $newArrivalsQuery->where('created_at', '>=', now()->subDays(30))
            ->with(['categories', 'brand', 'images', 'packPrice', 'currency'])
            ->orderBy('created_at', 'desc');
        $newArrivals = $newArrivalsQuery->paginate($perPage, ['*'], 'new_arrivals_page', $page);

        // Trending query
        $trendingQuery = clone $baseQuery;
        $trendingQuery->where('is_trending', true)
            ->with(['categories', 'brand', 'images', 'packPrice', 'currency']);
        $trending = $trendingQuery->paginate($perPage, ['*'], 'trending_page', $page);

        return response()->json([
            'success' => true,
            'message' => 'Homepage sections retrieved successfully',
            'data' => [
                'best_sellers' => $bestSellers->items(),
                'new_arrivals' => $newArrivals->items(),
                'trending' => $trending->items(),
            ],
            'meta' => [
                'code' => 200,
                'timestamp' => now()->toIso8601String(),
                'pagination' => [
                    'best_sellers' => [
                        'total' => $bestSellers->total(),
                        'per_page' => $bestSellers->perPage(),
                        'current_page' => $bestSellers->currentPage(),
                        'last_page' => $bestSellers->lastPage(),
                        'from' => $bestSellers->firstItem(),
                        'to' => $bestSellers->lastItem(),
                    ],
                    'new_arrivals' => [
                        'total' => $newArrivals->total(),
                        'per_page' => $newArrivals->perPage(),
                        'current_page' => $newArrivals->currentPage(),
                        'last_page' => $newArrivals->lastPage(),
                        'from' => $newArrivals->firstItem(),
                        'to' => $newArrivals->lastItem(),
                    ],
                    'trending' => [
                        'total' => $trending->total(),
                        'per_page' => $trending->perPage(),
                        'current_page' => $trending->currentPage(),
                        'last_page' => $trending->lastPage(),
                        'from' => $trending->firstItem(),
                        'to' => $trending->lastItem(),
                    ],
                ],
            ],
        ], 200);
    }

    /**
     * Display a listing of the products for the authenticated vendor.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function vendorProducts(Request $request): JsonResponse
    {
        $userId = $request->user()->id;
        
        $query = Product::where('vendor_id', $userId)
                    ->with(['packPrice', 'bulkPrices', 'images', 'brand', 'categories', 'currency'])
                    ->orderBy('created_at', 'desc');
        
        // Apply filters
        if ($request->has('search')) {
            $search = $request->input('search');
            $query->search($search);
        }
        
        if ($request->has('brand_id')) {
            $query->where('brand_id', $request->input('brand_id'));
        }
        
        if ($request->has('category_id')) {
            $query->byCategory($request->input('category_id'));
        }
        
        if ($request->has('currency_code')) {
            $query->where('currency_code', $request->input('currency_code'));
        }
        
        if ($request->has('approval_status')) {
            $query->where('approval_status', $request->input('approval_status'));
        }
        
        // Apply sorting
        $sortBy = $request->input('sort_by', 'created_at');
        $sortDirection = $request->input('sort_direction', 'desc');
        
        switch ($sortBy) {
            case 'name':
                $query->orderBy('name', $sortDirection);
                break;
            case 'price':
                $query->join('product_pack_prices', 'products.id', '=', 'product_pack_prices.product_id')
                      ->select('products.*')
                      ->orderBy('product_pack_prices.per_pack_price', $sortDirection);
                break;
            case 'approval_status':
                $query->orderBy('approval_status', $sortDirection);
                break;
            case 'created_at':
                $query->orderBy('created_at', $sortDirection);
                break;
            default:
                $query->orderBy('created_at', 'desc');
                break;
        }
        
        // Pagination
        $perPage = $request->input('per_page', 10);
        $products = $query->paginate($perPage);
        
        return response()->json([
            'success' => true,
            'message' => 'Vendor products retrieved successfully',
            'data' => ['products' => $products],
            'meta' => [
                'code' => 200,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 200);
    }

    /**
     * Display a listing of all products with advanced filtering for superadmins.
     *
     * This endpoint is specifically designed for superadmins to view all products
     * in the system regardless of approval status or active status, with extensive
     * filtering options.
     * 
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function superadminProducts(Request $request): JsonResponse
    {
        try {
            // Add logging at the beginning
            Log::info('SuperAdmin product listing request', [
                'user_id' => $request->user()->id,
                'filters' => $request->except(['page']),
                'ip' => $request->ip(),
                'timestamp' => now()->toIso8601String()
            ]);

            // Check if user is superadmin
            if (!$request->user()->hasRole('superadmin')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized. Superadmin access required.',
                    'meta' => [
                        'code' => 403,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 403);
            }

            // Start building the query - no caching for real-time data
            $query = Product::with([
                'brand', 
                'vendor', 
                'categories', 
                'images', 
                'packPrice', 
                'bulkPrices', 
                'currency',
                'approver'
            ]);

            // Filter by vendor
            if ($request->has('vendor_id')) {
                $vendorId = $request->input('vendor_id');
                if (is_numeric($vendorId)) {
                    $query->where('vendor_id', (int)$vendorId);
                }
            }

            // Filter by vendor name (partial match)
            if ($request->has('vendor_name')) {
                $vendorName = $request->input('vendor_name');
                $query->whereHas('vendor', function ($q) use ($vendorName) {
                    $q->where('name', 'like', "%{$vendorName}%");
                });
            }

            // Filter by brand
            if ($request->has('brand_id')) {
                $query->where('brand_id', $request->input('brand_id'));
            }

            // Filter by brand name (partial match)
            if ($request->has('brand_name')) {
                $brandName = $request->input('brand_name');
                $query->whereHas('brand', function ($q) use ($brandName) {
                    $q->where('name', 'like', "%{$brandName}%");
                });
            }

            // Filter by category
            if ($request->has('category_id')) {
                $categoryId = $request->input('category_id');
                $query->whereHas('categories', function ($q) use ($categoryId) {
                    $q->where('categories.id', $categoryId);
                });
            }

            // Filter by approval status
            if ($request->has('approval_status')) {
                $query->where('approval_status', $request->input('approval_status'));
            }

            // Filter by active status
            if ($request->has('is_active')) {
                $query->where('is_active', $request->input('is_active'));
            }

            // Filter by featured status
            if ($request->has('is_featured')) {
                $query->where('is_featured', $request->input('is_featured'));
            }

            // Filter by bestseller status
            if ($request->has('is_bestseller')) {
                $query->where('is_bestseller', $request->input('is_bestseller'));
            }

            // Filter by trending status
            if ($request->has('is_trending')) {
                $query->where('is_trending', $request->input('is_trending'));
            }

            // Filter by price range
            if ($request->has('min_price') && $request->has('max_price')) {
                $query->whereHas('packPrice', function ($q) use ($request) {
                    $q->whereBetween('per_pack_price', [
                        $request->input('min_price'),
                        $request->input('max_price')
                    ]);
                });
            }

            // Filter by stock status
            if ($request->has('stock_status')) {
                $query->where('stock_status', $request->input('stock_status'));
            }

            // Filter by quantity range
            if ($request->has('min_quantity') && $request->has('max_quantity')) {
                $query->whereBetween('quantity', [
                    $request->input('min_quantity'),
                    $request->input('max_quantity')
                ]);
            }

            // Filter by min order quantity
            if ($request->has('min_order_quantity')) {
                $query->where('min_order_quantity', $request->input('min_order_quantity'));
            }

            // Filter by min order value
            if ($request->has('min_order_value')) {
                $query->where('min_order_value', $request->input('min_order_value'));
            }

            // Sort by
            $sortBy = $request->input('sort_by', 'created_at');
            $sortDirection = $request->input('sort_direction', 'desc');
            $query->orderBy($sortBy, $sortDirection);

            // Pagination
            $perPage = $request->input('per_page', 10);
            $products = $query->paginate($perPage);

            // Transform the products using formatProductData
            $products->getCollection()->transform(function ($product) {
                return $this->formatProductData($product);
            });

            return response()->json([
                'success' => true,
                'data' => [
                    'products' => $products
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 200);

        } catch (\Exception $e) {
            // Log error and return error response
            Log::error('Error retrieving products for superadmin', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving products',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }

    /**
     * Helper method to get all descendant category IDs recursively.
     *
     * @param  \App\Models\Category  $category
     * @return array
     */
    private function getAllDescendantCategoryIds($category)
    {
        $ids = [];
        
        foreach ($category->children as $child) {
            $ids[] = $child->id;
            $ids = array_merge($ids, $this->getAllDescendantCategoryIds($child));
        }
        
        return $ids;
    }

    /**
 * Update product brand and categories for superadmin.
 *
 * @param  \Illuminate\Http\Request  $request
 * @param  int  $id
 * @return \Illuminate\Http\JsonResponse
 */
public function updateProductMetadata(Request $request, $id)
{
    try {
        // Validate request
        $validator = Validator::make($request->all(), [
            'brand_id' => 'sometimes|required|exists:brands,id',
            'category_ids' => 'sometimes|required|array',
            'category_ids.*' => 'exists:categories,id'
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
                'meta' => [
                    'code' => 422,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 422);
        }
        
        // Find the product
        $product = Product::findOrFail($id);
        
        // Begin transaction
        DB::beginTransaction();
        
        try {
            // Update brand if provided
            if ($request->has('brand_id')) {
                $product->brand_id = $request->brand_id;
                $product->save();
            }
            
            // Update categories if provided
            if ($request->has('category_ids')) {
                $product->categories()->sync($request->category_ids);
            }
            
            // Commit transaction
            DB::commit();
            
            // Return response with updated product
            return response()->json([
                'success' => true,
                'message' => 'Product metadata updated successfully',
                'data' => [
                    'product' => $product->fresh(['brand', 'categories'])
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
        
    } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Product not found',
            'meta' => [
                'code' => 404,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 404);
    } catch (\Exception $e) {
        Log::error('Error updating product metadata', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
            'product_id' => $id
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Error updating product metadata',
            'error' => $e->getMessage(),
            'meta' => [
                'code' => 500,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 500);
    }
}

/**
 * Get product metadata (brands and categories) for superadmin.
 *
 * @param int $id
 * @return \Illuminate\Http\JsonResponse
 */
public function getProductMetadata($id)
{
    try {
        // Find the product with its brand, categories, and images
        $product = Product::with(['brand', 'categories', 'images'])
            ->findOrFail($id);
        
        // Format the response
        return response()->json([
            'success' => true,
            'data' => [
                'product_id' => $product->id,
                'product_name' => $product->name,
                'brand' => $product->brand,
                'categories' => $product->categories,
                'images' => $product->images->map(function($image) {
                    return [
                        'id' => $image->id,
                        'image_url' => $image->image_url,
                        'full_image_url' => $image->full_image_url,
                        'alt_text' => $image->alt_text,
                        'sort_order' => $image->sort_order
                    ];
                })
            ],
            'meta' => [
                'code' => 200,
                'timestamp' => now()->toIso8601String(),
            ],
        ]);
        
    } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Product not found',
            'meta' => [
                'code' => 404,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 404);
    } catch (\Exception $e) {
        Log::error('Error retrieving product metadata', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
            'product_id' => $id
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Error retrieving product metadata',
            'error' => $e->getMessage(),
            'meta' => [
                'code' => 500,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 500);
    }
}

/**
 * Update product categories for superadmin.
 *
 * @param  \Illuminate\Http\Request  $request
 * @param  int  $id
 * @return \Illuminate\Http\JsonResponse
 */
public function updateProductCategories(Request $request, $id)
{
    try {
        // Validate that category_ids is present and is an array (can be empty)
        $validator = Validator::make($request->all(), [
            'category_ids' => 'required|array',
            'category_ids.*' => 'exists:categories,id'
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
                'meta' => [
                    'code' => 422,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 422);
        }
        
        $product = Product::findOrFail($id);
        
        // If empty, assign to General Products (id = 1)
        $categoryIds = $request->category_ids;
        if (empty($categoryIds)) {
            $categoryIds = [1]; // 1 = General Products
        }

        $product->categories()->sync($categoryIds);
        
        return response()->json([
            'success' => true,
            'message' => 'Product categories updated successfully',
            'data' => [
                'product' => $product->fresh(['categories'])
            ],
            'meta' => [
                'code' => 200,
                'timestamp' => now()->toIso8601String(),
            ],
        ]);
    } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Product not found',
            'meta' => [
                'code' => 404,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 404);
    } catch (\Exception $e) {
        Log::error('Error updating product categories', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
            'product_id' => $id
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Error updating product categories',
            'error' => $e->getMessage(),
            'meta' => [
                'code' => 500,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 500);
    }
}

/**
 * Update product brand for superadmin.
 *
 * @param  \Illuminate\Http\Request  $request
 * @param  int  $id
 * @return \Illuminate\Http\JsonResponse
 */
public function updateProductBrand(Request $request, $id)
{
    try {
        // Validate request
        $validator = Validator::make($request->all(), [
            'brand_id' => 'required|exists:brands,id'
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
                'meta' => [
                    'code' => 422,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 422);
        }
        
        // Find the product
        $product = Product::findOrFail($id);
        
        // Update the brand
        $product->brand_id = $request->brand_id;
        $product->save();
        
        // Return response with updated product
        return response()->json([
            'success' => true,
            'message' => 'Product brand updated successfully',
            'data' => [
                'product' => $product->fresh(['brand'])
            ],
            'meta' => [
                'code' => 200,
                'timestamp' => now()->toIso8601String(),
            ],
        ]);
        
    } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Product not found',
            'meta' => [
                'code' => 404,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 404);
    } catch (\Exception $e) {
        Log::error('Error updating product brand', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
            'product_id' => $id
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Error updating product brand',
            'error' => $e->getMessage(),
            'meta' => [
                'code' => 500,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 500);
    }
}

/**
 * Get all products under a specific brand with pagination.
 *
 * @param int $brandId
 * @param Request $request
 * @return JsonResponse
 */
public function getProductsByBrand(int $brandId, Request $request): JsonResponse
{
    try {
        // First verify the brand exists
        $brand = Brand::findOrFail($brandId);
        
        // Get query parameters for filtering/sorting
        $perPage = $request->input('per_page', 12);
        $sortBy = $request->input('sort_by', 'created_at');
        $sortDirection = $request->input('sort_direction', 'desc');
        
        // Start building the query
        $query = Product::where('brand_id', $brandId)
            ->where('approval_status', 'approved')
            ->where('is_active', true)
            ->with([
                'categories',
                'images',
                'packPrice',
                'bulkPrices',
                'currency',
                'brand'
            ]);
        
        // Search filter
        if ($request->has('search')) {
            $search = $request->input('search');
            $query->search($search);
        }
        
        // Price range filtering
        if ($request->has('min_price') && $request->has('max_price')) {
            $query->byPriceRange($request->input('min_price'), $request->input('max_price'));
        }
        
        // Apply sorting
        $query->orderBy($sortBy, $sortDirection);
        
        $products = $query->paginate($perPage);
        
        // Transform the data to include all necessary fields
        $products->getCollection()->transform(function ($product) {
            return [
                'id' => $product->id,
                'name' => $product->name,
                'slug' => $product->slug,
                'description' => $product->description,
                'brand_id' => $product->brand_id,
                'brand' => $product->brand,
                'categories' => $product->categories,
                'images' => $product->images,
                'pack_price' => $product->packPrice,
                'bulk_prices' => $product->bulkPrices,
                'currency' => $product->currency,
                'customer_margin' => $product->default_customer_margin,
                'meta_title' => $product->meta_title,
                'meta_description' => $product->meta_description,
                'meta_keywords' => $product->meta_keywords,
                'min_order_quantity' => $product->min_order_quantity,
                'min_order_value' => $product->min_order_value,
                'free_shipping_threshold' => $product->free_shipping_threshold,
                'products_per_pallet' => $product->products_per_pallet,
                'pallet_delivery_fee' => $product->pallet_delivery_fee,
                'default_partner_margin' => $product->default_partner_margin,
                'default_customer_margin_type' => $product->default_customer_margin_type,
                'default_partner_margin_type' => $product->default_partner_margin_type,
                'approval_status' => $product->approval_status,
                'is_active' => $product->is_active,
                'is_approved' => $product->is_approved,
                'created_at' => $product->created_at,
                'updated_at' => $product->updated_at,
            ];
        });
        
        return response()->json([
            'success' => true,
            'message' => 'Products retrieved successfully',
            'data' => [
                'brand' => [
                    'id' => $brand->id,
                    'name' => $brand->name,
                    'slug' => $brand->slug,
                    'description' => $brand->description,
                    'logo' => $brand->logo,
                ],
                'products' => $products->items(),
            ],
            'meta' => [
                'code' => 200,
                'timestamp' => now()->toIso8601String(),
                'pagination' => [
                    'total' => $products->total(),
                    'per_page' => $products->perPage(),
                    'current_page' => $products->currentPage(),
                    'last_page' => $products->lastPage(),
                    'from' => $products->firstItem(),
                    'to' => $products->lastItem(),
                ],
            ],
        ], 200);
        
    } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Brand not found',
            'meta' => [
                'code' => 404,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 404);
    } catch (\Exception $e) {
        Log::error('Error retrieving products by brand', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
            'brand_id' => $brandId
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Error retrieving products',
            'error' => $e->getMessage(),
            'meta' => [
                'code' => 500,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 500);
    }
}

/**
 * Get all products under a specific category with pagination.
 *
 * @param int $categoryId
 * @param Request $request
 * @return JsonResponse
 */
public function getProductsByCategory(int $categoryId, Request $request): JsonResponse
{
    try {
        $perPage = $request->input('per_page', 12);
        $sortBy = $request->input('sort_by', 'created_at');
        $sortDirection = $request->input('sort_direction', 'desc');
        $includeSubcategories = $request->boolean('include_subcategories', false);

            $query = Product::with(['brand', 'images', 'packPrice', 'currency'])
            ->whereHas('categories', function ($q) use ($categoryId, $includeSubcategories) {
                if ($includeSubcategories) {
                    $allCategoryIds = $this->getAllCategoryAndDescendantIds($categoryId);
                    $q->whereIn('categories.id', $allCategoryIds);
                } else {
                    $q->where('categories.id', $categoryId);
                }
            });

            // Apply other filters
            if ($request->has('search')) {
                $query->search($request->input('search'));
            }

            if ($request->has('brand_id')) {
                $query->where('brand_id', $request->input('brand_id'));
            }

            if ($request->has('min_price') && $request->has('max_price')) {
                $query->byPriceRange($request->input('min_price'), $request->input('max_price'));
            }

            // Apply sorting
            $query->orderBy($sortBy, $sortDirection);

        $products = $query->paginate($perPage);

        // Transform the data to include only necessary fields
        $products->getCollection()->transform(function ($product) {
            return $this->getPublicProductData($product);
        });

        return response()->json([
            'success' => true,
            'data' => [
                'products' => $products
            ]
        ]);
    } catch (\Exception $e) {
        Log::error('Error fetching category products', ['error' => $e->getMessage()]);
        return response()->json([
            'success' => false,
            'message' => 'Error fetching category products',
            'error' => $e->getMessage()
        ], 500);
    }
}

/**
 * Helper method to get a category and all its descendant category IDs.
 *
 * @param int $categoryId
 * @return array
 */
private function getAllCategoryAndDescendantIds($categoryId)
{
    $allCategoryIds = [$categoryId]; // Start with the current category
    
    $category = Category::find($categoryId);
    if ($category) {
        // Get all descendants
        $descendantIds = $this->getAllDescendantCategoryIds($category);
        $allCategoryIds = array_merge($allCategoryIds, $descendantIds);
    }
    
    return $allCategoryIds;
}

    /**
     * Get detailed product information.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function getProductDetails(int $id): JsonResponse
    {
        Log::info('Request received for retrieving product', ['product_id' => $id]);

        // Updated relationships to include currency and categories
        $product = Product::with(['brand', 'categories', 'images', 'packPrice', 'bulkPrices', 'currency'])
            ->findOrFail($id);

        return response()->json([
            'success' => true,
            'message' => 'Product retrieved successfully',
            'data' => ['product' => $product],
            'meta' => [
                'code' => 200,
                'timestamp' => now()->toIso8601String(),
            ],
        ], 200);
    }

    public function publicProductViewAll(Request $request): JsonResponse
    {
        try {
            $perPage = $request->input('per_page', 12);
            $sortBy = $request->input('sort_by', 'created_at');
            $sortDirection = $request->input('sort_direction', 'desc');

            $query = Product::with([
                'categories:id,name,slug,description',
                'brand:id,name,slug,description,logo',
                'images:id,product_id,image_url,alt_text,sort_order',
                'packPrice:id,product_id,number_of_products,per_pack_price,per_pack_special_price,customer_margin,partner_margin,customer_margin_type,partner_margin_type,delivery_fee',
                'bulkPrices:id,product_id,number_of_packs,per_pack_price,per_pack_special_price,customer_margin,partner_margin,customer_margin_type,partner_margin_type,delivery_fee,sort_order,tax_rate',
                'currency:id,code,name,symbol,exchange_rate',
                'variantStocks:id,product_id,variant_attributes,quantity,stock_status'
            ])
            ->select([
                'id', 'name', 'slug', 'description', 'sku', 'quantity', 'stock_status',
                'brand_id', 'approval_status', 'is_active', 'is_approved',
                'meta_title', 'meta_description', 'meta_keywords',
                'min_order_quantity', 'min_order_value', 'free_shipping_threshold',
                'products_per_pallet', 'pallet_delivery_fee',
                'default_customer_margin', 'default_partner_margin',
                'default_customer_margin_type', 'default_partner_margin_type',
                'tax_rate'
            ])
            ->where('approval_status', 'approved')
            ->where('is_active', true);

            // Apply filters
            if ($request->has('search')) {
                $query->search($request->input('search'));
            }

            if ($request->has('brand_id')) {
                $query->where('brand_id', $request->input('brand_id'));
            }

            if ($request->has('min_price') && $request->has('max_price')) {
                $query->byPriceRange($request->input('min_price'), $request->input('max_price'));
            }

            // Apply sorting
            $query->orderBy($sortBy, $sortDirection);

            $products = $query->paginate($perPage);

            // Transform the data to include only necessary fields
            $products->getCollection()->transform(function ($product) {
                return $this->getPublicProductData($product);
            });

            return response()->json([
                'success' => true,
                'data' => [
                    'products' => $products
                ]
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in publicProductViewAll: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving products',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified product regardless of its status (for admins and superadmins).
     *
     * @param int $id
     * @return JsonResponse
     */
    public function showAnyStatus(int $id): JsonResponse
    {
        try {
            // Log authentication details
            $user = auth()->user();
            Log::info('Admin/Superadmin accessing product with any status', [
                'user_id' => $user ? $user->id : 'not authenticated',
                'user_email' => $user ? $user->email : 'not authenticated',
                'is_authenticated' => auth()->check(),
                'token_present' => request()->bearerToken() ? 'yes' : 'no',
                'product_id' => $id
            ]);

            // Find the product without any status conditions
            $product = Product::with([
                'brand',
                'categories',
                'images',
                'packPrice',
                'bulkPrices',
                'currency',
                'vendor:id,name,email',
                'approver:id,name,email'
            ])->find($id);

            if (!$product) {
                Log::info('Product not found', ['product_id' => $id]);
                return response()->json([
                    'success' => false,
                    'message' => 'Product not found'
                ], 404);
            }

            // Return detailed product information including status
            return response()->json([
                'success' => true,
                'data' => [
                    'product' => [
                        'id' => $product->id,
                        'name' => $product->name,
                        'slug' => $product->slug,
                        'description' => $product->description,
                        'sku' => $product->sku,
                        'quantity' => $product->quantity,
                        'stock_status' => $product->stock_status,
                        'meta' => $product->meta ?? [
                            'title' => $product->meta_title,
                            'description' => $product->meta_description,
                            'keywords' => $product->meta_keywords,
                        ],
                        'features' => $product->features,
                        'min_order_quantity' => $product->min_order_quantity,
                        'min_order_value' => $product->min_order_value,
                        'free_shipping_threshold' => $product->free_shipping_threshold,
                        'tax_rate' => $product->getTaxRate(),
                        'brand' => $product->brand ? [
                            'id' => $product->brand->id,
                            'name' => $product->brand->name,
                            'slug' => $product->brand->slug,
                            'description' => $product->brand->description,
                            'logo' => $product->brand->logo,
                        ] : null,
                        'categories' => $product->categories->map(function ($category) {
                            return [
                                'id' => $category->id,
                                'name' => $category->name,
                                'slug' => $category->slug,
                                'description' => $category->description,
                            ];
                        }),
                        'images' => $product->images->map(function ($image) {
                            return [
                                'id' => $image->id,
                                'url' => $image->image_url,
                                'alt_text' => $image->alt_text,
                                'sort_order' => $image->sort_order,
                            ];
                        }),
                        'pack_price' => $product->packPrice ? [
                            'number_of_products' => $product->packPrice->number_of_products,
                            'per_pack_price' => $product->packPrice->per_pack_price,
                            'per_pack_special_price' => $product->packPrice->per_pack_special_price,
                            'customer_margin' => $product->packPrice->customer_margin,
                            'partner_margin' => $product->packPrice->partner_margin,
                            'customer_margin_type' => $product->packPrice->customer_margin_type,
                            'partner_margin_type' => $product->packPrice->partner_margin_type,
                            'delivery_fee' => $product->packPrice->delivery_fee,
                        ] : null,
                        'bulk_prices' => $product->bulkPrices->map(function ($price) {
                            return [
                                'number_of_packs' => $price->number_of_packs,
                                'per_pack_price' => $price->per_pack_price,
                                'per_pack_special_price' => $price->per_pack_special_price,
                                'customer_margin' => $price->customer_margin,
                                'partner_margin' => $price->partner_margin,
                                'customer_margin_type' => $price->customer_margin_type,
                                'partner_margin_type' => $price->partner_margin_type,
                                'delivery_fee' => $price->delivery_fee,
                                'sort_order' => $price->sort_order,
                            ];
                        }),
                        'currency' => $product->currency ? [
                            'id' => $product->currency->id,
                            'code' => $product->currency->code,
                            'name' => $product->currency->name,
                            'symbol' => $product->currency->symbol,
                            'exchange_rate' => $product->currency->exchange_rate,
                        ] : null,
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching product with any status', [
                'error' => $e->getMessage(),
                'product_id' => $id,
                'trace' => $e->getTraceAsString(),
                'user_id' => auth()->id(),
                'is_authenticated' => auth()->check()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error fetching product',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update any product attribute (admin/superadmin only).
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function updateAnyAttribute(Request $request, int $id): JsonResponse
    {
        // Debug: Log user info and request
        Log::info('updateAnyAttribute called', [
            'user_id' => $request->user()->id,
            'user_email' => $request->user()->email,
            'user_role' => $request->user()->role ? $request->user()->role->slug : null,
            'request_data' => $request->all(),
            'product_id' => $id
        ]);
        try {
            // Check if user is admin or superadmin
            if (!$request->user()->hasRole(['superadmin', 'admin'])) {
                Log::warning('Unauthorized updateAnyAttribute attempt', [
                    'user_id' => $request->user()->id,
                    'user_role' => $request->user()->role ? $request->user()->role->slug : null,
                    'product_id' => $id
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized. Admin or Superadmin access required.',
                    'meta' => [
                        'code' => 403,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 403);
            }

            // Find the product
            $product = Product::findOrFail($id);

            // Get all fillable fields from the Product model
            $fillableFields = $product->getFillable();
            // Define image fields that should be accepted regardless of fillable status
            $imageFields = ['images', 'image', 'product_images', 'product_image', 'photos', 'photo'];
            // Combine fillable fields with image fields for validation
            $allAcceptedFields = array_merge($fillableFields, $imageFields);
            
            Log::debug('Product fillable fields', [
                'fillable' => $fillableFields,
                'image_fields' => $imageFields,
                'all_accepted_fields' => $allAcceptedFields
            ]);

            // Debug: Log the request data for validation
            Log::debug('Validation request data', [
                'request_all' => $request->all(),
                'attributes' => $request->input('attributes'),
                'values' => $request->input('values'),
                'fillable_fields' => $fillableFields,
                'attributes_count' => count($request->input('attributes', [])),
                'values_count' => count($request->input('values', [])),
                'accepted_image_fields' => ['images', 'image', 'product_images', 'product_image', 'photos', 'photo']
            ]);

            // Validate the request: expect 'attributes' and 'values' arrays
            $validator = Validator::make($request->all(), [
                'attributes' => 'required|array|min:1',
                'attributes.*' => 'required|string',
                'values' => 'required|array|min:1',
                'values.*' => 'required',
            ], [
                'attributes.required' => 'The attributes array is required.',
                'attributes.array' => 'The attributes must be an array.',
                'attributes.min' => 'At least one attribute must be provided.',
                'attributes.*.required' => 'Each attribute must be a string.',
                'attributes.*.string' => 'Each attribute must be a string.',
                'values.required' => 'The values array is required.',
                'values.array' => 'The values must be an array.',
                'values.min' => 'At least one value must be provided.',
                'values.*.required' => 'Each value is required.',
            ]);

            // Custom validation for attributes - more flexible for images
            $attributes = $request->input('attributes');
            $values = $request->input('values');
            $validationErrors = [];
            
            // Check if arrays have the same length
            if (count($attributes) !== count($values)) {
                $validationErrors['arrays'] = [
                    "The attributes and values arrays must have the same length. " .
                    "Attributes count: " . count($attributes) . ", Values count: " . count($values)
                ];
            }
            
            foreach ($attributes as $index => $attribute) {
                if (in_array(strtolower($attribute), array_map('strtolower', $imageFields))) {
                    // For image fields, validate that the value is an array
                    if (!is_array($values[$index] ?? null)) {
                        $validationErrors["values.{$index}"] = [
                            "The {$attribute} field must be an array of image URLs. " .
                            "Received: " . gettype($values[$index] ?? null) . 
                            ($values[$index] ? " ('" . (is_string($values[$index]) ? $values[$index] : json_encode($values[$index])) . "')" : " (null)")
                        ];
                    }
                } elseif (!in_array($attribute, $allAcceptedFields)) {
                    // For non-image fields, check against all accepted fields
                    $validationErrors["attributes.{$index}"] = [
                        "The attribute '{$attribute}' is not valid. " .
                        "Accepted fields: " . implode(', ', $fillableFields) . ". " .
                        "For images, use: " . implode(', ', $imageFields)
                    ];
                }
            }
            
            // Add custom validation errors if any
            if (!empty($validationErrors)) {
                foreach ($validationErrors as $field => $errors) {
                    foreach ($errors as $error) {
                        $validator->errors()->add($field, $error);
                    }
                }
            }

            if ($validator->fails()) {
                Log::warning('Validation failed in updateAnyAttribute', [
                    'errors' => $validator->errors(),
                    'user_id' => $request->user()->id,
                    'product_id' => $id
                ]);
                
                // Enhanced error response with detailed information
                $detailedErrors = [];
                $errorMessages = [];
                
                foreach ($validator->errors()->toArray() as $field => $errors) {
                    $detailedErrors[$field] = $errors;
                    
                    // Add helpful context for each error
                    foreach ($errors as $error) {
                        if (str_contains($field, 'attributes.')) {
                            $index = str_replace('attributes.', '', $field);
                            $attribute = $request->input("attributes.{$index}");
                            $errorMessages[] = "Attribute '{$attribute}' at position {$index}: {$error}";
                        } elseif (str_contains($field, 'values.')) {
                            $index = str_replace('values.', '', $field);
                            $attribute = $request->input("attributes.{$index}");
                            $errorMessages[] = "Value for '{$attribute}' at position {$index}: {$error}";
                        } else {
                            $errorMessages[] = "{$field}: {$error}";
                        }
                    }
                }
                
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed. Please check the details below.',
                    'errors' => $detailedErrors,
                    'error_summary' => $errorMessages,
                    'help' => [
                        'accepted_image_fields' => $imageFields,
                        'example_payload' => [
                            'attributes' => ['images'],
                            'values' => [
                                [
                                    'storage/uploads/images/image1.jpg',
                                    'storage/uploads/images/image2.jpg'
                                ]
                            ]
                        ],
                        'fillable_fields' => $fillableFields,
                        'all_accepted_fields' => $allAcceptedFields
                    ],
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],  
                ], 422);
            }

            DB::beginTransaction();

            try {
                $attributes = $request->input('attributes');
                $values = $request->input('values');

                $updates = [];
                $imageUrls = null;
                
                // Define image fields for processing
                $processingImageFields = ['images', 'image', 'product_images', 'product_image', 'photos', 'photo'];
                
                Log::info('Starting attribute processing', [
                    'attributes' => $attributes,
                    'values' => $values,
                    'processing_image_fields' => $processingImageFields
                ]);
                
                foreach ($attributes as $index => $attribute) {
                    Log::info('Processing attribute', [
                        'index' => $index,
                        'attribute' => $attribute,
                        'value' => $values[$index],
                        'is_image_field' => in_array(strtolower($attribute), array_map('strtolower', $processingImageFields))
                    ]);
                    
                    // Handle various image field names
                    if (in_array(strtolower($attribute), array_map('strtolower', $processingImageFields))) {
                        $imageUrls = $values[$index];
                        Log::info('Processing image field', [
                            'attribute' => $attribute,
                            'image_urls' => $imageUrls
                        ]);
                    } else {
                        $updates[$attribute] = $values[$index];
                        Log::info('Processing regular field', [
                            'attribute' => $attribute,
                            'value' => $values[$index]
                        ]);
                    }
                }

                Log::info('Prepared updates for product', [
                    'product_id' => $id,
                    'updates' => $updates,
                    'image_urls' => $imageUrls,
                    'updates_count' => count($updates),
                    'updates_keys' => array_keys($updates),
                    'will_update_product' => !empty($updates),
                    'processing_image_fields' => $processingImageFields,
                    'tax_rate_in_updates' => isset($updates['tax_rate']),
                    'tax_rate_value' => $updates['tax_rate'] ?? 'NOT_SET'
                ]);

                // Special handling for certain fields
                if (isset($updates['name'])) {
                    // Generate new slug if name is being updated
                    $slug = Str::slug($updates['name']);
                    $slugCount = Product::where('slug', 'LIKE', $slug . '%')
                        ->where('id', '!=', $id)
                        ->count();
                    if ($slugCount > 0) {
                        $slug = $slug . '-' . ($slugCount + 1);
                    }
                    $updates['slug'] = $slug;
                }

                // Update stock status if quantity is being updated
                if (isset($updates['quantity'])) {
                    $updates['stock_status'] = $updates['quantity'] > 0 ? 
                        ($updates['quantity'] <= 5 ? 'low_stock' : 'in_stock') : 
                        'out_of_stock';
                }

                // Double-check: Remove any image fields from updates array
                $finalUpdates = [];
                foreach ($updates as $key => $value) {
                    if (!in_array(strtolower($key), array_map('strtolower', $processingImageFields))) {
                        $finalUpdates[$key] = $value;
                        Log::info('Adding field to final updates', [
                            'field' => $key,
                            'value' => $value
                        ]);
                    } else {
                        Log::warning('Removing image field from updates array', [
                            'field' => $key,
                            'value' => $value
                        ]);
                    }
                }
                
                // Update the product only if there are valid updates
                if (!empty($finalUpdates)) {
                    $product->update($finalUpdates);
                    Log::info('Product updated successfully', [
                        'product_id' => $id,
                        'updates' => $finalUpdates,
                        'user_id' => $request->user()->id
                    ]);
                    
                    // Debug: Check if tax_rate was actually updated
                    $product->refresh();
                    Log::info('After update check', [
                        'product_id' => $id,
                        'tax_rate_after_update' => $product->tax_rate,
                        'updated_at' => $product->updated_at
                    ]);
                } else {
                    Log::info('No product fields to update, only processing images', [
                        'product_id' => $id,
                        'user_id' => $request->user()->id
                    ]);
                }

                // Clear all product-related caches
                $this->clearModelCache($product);
                Cache::forget('product_' . $product->id);
                Cache::forget('products_list_*');
                Cache::forget('public_products_list_*');
                Cache::forget('vendor_products_list_*');
                Cache::forget('admin_products_list_*');
                Cache::forget('superadmin_products_list_*');

                DB::commit();

                // Load the updated product with all relationships
                $product->refresh();
                $product->load(['brand', 'categories', 'images', 'packPrice', 'bulkPrices', 'currency', 'variantStocks']);
                
                // Debug: Log the final product state
                Log::info('Final product state after update', [
                    'product_id' => $id,
                    'tax_rate' => $product->tax_rate,
                    'updated_at' => $product->updated_at,
                    'all_attributes' => $product->toArray()
                ]);

                // Update images if provided
                if ($imageUrls && is_array($imageUrls)) {
                    // Delete existing images
                    $product->images()->delete();
                    // Add new images
                    foreach ($imageUrls as $index => $imageUrl) {
                        if ($imageUrl) {
                            ProductImage::create([
                                'product_id' => $product->id,
                                'image_url' => $imageUrl,
                                'alt_text' => $product->name,
                                'sort_order' => $index,
                            ]);
                        }
                    }
                }

                return response()->json([
                    'success' => true,
                    'message' => 'Product attributes updated successfully',
                    'data' => [
                        'product' => $product
                    ],
                    'meta' => [
                        'code' => 200,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 200);

            } catch (\Exception $e) {
                DB::rollBack();
                Log::error('Exception during DB update in updateAnyAttribute', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                    'user_id' => $request->user()->id,
                    'product_id' => $id
                ]);
                throw $e;
            }

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            Log::warning('Product not found in updateAnyAttribute', [
                'product_id' => $id,
                'user_id' => $request->user()->id
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Product not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error updating product attributes', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'product_id' => $id,
                'user_id' => $request->user()->id
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error updating product attributes',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }

    /**
     * Test endpoint to show valid attributes for a product
     *
     * @param int $id
     * @return JsonResponse
     */
    public function testValidAttributes(int $id): JsonResponse
    {
        try {
            $product = Product::findOrFail($id);
            $fillableFields = $product->getFillable();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'product_id' => $id,
                    'valid_attributes' => $fillableFields,
                    'current_images' => $product->images->map(function($image) {
                        return [
                            'id' => $image->id,
                            'image_url' => $image->image_url,
                            'full_image_url' => $image->full_image_url,
                            'alt_text' => $image->alt_text,
                            'sort_order' => $image->sort_order
                        ];
                    }),
                    'example_payload' => [
                        'attributes' => ['images'],
                        'values' => [
                            [
                                'storage/uploads/images/image1.jpg',
                                'storage/uploads/images/image2.jpg'
                            ]
                        ]
                    ],
                    'validation_info' => [
                        'images_in_fillable' => in_array('images', $fillableFields),
                        'fillable_count' => count($fillableFields),
                        'accepted_image_fields' => ['images', 'image', 'product_images', 'product_image', 'photos', 'photo']
                    ]
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Permanently delete a product and all its related records.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function hardDelete(int $id): JsonResponse
    {
        try {
            $product = Product::findOrFail($id);
            $user = auth()->user();
            
            // Check authorization
            if (!$user->hasRole(['superadmin', 'admin']) && 
                ($user->hasRole('vendor') && (int)$user->id !== (int)$product->vendor_id)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized. You can only delete your own products.'
                ], 403);
            }

            DB::beginTransaction();
            
            // Delete all related records
            $product->images()->delete();
            $product->packPrice()->delete();
            $product->bulkPrices()->delete();
            $product->offers()->delete();
            $product->variantStocks()->delete();
            
            // Detach categories
            $product->categories()->detach();
            
            // Delete the product
            $product->delete();
            
            DB::commit();
            
            // Clear all product-related caches
            Cache::flush();
            
            return response()->json([
                'success' => true,
                'message' => 'Product permanently deleted successfully'
            ]);
            
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error deleting product',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update product sorting order in a category
     */
    public function updateProductSortOrder(Request $request, int $categoryId): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'product_orders' => 'required|array',
                'product_orders.*.product_id' => 'required|exists:products,id',
                'product_orders.*.sort_order' => 'required|integer|min:0'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Check if category exists
            $category = Category::findOrFail($categoryId);

            DB::beginTransaction();

            foreach ($request->product_orders as $order) {
                DB::table('product_categories')
                    ->where('category_id', $categoryId)
                    ->where('product_id', $order['product_id'])
                    ->update(['sort_order' => $order['sort_order']]);
            }

            DB::commit();

            // Clear category cache
            Cache::flush();

            return response()->json([
                'success' => true,
                'message' => 'Product sorting order updated successfully'
            ]);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found'
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error updating product sorting order',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get sorted products by category with all relevant information
     */
    public function getSortedProductsByCategory(int $categoryId): JsonResponse
    {
        try {
            $category = Category::findOrFail($categoryId);
            
            $products = $category->products()
                ->with([
                    'brand',
                    'images',
                    'packPrice',
                    'bulkPrices',
                    'activeOffers',
                    'variantStocks',
                    'currency'
                ])
                ->where('is_active', true)
                ->where('is_approved', true)
                ->orderByRaw('CASE WHEN product_categories.sort_order IS NULL OR product_categories.sort_order = 0 THEN 1 ELSE 0 END, product_categories.sort_order ASC, products.created_at DESC')
                ->get()
                ->map(function ($product) {
                    $formattedProduct = $this->formatProductData($product);
                    
                    // Add additional product information
                    $formattedProduct['pack_price'] = $product->packPrice ? [
                        'per_pack_price' => $product->packPrice->per_pack_price,
                        'per_pack_special_price' => $product->packPrice->per_pack_special_price,
                        'customer_margin' => $product->packPrice->customer_margin,
                        'partner_margin' => $product->packPrice->partner_margin,
                        'customer_margin_type' => $product->packPrice->customer_margin_type,
                        'partner_margin_type' => $product->packPrice->partner_margin_type,
                        'delivery_fee' => $product->packPrice->delivery_fee
                    ] : null;

                    $formattedProduct['bulk_prices'] = $product->bulkPrices->map(function ($bulkPrice) {
                        return [
                            'id' => $bulkPrice->id,
                            'number_of_packs' => $bulkPrice->number_of_packs,
                            'per_pack_price' => $bulkPrice->per_pack_price,
                            'per_pack_special_price' => $bulkPrice->per_pack_special_price,
                            'customer_margin' => $bulkPrice->customer_margin,
                            'partner_margin' => $bulkPrice->partner_margin,
                            'customer_margin_type' => $bulkPrice->customer_margin_type,
                            'partner_margin_type' => $bulkPrice->partner_margin_type,
                            'delivery_fee' => $bulkPrice->delivery_fee
                        ];
                    });

                    $formattedProduct['variant_stocks'] = $product->variantStocks->map(function ($stock) {
                        return [
                            'id' => $stock->id,
                            'variant_name' => $stock->variant_name,
                            'sku' => $stock->sku,
                            'quantity' => $stock->quantity,
                            'stock_status' => $stock->stock_status
                        ];
                    });

                    $formattedProduct['active_offers'] = $product->activeOffers->map(function ($offer) {
                        return [
                            'id' => $offer->id,
                            'name' => $offer->name,
                            'description' => $offer->description,
                            'price' => $offer->price,
                            'start_date' => $offer->start_date,
                            'end_date' => $offer->end_date
                        ];
                    });

                    $formattedProduct['shipping_info'] = [
                        'eligible_for_free_shipping' => $product->eligible_for_free_shipping,
                        'ships_from' => $product->ships_from,
                        'fulfilled_by' => $product->fulfilled_by,
                        'min_order_quantity' => $product->min_order_quantity,
                        'min_order_value' => $product->min_order_value,
                        'free_shipping_threshold' => $product->free_shipping_threshold,
                        'products_per_pallet' => $product->products_per_pallet,
                        'pallet_delivery_fee' => $product->pallet_delivery_fee
                    ];

                    $formattedProduct['tax_rate'] = $product->getTaxRate();

                    $formattedProduct['currency'] = $product->currency ? [
                        'code' => $product->currency->code,
                        'symbol' => $product->currency->symbol,
                        'name' => $product->currency->name
                    ] : null;

                    // Remove meta fields and timestamps
                    unset($formattedProduct['meta'], $formattedProduct['meta_title'], $formattedProduct['meta_description'], $formattedProduct['meta_keywords'], $formattedProduct['created_at'], $formattedProduct['updated_at']);

                    return $formattedProduct;
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'category' => [
                        'id' => $category->id,
                        'name' => $category->name,
                        'slug' => $category->slug,
                        'description' => $category->description,
                        'image' => $category->image
                    ],
                    'products' => $products
                ]
            ]);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching products',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update a single product's ranking in a category
     */
    public function updateProductRanking(Request $request, int $categoryId, int $productId): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'rank' => 'required|integer|min:0'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Check if category exists
            $category = Category::findOrFail($categoryId);
            
            // Check if product exists and belongs to the category
            $product = $category->products()->where('products.id', $productId)->firstOrFail();

            // Update the ranking
            DB::table('product_categories')
                ->where('category_id', $categoryId)
                ->where('product_id', $productId)
                ->update(['sort_order' => $request->rank]);

            // Clear category cache
            Cache::flush();

            return response()->json([
                'success' => true,
                'message' => 'Product ranking updated successfully',
                'data' => [
                    'product_id' => $productId,
                    'category_id' => $categoryId,
                    'rank' => $request->rank
                ]
            ]);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Category or product not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating product ranking',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update multiple products' rankings in a category
     */
    public function updateProductsRanking(Request $request, int $categoryId): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'rankings' => 'required|array',
                'rankings.*.product_id' => 'required|exists:products,id',
                'rankings.*.rank' => 'required|integer|min:0'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Check if category exists
            $category = Category::findOrFail($categoryId);

            DB::beginTransaction();

            foreach ($request->rankings as $ranking) {
                // Verify product belongs to category
                $exists = DB::table('product_categories')
                    ->where('category_id', $categoryId)
                    ->where('product_id', $ranking['product_id'])
                    ->exists();

                if (!$exists) {
                    throw new \Exception("Product ID {$ranking['product_id']} does not belong to this category");
                }

                // Update ranking
                DB::table('product_categories')
                    ->where('category_id', $categoryId)
                    ->where('product_id', $ranking['product_id'])
                    ->update(['sort_order' => $ranking['rank']]);
            }

            DB::commit();

            // Clear category cache
            Cache::flush();

            return response()->json([
                'success' => true,
                'message' => 'Products ranking updated successfully',
                'data' => [
                    'category_id' => $categoryId,
                    'updated_rankings' => $request->rankings
                ]
            ]);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found'
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error updating products ranking',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove a specific category from a product.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function removeProductCategory(Request $request, $id)
    {
        try {
            // Validate request
            $validator = Validator::make($request->all(), [
                'category_id' => 'required|exists:categories,id'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }
            
            // Find the product
            $product = Product::findOrFail($id);
            
            // Check authorization
            $user = auth()->user();
            if (!$user->hasRole(['superadmin', 'admin']) && 
                ($user->hasRole('vendor') && (int)$user->id !== (int)$product->vendor_id)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized. You can only remove categories from your own products.'
                ], 403);
            }
            
            // Remove the specific category
            $product->categories()->detach($request->category_id);
            
            // Return response with updated product
            return response()->json([
                'success' => true,
                'message' => 'Category removed from product successfully',
                'data' => [
                    'product' => $product->fresh(['categories'])
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error removing product category', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'product_id' => $id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error removing product category',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }

    /**
     * Bulk assign a brand to multiple products (superadmin/admin only).
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function bulkAssignBrand(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'brand_id' => 'required|exists:brands,id',
            'product_ids' => 'required|array|min:1',
            'product_ids.*' => 'exists:products,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        // Only allow for superadmin/admin
        if (!auth()->user()->hasRole(['superadmin', 'admin'])) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        Product::whereIn('id', $request->product_ids)->update(['brand_id' => $request->brand_id]);

        return response()->json([
            'success' => true,
            'message' => 'Brand assigned to selected products successfully'
        ]);
    }

    /**
     * De-assign (remove) the brand from a product (superadmin/admin only).
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function removeProductBrand($id)
    {
        try {
            // Only allow for superadmin/admin
            if (!auth()->user()->hasRole(['superadmin', 'admin'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized'
                ], 403);
            }

            $product = Product::findOrFail($id);
            $product->brand_id = null;
            $product->save();

            return response()->json([
                'success' => true,
                'message' => 'Brand de-assigned from product successfully',
                'data' => [
                    'product' => $product->fresh(['brand'])
                ]
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        } catch (\Exception $e) {
            \Log::error('Error de-assigning brand from product', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'product_id' => $id
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error de-assigning brand from product',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Public: Get products by vendor id (approved and active only).
     *
     * @param int $vendorId
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProductsByVendor($vendorId, Request $request)
    {
        $perPage = $request->input('per_page', 12);
        $sortBy = $request->input('sort_by', 'created_at');
        $sortDirection = $request->input('sort_direction', 'desc');

        $query = Product::with([
            'brand',
            'categories',
            'images',
            'packPrice',
            'bulkPrices',
            'currency'
        ])
        ->where('vendor_id', $vendorId)
        ->where('approval_status', 'approved')
        ->where('is_active', true)
        ->orderBy($sortBy, $sortDirection);

        $products = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => [
                'products' => $products
            ]
        ]);
    }

    /**
     * Update the features/specs JSON for a product (flexible attributes).
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateFeatures(Request $request, $id)
    {
        $product = Product::findOrFail($id);

        $data = $request->validate([
            'features' => 'required|array',
        ]);

        $product->features = $data['features'];
        $product->save();

        return response()->json([
            'success' => true,
            'features' => $product->features,
        ]);
    }

    /**
     * Get the features/specs JSON for a product (flexible attributes).
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function getFeatures($id)
    {
        $product = Product::findOrFail($id);

        return response()->json([
            'success' => true,
            'features' => $product->features,
        ]);
    }

    /**
     * Update the meta JSON for a product (Open Graph, Twitter, meta tags).
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateMeta(Request $request, $id)
    {
        $product = Product::findOrFail($id);

        $data = $request->validate([
            'meta' => 'required|array',
        ]);

        $product->meta = $data['meta'];
        $product->save();

        return response()->json([
            'success' => true,
            'meta' => $product->meta,
        ]);
    }

    /**
     * Get the meta JSON for a product (Open Graph, Twitter, meta tags).
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function getMeta($id)
    {
        $product = Product::findOrFail($id);

        return response()->json([
            'success' => true,
            'meta' => $product->meta,
        ]);
    }

    /**
     * Get formatted product data for public view
     */
    private function getPublicProductData($product)
    {
        return [
            'id' => $product->id,
            'name' => $product->name,
            'slug' => $product->slug,
            'sku' => $product->sku,
            'description' => $product->description,
            'price' => $product->current_price,
            'special_price' => $product->packPrice?->per_pack_special_price,
            'currency' => $product->currency?->code,
            'image' => $product->images->first()?->image_url,
            'brand' => $product->brand?->name,
            'rating' => $product->rating,
            'review_count' => $product->review_count,
            'is_active' => $product->is_active,
            'approval_status' => $product->approval_status,
            'meta' => $product->meta ?? [
                'title' => $product->meta_title,
                'description' => $product->meta_description,
                'keywords' => $product->meta_keywords,
            ],
            'min_order_quantity' => $product->min_order_quantity,
            'min_order_value' => $product->min_order_value,
            'free_shipping_threshold' => $product->free_shipping_threshold,
            'tax_rate' => $product->getTaxRate(),
            'pack_price' => $product->packPrice ? [
                'number_of_products' => $product->packPrice->number_of_products,
                'per_pack_price' => $product->packPrice->per_pack_price,
                'per_pack_special_price' => $product->packPrice->per_pack_special_price,
                'customer_margin' => $product->packPrice->customer_margin,
                'partner_margin' => $product->packPrice->partner_margin,
                'customer_margin_type' => $product->packPrice->customer_margin_type,
                'partner_margin_type' => $product->packPrice->partner_margin_type,
                'delivery_fee' => $product->packPrice->delivery_fee,
                'tax_rate' => $product->packPrice->getTaxRate()
            ] : null,
            'bulk_prices' => $product->bulkPrices->map(fn($price) => [
                'number_of_packs' => $price->number_of_packs,
                'per_pack_price' => $price->per_pack_price,
                'per_pack_special_price' => $price->per_pack_special_price,
                'customer_margin' => $price->customer_margin,
                'partner_margin' => $price->partner_margin,
                'customer_margin_type' => $price->customer_margin_type,
                'partner_margin_type' => $price->partner_margin_type,
                'delivery_fee' => $price->delivery_fee,
                'sort_order' => $price->sort_order,
                'tax_rate' => $price->getTaxRate()
            ]),
        ];
    }

    /**
     * Display the specified product for a vendor.
     * This method allows vendors to view their own products regardless of approval status.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function vendorShow(int $id): JsonResponse
    {
        try {
            $user = auth()->user();
            
            // Log the request details
            Log::info('Vendor retrieving product', [
                'product_id' => $id,
                'user_id' => $user->id,
                'user_role' => $user->role ? $user->role->slug : null
            ]);

            // First check if user is a vendor
            if (!$user->hasRole('vendor')) {
                Log::warning('User is not a vendor', [
                    'user_id' => $user->id,
                    'user_role' => $user->role ? $user->role->slug : null
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized. Only vendors can access this endpoint.'
                ], 403);
            }

            // Get the product with all necessary relationships
            $product = Product::with([
                'brand',
                'categories',
                'images',
                'packPrice',
                'bulkPrices',
                'currency',
                'variantStocks',
                'variantStocks.variantAttributes'
            ])->find($id);

            if (!$product) {
                Log::warning('Product not found', ['product_id' => $id]);
                return response()->json([
                    'success' => false,
                    'message' => 'Product not found'
                ], 404);
            }

            // Log product details for debugging
            Log::info('Product details', [
                'product_id' => $product->id,
                'product_vendor_id' => $product->vendor_id,
                'authenticated_user_id' => $user->id,
                'is_vendor' => $user->hasRole('vendor')
            ]);

            // Verify that the product belongs to the vendor
            if ((int)$product->vendor_id !== (int)$user->id) {
                Log::warning('Vendor authorization failed', [
                    'product_id' => $product->id,
                    'product_vendor_id' => $product->vendor_id,
                    'authenticated_user_id' => $user->id
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized. This product does not belong to you.'
                ], 403);
            }

            // Format and return the product data
            $productData = $this->formatProductData($product);

            return response()->json([
                'success' => true,
                'data' => [
                    'product' => $productData
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching vendor product', [
                'error' => $e->getMessage(),
                'product_id' => $id,
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error fetching product',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Vendor: Update any attribute of their own product.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function updateAnyAttributeAsVendor(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        $product = Product::findOrFail($id);
        if (!$user->hasRole('vendor') || ((int)$user->id !== (int)$product->vendor_id)) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Only the owning vendor can update this product.',
                'meta' => [
                    'code' => 403,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 403);
        }
        return $this->performAttributeUpdate($request, $product);
    }

    /**
     * Shared logic for updating product attributes (used by admin and vendor).
     *
     * @param Request $request
     * @param Product $product
     * @return JsonResponse
     */
    private function performAttributeUpdate(Request $request, $product): JsonResponse
    {
        $fillableFields = $product->getFillable();
        $imageFields = ['images', 'image', 'product_images', 'product_image', 'photos', 'photo'];
        $allAcceptedFields = array_merge($fillableFields, $imageFields);

        $validator = Validator::make($request->all(), [
            'attributes' => 'required|array|min:1',
            'attributes.*' => 'required|string',
            'values' => 'required|array|min:1',
            'values.*' => 'required',
        ]);

        $attributes = $request->input('attributes');
        $values = $request->input('values');
        $validationErrors = [];
        if (count($attributes) !== count($values)) {
            $validationErrors['arrays'] = [
                "The attributes and values arrays must have the same length. Attributes count: " . count($attributes) . ", Values count: " . count($values)
            ];
        }
        foreach ($attributes as $index => $attribute) {
            if (in_array(strtolower($attribute), array_map('strtolower', $imageFields))) {
                if (!is_array($values[$index] ?? null)) {
                    $validationErrors["values.{$index}"] = [
                        "The {$attribute} field must be an array of image URLs. Received: " . gettype($values[$index] ?? null)
                    ];
                }
            } elseif (!in_array($attribute, $allAcceptedFields)) {
                $validationErrors["attributes.{$index}"] = [
                    "The attribute '{$attribute}' is not valid. Accepted fields: " . implode(', ', $fillableFields) . ". For images, use: " . implode(', ', $imageFields)
                ];
            }
        }
        if (!empty($validationErrors)) {
            foreach ($validationErrors as $field => $errors) {
                foreach ($errors as $error) {
                    $validator->errors()->add($field, $error);
                }
            }
        }
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed. Please check the details below.',
                'errors' => $validator->errors(),
                'meta' => [
                    'code' => 422,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 422);
        }
        DB::beginTransaction();
        try {
            $imageUrls = null;
            $updates = [];
            foreach ($attributes as $index => $attribute) {
                $value = $values[$index];
                if (in_array(strtolower($attribute), array_map('strtolower', $imageFields))) {
                    $imageUrls = $value;
                } else {
                    $updates[$attribute] = $value;
                }
            }
            // Update product fields (excluding images)
            if (!empty($updates)) {
                $product->update($updates);
            }
            // Handle images as a related table
            if ($imageUrls && is_array($imageUrls)) {
                $product->images()->delete();
                foreach ($imageUrls as $index => $imageUrl) {
                    if ($imageUrl) {
                        \App\Models\ProductImage::create([
                            'product_id' => $product->id,
                            'image_url' => $imageUrl,
                            'alt_text' => $product->name,
                            'sort_order' => $index,
                        ]);
                    }
                }
            }
            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Product attributes updated successfully.',
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error updating product attributes.',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
}