<?php

namespace App\Http\Controllers;

use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Services\StripeService;
use App\Mail\PaymentFailed;
use Illuminate\Support\Facades\Mail;
use App\Models\Cart;

class OrderController extends Controller
{
    protected $stripeService;

    public function __construct(StripeService $stripeService)
    {
        $this->stripeService = $stripeService;
    }

    /**
     * Get customer's orders
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $userId = $request->user()->id;
            
            $query = Order::where('user_id', $userId)
                ->with(['items.product'])
                ->orderBy('created_at', 'desc');
            
            // Filter by status if provided
            if ($request->has('status')) {
                $query->where('status', $request->status);
            }
            
            $orders = $query->paginate(10);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'orders' => $orders
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving orders', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving orders',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get a specific order
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Request $request, $id)
    {
        try {
            $userId = $request->user()->id;
            
            $order = Order::where('user_id', $userId)
                ->with(['items.product', 'history'])
                ->findOrFail($id);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'order' => $order
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving order', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving order',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function processOrder(Request $request)
    {
        try {
            Log::info('Starting order processing', [
                'payment_intent_id' => $request->payment_intent_id,
                'cart_id' => $request->cart_id,
                'user_id' => auth()->id(),
                'payment_method' => $request->payment_method ? 'present' : 'missing'
            ]);

            // First, confirm the payment with the payment method
            Log::info('Attempting to confirm payment', [
                'payment_intent_id' => $request->payment_intent_id,
                'payment_method_type' => $request->payment_method['type'] ?? 'unknown'
            ]);

            $paymentConfirmation = $this->stripeService->confirmPaymentWithMethod(
                $request->payment_intent_id,
                $request->payment_method
            );

            Log::info('Payment confirmation result', [
                'success' => $paymentConfirmation['success'],
                'error' => $paymentConfirmation['error'] ?? null,
                'status' => $paymentConfirmation['status'] ?? null,
                'currency' => $paymentConfirmation['currency'] ?? null
            ]);

            if (!$paymentConfirmation['success']) {
                Log::error('Payment confirmation failed', [
                    'error' => $paymentConfirmation['error'],
                    'payment_intent_id' => $request->payment_intent_id,
                    'error_code' => $paymentConfirmation['error_code'] ?? null,
                    'http_status' => $paymentConfirmation['http_status'] ?? null
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Payment confirmation failed',
                    'error' => $paymentConfirmation['error']
                ], 400);
            }

            // Then verify the payment
            Log::info('Starting payment verification after confirmation', [
                'payment_intent_id' => $request->payment_intent_id
            ]);

            $paymentVerification = $this->stripeService->verifyPayment($request->payment_intent_id);
            
            Log::info('Payment verification result', [
                'success' => $paymentVerification['success'],
                'error' => $paymentVerification['error'] ?? null,
                'amount' => $paymentVerification['amount'] ?? null,
                'status' => $paymentVerification['status'] ?? null,
                'currency' => $paymentVerification['currency'] ?? null
            ]);

            if (!$paymentVerification['success']) {
                Log::info('Payment verification failed, preparing to send email');
                try {
                    // Send payment failed email
                    Mail::to($request->shipping_address['contact_email'])
                        ->send(new PaymentFailed($request->payment_intent_id));
                    Log::info('Payment failed email sent successfully');
                } catch (\Exception $e) {
                    Log::error('Failed to send payment failed email', [
                        'error' => $e->getMessage(),
                        'email' => $request->shipping_address['contact_email']
                    ]);
                }

                return response()->json([
                    'success' => false,
                    'message' => 'Payment verification failed',
                    'error' => $paymentVerification['error']
                ], 400);
            }

            // Get the cart
            $cart = Cart::findOrFail($request->cart_id);
            
            // Verify the payment amount matches the cart total
            $totalAmount = $cart->total_amount;
            $paymentAmount = $paymentVerification['amount'];
            $paymentCurrency = $paymentVerification['currency'];
            $cartCurrency = $cart->currency_code;

            Log::info('Verifying payment amount', [
                'cart_total' => $totalAmount,
                'cart_currency' => $cartCurrency,
                'payment_amount' => $paymentAmount,
                'payment_currency' => $paymentCurrency
            ]);

            // If currencies don't match, we need to handle the conversion
            if ($paymentCurrency !== $cartCurrency) {
                Log::info('Currency mismatch detected', [
                    'payment_currency' => $paymentCurrency,
                    'cart_currency' => $cartCurrency
                ]);
                
                // Here you would typically convert the payment amount to the cart currency
                // For now, we'll just log a warning and proceed
                Log::warning('Currency conversion not implemented', [
                    'payment_amount' => $paymentAmount,
                    'payment_currency' => $paymentCurrency,
                    'cart_amount' => $totalAmount,
                    'cart_currency' => $cartCurrency
                ]);
            }

            // Rest of your existing code...
            // ... existing code ...
        } catch (\Exception $e) {
            Log::error('Error processing order', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error processing order',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Superadmin: Create a manual order (and invoice) directly, bypassing the cart.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function manualCreate(Request $request)
    {
        $user = $request->user();
        if (!$user || !$user->hasRole('superadmin')) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Only superadmin can create manual orders.'
            ], 403);
        }

        $validated = $request->validate([
            'customer_id' => 'nullable|exists:users,id',
            'customer_name' => 'nullable|string|max:255',
            'customer_phone' => 'nullable|string|max:30',
            'customer_email' => 'nullable|email|max:255',
            'products' => 'required|array|min:1',
            'products.*.product_id' => 'nullable|exists:products,id',
            'products.*.name' => 'required_without:products.*.product_id|string|max:255',
            'products.*.quantity' => 'required|integer|min:1',
            'products.*.price' => 'nullable|numeric|min:0',
            'products.*.discount' => 'nullable|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'delivery_fee' => 'nullable|numeric|min:0',
            'order_discount' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        // 1. Determine customer
        $customerId = $validated['customer_id'] ?? null;
        if (!$customerId) {
            // Create or use a generic walk-in customer
            $customer = \App\Models\User::firstOrCreate(
                [
                    'email' => $validated['customer_email'] ?? 'walkin_' . uniqid() . '@example.com'
                ],
                [
                    'name' => $validated['customer_name'] ?? 'Walk-in Customer',
                    'phone' => $validated['customer_phone'] ?? null,
                    'role_id' => 5 // Assuming 5 is buyer/customer
                ]
            );
            $customerId = $customer->id;
        }

        // 2. Create the order
        $order = new \App\Models\Order();
        $order->user_id = $customerId;
        $order->status = 'manual';
        $order->payment_status = 'pending';
        $order->shipping_status = 'pending';
        $order->notes = $validated['notes'] ?? null;
        $order->save();

        $subtotal = 0;
        $orderItems = [];
        foreach ($validated['products'] as $item) {
            $productId = $item['product_id'] ?? null;
            $name = $item['name'] ?? null;
            $quantity = $item['quantity'];
            $price = isset($item['price']) ? $item['price'] : null;
            $discount = $item['discount'] ?? 0;

            if ($productId) {
                $product = \App\Models\Product::find($productId);
                $name = $name ?: $product->name;
                $price = $price ?? $product->price;
            } else {
                $product = null;
            }

            $lineTotal = ($price * $quantity) - $discount;
            $subtotal += $lineTotal;

            $orderItems[] = [
                'order_id' => $order->id,
                'product_id' => $productId,
                'name' => $name,
                'quantity' => $quantity,
                'unit_price' => $price,
                'discount' => $discount,
                'total' => $lineTotal
            ];
        }

        // 3. Insert order items
        foreach ($orderItems as $itemData) {
            $order->items()->create($itemData);
        }

        // 4. Calculate totals
        $tax = $validated['tax_amount'] ?? 0; // Tax will be calculated per item in the order items
        $delivery = $validated['delivery_fee'] ?? 0;
        $orderDiscount = $validated['order_discount'] ?? 0;
        $finalTotal = $subtotal + $tax + $delivery - $orderDiscount;

        $order->subtotal = $subtotal;
        $order->tax_amount = $tax;
        $order->shipping_amount = $delivery;
        $order->discount_amount = $orderDiscount;
        $order->total_amount = $finalTotal;
        $order->save();

        // 5. Optionally, create an invoice (if you have an Invoice model)
        if (class_exists('App\\Models\\Invoice')) {
            $invoice = new \App\Models\Invoice();
            $invoice->order_id = $order->id;
            $invoice->user_id = $customerId;
            $invoice->amount = $finalTotal;
            $invoice->status = 'pending';
            $invoice->save();
        }

        return response()->json([
            'success' => true,
            'message' => 'Manual order created successfully',
            'data' => [
                'order' => $order->load('items'),
                'invoice' => isset($invoice) ? $invoice : null
            ]
        ], 201);
    }
}