<?php

namespace App\Http\Controllers;

use App\Models\ManualInvoice;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ManualInvoiceController extends Controller
{
    /**
     * Superadmin: Create a manual invoice with products from DB and a payment link.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'products' => 'required|array|min:1',
            'products.*.product_id' => 'required|exists:products,id',
            'products.*.quantity' => 'required|integer|min:1',
            'products.*.price_override' => 'nullable|numeric|min:0',
            'products.*.discount' => 'nullable|numeric|min:0',
            'delivery_address' => 'required|array',
            'delivery_address.line1' => 'required|string',
            'delivery_address.line2' => 'nullable|string',
            'delivery_address.city' => 'required|string',
            'delivery_address.state' => 'required|string',
            'delivery_address.postal_code' => 'required|string',
            'delivery_address.country' => 'required|string',
            'delivery_address.name' => 'nullable|string',
            'delivery_address.phone' => 'nullable|string',
            'notes' => 'nullable|string',
            'due_date' => 'nullable|date',
            'currency' => 'nullable|string|size:3',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            $delivery = $request->input('delivery_address');
            $currency = $request->input('currency', 'INR');

            // Create manual order as anchor for items and totals
            $order = new Order();
            $order->user_id = auth()->id(); // creator as owner; no customer info required
            $order->status = 'manual';
            $order->payment_status = 'pending';
            $order->shipping_status = 'pending';
            $order->currency_code = $currency;
            $order->notes = $request->input('notes');
            // Use existing shipping_address/phone fields to store delivery address without schema changes
            $order->shipping_address = json_encode([
                'name' => $delivery['name'] ?? null,
                'line1' => $delivery['line1'],
                'line2' => $delivery['line2'] ?? null,
                'city' => $delivery['city'],
                'state' => $delivery['state'],
                'postal_code' => $delivery['postal_code'],
                'country' => $delivery['country'],
            ]);
            $order->shipping_phone = $delivery['phone'] ?? null;
            $order->save();

            $subtotal = 0;
            $totalDiscount = 0;
            $totalTax = 0; // Keep zero unless you have a tax service to compute

            foreach ($request->input('products') as $item) {
                /** @var Product $product */
                $product = Product::find($item['product_id']);
                $quantity = (int) $item['quantity'];
                $unitPrice = isset($item['price_override']) ? (float) $item['price_override'] : (float) ($product->price ?? 0);
                $discount = isset($item['discount']) ? (float) $item['discount'] : 0.0;

                $lineSubtotal = $unitPrice * $quantity;
                $lineTax = 0.0; // Placeholder; integrate tax service if available
                $lineTotal = $lineSubtotal - $discount + $lineTax;

                $orderItem = new OrderItem([
                    'order_id' => $order->id,
                    'product_id' => $product->id,
                    'product_name' => $product->name ?? 'Product',
                    'product_sku' => $product->sku ?? null,
                    'quantity' => $quantity,
                    'unit_price' => $unitPrice,
                    'subtotal' => $lineSubtotal,
                    'discount_amount' => $discount,
                    'tax_amount' => $lineTax,
                    'total_amount' => $lineTotal,
                ]);
                $orderItem->save();

                $subtotal += $lineSubtotal;
                $totalDiscount += $discount;
                $totalTax += $lineTax;
            }

            $order->subtotal = $subtotal;
            $order->discount_amount = $totalDiscount;
            $order->tax_amount = $totalTax;
            $order->shipping_amount = 0;
            $order->total_amount = $subtotal - $totalDiscount + $totalTax;
            $order->save();

            // Create manual invoice linked to order
            $invoice = new ManualInvoice();
            $invoice->order_id = $order->id;
            $invoice->created_by = auth()->id();
            $invoice->created_user_id = auth()->id();
            $invoice->invoice_number = method_exists($invoice, 'generateInvoiceNumber') ? $invoice->generateInvoiceNumber() : ('MINV-' . date('Ymd') . '-' . $order->id);
            $invoice->invoice_date = now();
            $invoice->due_date = $request->input('due_date', now()->copy()->addDays(7));
            $invoice->current_status = 'issued'; // unpaid by default
            $invoice->currency = $currency;
            $invoice->save();

            // Link on order for convenience if used elsewhere
            $order->invoice_id = $invoice->id;
            $order->save();

            // Generate payment link token stored in cache (avoid DB schema changes)
            $token = bin2hex(random_bytes(16));
            $ttlMinutes = 60 * 24 * 7; // 7 days
            Cache::put($this->cacheKeyForToken($token), [
                'manual_invoice_id' => $invoice->id,
                'expires_at' => now()->copy()->addMinutes($ttlMinutes)->toIso8601String(),
            ], now()->addMinutes($ttlMinutes));

            $paymentLinkUrl = url('/pay/' . $token);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Manual invoice created successfully',
                'data' => [
                    'invoice' => $invoice->load('order.items'),
                    'payment_link_url' => $paymentLinkUrl,
                    'payment_link_expires_at' => now()->addMinutes($ttlMinutes)->toIso8601String(),
                ],
            ], 201);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to create manual invoice',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /** Show a manual invoice with related data. */
    public function show(ManualInvoice $invoice)
    {
        $invoice->load('order.items');
        return response()->json([
            'success' => true,
            'data' => $invoice,
        ]);
    }

    /** Mark invoice as paid. */
    public function markPaid(ManualInvoice $invoice)
    {
        if ($invoice->current_status === 'paid') {
            return response()->json([
                'success' => true,
                'message' => 'Invoice already paid',
                'data' => $invoice,
            ]);
        }

        $invoice->markAsPaid();
        return response()->json([
            'success' => true,
            'message' => 'Invoice marked as paid',
            'data' => $invoice,
        ]);
    }

    /** Mark invoice as unpaid (issued). */
    public function markUnpaid(ManualInvoice $invoice)
    {
        if ($invoice->current_status === 'issued') {
            return response()->json([
                'success' => true,
                'message' => 'Invoice already unpaid',
                'data' => $invoice,
            ]);
        }

        $invoice->update([
            'current_status' => 'issued',
            'paid_at' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Invoice marked as unpaid',
            'data' => $invoice,
        ]);
    }

    /** Return existing payment link; if none in cache, generate one with default TTL. */
    public function resendLink(ManualInvoice $invoice)
    {
        // Try to find any token pointing to this invoice
        $token = $this->findExistingTokenForInvoice($invoice->id);
        if (!$token) {
            $token = bin2hex(random_bytes(16));
            Cache::put($this->cacheKeyForToken($token), [
                'manual_invoice_id' => $invoice->id,
                'expires_at' => now()->addDays(7)->toIso8601String(),
            ], now()->addDays(7));
        }
        return response()->json([
            'success' => true,
            'data' => [
                'payment_link_url' => url('/pay/' . $token),
            ],
        ]);
    }

    /** Invalidate old token and generate a new one. */
    public function regenerateLink(ManualInvoice $invoice)
    {
        // No global registry of tokens without DB; just create a new token
        $token = bin2hex(random_bytes(16));
        Cache::put($this->cacheKeyForToken($token), [
            'manual_invoice_id' => $invoice->id,
            'expires_at' => now()->addDays(7)->toIso8601String(),
        ], now()->addDays(7));

        return response()->json([
            'success' => true,
            'message' => 'Payment link regenerated',
            'data' => [
                'payment_link_url' => url('/pay/' . $token),
            ],
        ]);
    }

    private function cacheKeyForToken(string $token): string
    {
        return 'manual_invoice_token:' . $token;
    }

    private function findExistingTokenForInvoice(int $invoiceId): ?string
    {
        // Without a reverse index in DB or cache, we cannot search efficiently.
        // For now, always return null to force regeneration in resend if not available.
        return null;
    }
}


