<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\Order;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class InvoiceController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = Invoice::with(['order', 'creator', 'createdUser']);

        // Apply filters
        if ($request->has('status')) {
            $query->where('current_status', $request->status);
        }

        if ($request->has('type')) {
            $query->where('invoice_type', $request->type);
        }

        if ($request->has('date_from')) {
            $query->where('invoice_date', '>=', $request->date_from);
        }

        if ($request->has('date_to')) {
            $query->where('invoice_date', '<=', $request->date_to);
        }

        $invoices = $query->latest()->paginate($request->get('per_page', 15));

        return response()->json([
            'status' => 'success',
            'data' => $invoices
        ]);
    }

    public function show(Invoice $invoice): JsonResponse
    {
        $invoice->load(['order', 'creator', 'createdUser']);
        
        return response()->json([
            'status' => 'success',
            'data' => $invoice
        ]);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|exists:orders,id',
            'invoice_type' => 'required|in:retail,supplier',
            'invoice_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:invoice_date',
            'payment_terms' => 'required|string',
            'company_name' => 'required|string',
            'company_registration_number' => 'required|string',
            'company_address' => 'required|string',
            'company_phone' => 'required|string',
            'company_email' => 'required|email',
            'company_website' => 'nullable|url',
            'terms' => 'nullable|string',
            'supplier_notes' => 'nullable|string',
            'retailer_notes' => 'nullable|string',
            'internal_notes' => 'nullable|string',
            'footer' => 'nullable|string',
            'currency' => 'required|string|size:3'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            $invoice = new Invoice($request->all());
            $invoice->created_by = auth()->id();
            $invoice->created_user_id = auth()->id();
            $invoice->invoice_number = $invoice->generateInvoiceNumber();
            $invoice->current_status = 'draft';
            $invoice->is_retail = $request->invoice_type === 'retail';
            $invoice->is_supplier = $request->invoice_type === 'supplier';
            $invoice->save();

            DB::commit();

            return response()->json([
                'status' => 'success',
                'message' => 'Invoice created successfully',
                'data' => $invoice
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to create invoice',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, Invoice $invoice): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'invoice_date' => 'date',
            'due_date' => 'date|after_or_equal:invoice_date',
            'payment_terms' => 'string',
            'company_name' => 'string',
            'company_registration_number' => 'string',
            'company_address' => 'string',
            'company_phone' => 'string',
            'company_email' => 'email',
            'company_website' => 'nullable|url',
            'terms' => 'nullable|string',
            'supplier_notes' => 'nullable|string',
            'retailer_notes' => 'nullable|string',
            'internal_notes' => 'nullable|string',
            'footer' => 'nullable|string',
            'currency' => 'string|size:3'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            $invoice->update($request->all());

            DB::commit();

            return response()->json([
                'status' => 'success',
                'message' => 'Invoice updated successfully',
                'data' => $invoice
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update invoice',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function issue(Invoice $invoice): JsonResponse
    {
        if ($invoice->current_status !== 'draft') {
            return response()->json([
                'status' => 'error',
                'message' => 'Only draft invoices can be issued'
            ], 422);
        }

        try {
            $invoice->issue();
            return response()->json([
                'status' => 'success',
                'message' => 'Invoice issued successfully',
                'data' => $invoice
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to issue invoice',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function markAsPaid(Invoice $invoice): JsonResponse
    {
        if ($invoice->current_status !== 'issued') {
            return response()->json([
                'status' => 'error',
                'message' => 'Only issued invoices can be marked as paid'
            ], 422);
        }

        try {
            $invoice->markAsPaid();
            return response()->json([
                'status' => 'success',
                'message' => 'Invoice marked as paid successfully',
                'data' => $invoice
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to mark invoice as paid',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function void(Invoice $invoice): JsonResponse
    {
        if (!in_array($invoice->current_status, ['draft', 'issued'])) {
            return response()->json([
                'status' => 'error',
                'message' => 'Only draft or issued invoices can be voided'
            ], 422);
        }

        try {
            $invoice->void();
            return response()->json([
                'status' => 'success',
                'message' => 'Invoice voided successfully',
                'data' => $invoice
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to void invoice',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function cancel(Invoice $invoice): JsonResponse
    {
        if (!in_array($invoice->current_status, ['draft', 'issued'])) {
            return response()->json([
                'status' => 'error',
                'message' => 'Only draft or issued invoices can be cancelled'
            ], 422);
        }

        try {
            $invoice->cancel();
            return response()->json([
                'status' => 'success',
                'message' => 'Invoice cancelled successfully',
                'data' => $invoice
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to cancel invoice',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy(Invoice $invoice): JsonResponse
    {
        if ($invoice->current_status !== 'draft') {
            return response()->json([
                'status' => 'error',
                'message' => 'Only draft invoices can be deleted'
            ], 422);
        }

        try {
            $invoice->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Invoice deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to delete invoice',
                'error' => $e->getMessage()
            ], 500);
        }
    }
} 