<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class EmailSubscriptionController extends Controller
{
    /**
     * Store a new email subscription
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required|email|unique:email_subscribers,email'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            DB::table('email_subscribers')->insert([
                'email' => $request->email,
                'source' => 'homepage',
                'created_at' => now(),
                'updated_at' => now()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Thank you for subscribing!'
            ]);

        } catch (\Exception $e) {
            Log::error('Email subscription error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while processing your subscription'
            ], 500);
        }
    }

    /**
     * Get all email subscribers (SuperAdmin only)
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 15);
            $search = $request->input('search');
            $status = $request->input('status');
            $source = $request->input('source');
            $sortBy = $request->input('sort_by', 'created_at');
            $sortOrder = $request->input('sort_order', 'desc');

            $query = DB::table('email_subscribers');

            // Apply filters
            if ($search) {
                $query->where('email', 'like', "%{$search}%");
            }

            if ($status) {
                $query->where('status', $status);
            }

            if ($source) {
                $query->where('source', $source);
            }

            // Get total count before pagination
            $total = $query->count();

            // Apply sorting and pagination
            $subscribers = $query->orderBy($sortBy, $sortOrder)
                               ->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => [
                    'subscribers' => $subscribers->items(),
                    'total' => $total,
                    'per_page' => $perPage,
                    'current_page' => $subscribers->currentPage(),
                    'last_page' => $subscribers->lastPage()
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching email subscribers', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while fetching subscribers'
            ], 500);
        }
    }
} 