<?php

namespace App\Http\Controllers;

use App\Models\CustomerProfile;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class CustomerProfileController extends Controller
{
    /**
     * Display the authenticated customer's profile.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Request $request)
    {
        try {
            $user = $request->user();
            
            // Load the customer profile with related data
            $profile = CustomerProfile::with(['addresses', 'paymentMethods', 'documents'])
                ->where('user_id', $user->id)
                ->first();
            
            if (!$profile) {
                return response()->json([
                    'success' => false,
                    'message' => 'Customer profile not found'
                ], 404);
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'profile' => $profile,
                    'user' => $user->only(['id', 'name', 'email', 'phone'])
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving customer profile',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Create or update the customer profile.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $user = $request->user();
            
            // Validate the request
            $validator = Validator::make($request->all(), [
                'company_name' => 'required|string|max:255',
                'company_logo' => 'nullable|image|max:2048',
                'company_website' => 'nullable|url|max:255',
                'business_type' => 'nullable|string|max:100',
                'business_registration_number' => 'nullable|string|max:100',
                'tax_identification_number' => 'nullable|string|max:100',
                'industry' => 'nullable|string|max:100',
                'annual_revenue' => 'nullable|numeric|min:0',
                'number_of_employees' => 'nullable|integer|min:1',
                'established_year' => 'nullable|integer|min:1900|max:' . date('Y'),
                'notes' => 'nullable|string',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Handle logo upload if provided
            $logoPath = null;
            if ($request->hasFile('company_logo')) {
                $logoPath = $request->file('company_logo')->store('customer-logos', 'public');
            }
            
            // Create or update the profile
            $profile = CustomerProfile::updateOrCreate(
                ['user_id' => $user->id],
                array_merge(
                    $request->except('company_logo'),
                    $logoPath ? ['company_logo' => Storage::url($logoPath)] : []
                )
            );
            
            return response()->json([
                'success' => true,
                'message' => 'Customer profile updated successfully',
                'data' => [
                    'profile' => $profile->fresh()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating customer profile',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Admin endpoint to view a specific customer's profile.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function adminView($id)
    {
        try {
            // Check for admin permissions
            if (!auth()->user()->hasRole(['superadmin', 'admin', 'supervisor'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access'
                ], 403);
            }
            
            $user = User::findOrFail($id);
            
            // Check if user is a customer
            if (!$user->hasRole('buyer')) {
                return response()->json([
                    'success' => false,
                    'message' => 'User is not a customer'
                ], 400);
            }
            
            // Load the customer profile with related data
            $profile = CustomerProfile::with(['addresses', 'paymentMethods', 'documents'])
                ->where('user_id', $user->id)
                ->first();
            
            if (!$profile) {
                return response()->json([
                    'success' => false,
                    'message' => 'Customer profile not found'
                ], 404);
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'profile' => $profile,
                    'user' => $user->only(['id', 'name', 'email', 'phone', 'created_at'])
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving customer profile',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Admin endpoint to verify a customer profile.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function verify(Request $request, $id)
    {
        try {
            // Check for admin permissions
            if (!auth()->user()->hasRole(['superadmin', 'admin'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access'
                ], 403);
            }
            
            $profile = CustomerProfile::where('user_id', $id)->firstOrFail();
            
            // Validate request
            $validator = Validator::make($request->all(), [
                'discount_tier' => 'nullable|in:standard,bronze,silver,gold,platinum',
                'credit_limit' => 'nullable|numeric|min:0',
                'default_payment_terms' => 'nullable|integer|min:0|max:90',
                'notes' => 'nullable|string',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Update profile with verification information
            $profile->is_verified = true;
            $profile->verification_date = now();
            $profile->verified_by = auth()->id();
            
            // Update additional fields if provided
            if ($request->has('discount_tier')) {
                $profile->discount_tier = $request->discount_tier;
            }
            
            if ($request->has('credit_limit')) {
                $profile->credit_limit = $request->credit_limit;
            }
            
            if ($request->has('default_payment_terms')) {
                $profile->default_payment_terms = $request->default_payment_terms;
            }
            
            if ($request->has('notes')) {
                $profile->notes = $request->notes;
            }
            
            $profile->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Customer profile verified successfully',
                'data' => [
                    'profile' => $profile->fresh()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error verifying customer profile',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Admin endpoint to list all customers.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            // Check for admin permissions
            if (!auth()->user()->hasRole(['superadmin', 'admin', 'supervisor'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access'
                ], 403);
            }
            
            $query = CustomerProfile::with('user');
            
            // Apply filters
            if ($request->has('is_verified')) {
                $query->where('is_verified', $request->boolean('is_verified'));
            }
            
            if ($request->has('discount_tier')) {
                $query->where('discount_tier', $request->discount_tier);
            }
            
            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('company_name', 'like', "%{$search}%")
                      ->orWhere('business_registration_number', 'like', "%{$search}%")
                      ->orWhereHas('user', function($userQuery) use ($search) {
                          $userQuery->where('name', 'like', "%{$search}%")
                                   ->orWhere('email', 'like', "%{$search}%");
                      });
                });
            }
            
            // Apply sorting
            $sortBy = $request->input('sort_by', 'created_at');
            $sortDirection = $request->input('sort_direction', 'desc');
            
            if (in_array($sortBy, ['company_name', 'created_at', 'is_verified'])) {
                $query->orderBy($sortBy, $sortDirection);
            }
            
            // Paginate results
            $perPage = $request->input('per_page', 15);
            $customers = $query->paginate($perPage);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'customers' => $customers
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving customers',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Update the customer profile data.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request)
    {
        try {
            $user = $request->user();
            
            // Validate the request
            $validator = Validator::make($request->all(), [
                'company_name' => 'required|string|max:255',
                'company_logo' => 'nullable|image|max:2048',
                'company_website' => 'nullable|url|max:255',
                'business_type' => 'nullable|string|max:100',
                'business_registration_number' => 'nullable|string|max:100',
                'tax_identification_number' => 'nullable|string|max:100',
                'industry' => 'nullable|string|max:100',
                'annual_revenue' => 'nullable|numeric|min:0',
                'number_of_employees' => 'nullable|integer|min:1',
                'established_year' => 'nullable|integer|min:1900|max:' . date('Y'),
                'notes' => 'nullable|string',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Handle logo upload if provided
            $logoPath = null;
            if ($request->hasFile('company_logo')) {
                $logoPath = $request->file('company_logo')->store('customer-logos', 'public');
            }
            
            // Update the profile
            $profile = CustomerProfile::updateOrCreate(
                ['user_id' => $user->id],
                array_merge(
                    $request->except('company_logo'),
                    $logoPath ? ['company_logo' => Storage::url($logoPath)] : []
                )
            );
            
            return response()->json([
                'success' => true,
                'message' => 'Customer profile updated successfully',
                'data' => [
                    'profile' => $profile->fresh()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating customer profile',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get all customers for superadmin.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getAllCustomers()
    {
        try {
            // Check for superadmin permissions
            if (!auth()->user()->hasRole('superadmin')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access'
                ], 403);
            }
            
            // Get all customers
            $customers = CustomerProfile::with(['user'])->get();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'customers' => $customers
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving customers',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}