<?php

namespace App\Http\Controllers;

use App\Models\CustomerProfile;
use App\Models\CustomerMembership;
use Illuminate\Http\Request;

class CustomerMembershipController extends Controller
{
    // List all customer membership packages
    public function index()
    {
        return response()->json([
            'success' => true,
            'data' => CustomerMembership::all()
        ]);
    }

    // Create a new customer membership package
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100|unique:customer_memberships,name',
            'normal_discount' => 'required|numeric|min:0|max:100',
            'extra_discount' => 'required|numeric|min:0|max:100',
        ]);
        
        $membership = CustomerMembership::create($validated);
        
        return response()->json([
            'success' => true,
            'message' => 'Customer membership package created successfully.',
            'data' => $membership
        ], 201);
    }

    // Get a specific customer membership package
    public function show($id)
    {
        $membership = CustomerMembership::findOrFail($id);
        
        return response()->json([
            'success' => true,
            'data' => $membership
        ]);
    }

    // Update a customer membership package
    public function update(Request $request, $id)
    {
        $membership = CustomerMembership::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required|string|max:100|unique:customer_memberships,name,' . $id,
            'normal_discount' => 'required|numeric|min:0|max:100',
            'extra_discount' => 'required|numeric|min:0|max:100',
        ]);
        
        $membership->update($validated);
        
        return response()->json([
            'success' => true,
            'message' => 'Customer membership package updated successfully.',
            'data' => $membership
        ]);
    }

    // Delete a customer membership package
    public function destroy($id)
    {
        $membership = CustomerMembership::findOrFail($id);
        
        // Check if any customers are using this membership
        $customersCount = CustomerProfile::where('membership_id', $id)->count();
        
        if ($customersCount > 0) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete membership package. ' . $customersCount . ' customer(s) are currently using this membership.',
                'customers_count' => $customersCount
            ], 400);
        }
        
        $membership->delete();
        
        return response()->json([
            'success' => true,
            'message' => 'Customer membership package deleted successfully.'
        ]);
    }

    // Assign multiple customers to a membership package
    public function bulkAssign(Request $request)
    {
        $request->validate([
            'membership_id' => 'required|exists:customer_memberships,id',
            'customer_profile_ids' => 'required|array|min:1',
            'customer_profile_ids.*' => 'required|exists:customer_profiles,id',
        ]);

        $membershipId = $request->membership_id;
        $customerProfileIds = $request->customer_profile_ids;
        
        // Update all specified customer profiles
        $updatedCount = CustomerProfile::whereIn('id', $customerProfileIds)
            ->update(['membership_id' => $membershipId]);
        
        // Get the updated customer profiles with user data
        $updatedCustomers = CustomerProfile::whereIn('id', $customerProfileIds)
            ->with('user')
            ->get();
        
        return response()->json([
            'success' => true,
            'message' => $updatedCount . ' customer(s) assigned to membership package.',
            'updated_count' => $updatedCount,
            'data' => $updatedCustomers
        ]);
    }

    // Assign a membership to a customer
    public function assign(Request $request, $customerProfileId)
    {
        $request->validate([
            'membership_id' => 'required|exists:customer_memberships,id',
        ]);
        $customer = CustomerProfile::findOrFail($customerProfileId);
        $customer->membership_id = $request->membership_id;
        $customer->save();
        $customer->load('user');
        return response()->json([
            'success' => true,
            'message' => 'Membership assigned to customer.',
            'data' => $customer
        ]);
    }

    // Remove a membership from a customer
    public function remove($customerProfileId)
    {
        $customer = CustomerProfile::findOrFail($customerProfileId);
        $customer->membership_id = null;
        $customer->save();
        $customer->load('user');
        return response()->json([
            'success' => true,
            'message' => 'Membership removed from customer.',
            'data' => $customer
        ]);
    }

    // Get a customer's membership
    public function get($customerProfileId)
    {
        $customer = CustomerProfile::with('membership')->findOrFail($customerProfileId);
        return response()->json([
            'success' => true,
            'membership' => $customer->membership
        ]);
    }

    // List all customers under a specific membership
    public function customers($membershipId)
    {
        $membership = CustomerMembership::with('customerProfiles.user')->findOrFail($membershipId);
        return response()->json([
            'success' => true,
            'customers' => $membership->customerProfiles
        ]);
    }
} 