<?php

namespace App\Http\Controllers;

use App\Models\CustomerAddress;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CustomerAddressController extends Controller
{
    /**
     * Display a listing of the customer's addresses.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $userId = $request->user()->id;
            
            $query = CustomerAddress::where('user_id', $userId);
            
            // Filter by address type
            if ($request->has('address_type')) {
                $query->ofType($request->address_type);
            }
            
            $addresses = $query->get();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'addresses' => $addresses
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving addresses',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Store a newly created address.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $userId = $request->user()->id;
            
            // Validate the request
            $validator = Validator::make($request->all(), [
                'address_type' => 'required|in:billing,shipping,both',
                'is_default' => 'nullable|boolean',
                'company_name' => 'nullable|string|max:255',
                'contact_name' => 'required|string|max:255',
                'contact_email' => 'nullable|email|max:255',
                'contact_phone' => 'nullable|string|max:20',
                'address_line1' => 'required|string|max:255',
                'address_line2' => 'nullable|string|max:255',
                'city' => 'required|string|max:100',
                'state' => 'required|string|max:100',
                'postal_code' => 'required|string|max:20',
                'country' => 'required|string|max:100',
                'delivery_instructions' => 'nullable|string'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Check if setting as default
            $isDefault = $request->boolean('is_default', false);
            
            // Begin transaction
            \DB::beginTransaction();
            
            // If setting as default, clear other defaults of the same type
            if ($isDefault) {
                $addressType = $request->address_type;
                
                if ($addressType === 'both' || $addressType === 'billing') {
                    CustomerAddress::where('user_id', $userId)
                        ->where(function($query) {
                            $query->where('address_type', 'billing')
                                  ->orWhere('address_type', 'both');
                        })
                        ->update(['is_default' => false]);
                }
                
                if ($addressType === 'both' || $addressType === 'shipping') {
                    CustomerAddress::where('user_id', $userId)
                        ->where(function($query) {
                            $query->where('address_type', 'shipping')
                                  ->orWhere('address_type', 'both');
                        })
                        ->update(['is_default' => false]);
                }
            }
            
            // Create the address
            $address = CustomerAddress::create(array_merge(
                $request->all(),
                ['user_id' => $userId]
            ));
            
            \DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Address created successfully',
                'data' => [
                    'address' => $address
                ]
            ], 201);
        } catch (\Exception $e) {
            \DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Error creating address',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Display the specified address.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $userId = auth()->id();
            
            $address = CustomerAddress::where('id', $id)
                ->where('user_id', $userId)
                ->firstOrFail();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'address' => $address
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Address not found',
                'error' => $e->getMessage()
            ], 404);
        }
    }
    
    /**
     * Update the specified address.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $userId = $request->user()->id;
            
            // Find the address
            $address = CustomerAddress::where('id', $id)
                ->where('user_id', $userId)
                ->firstOrFail();
            
            // Validate the request
            $validator = Validator::make($request->all(), [
                'address_type' => 'nullable|in:billing,shipping,both',
                'is_default' => 'nullable|boolean',
                'company_name' => 'nullable|string|max:255',
                'contact_name' => 'nullable|string|max:255',
                'contact_email' => 'nullable|email|max:255',
                'contact_phone' => 'nullable|string|max:20',
                'address_line1' => 'nullable|string|max:255',
                'address_line2' => 'nullable|string|max:255',
                'city' => 'nullable|string|max:100',
                'state' => 'nullable|string|max:100',
                'postal_code' => 'nullable|string|max:20',
                'country' => 'nullable|string|max:100',
                'delivery_instructions' => 'nullable|string'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Check if address type is changing
            $addressTypeChanging = $request->has('address_type') && $request->address_type !== $address->address_type;
            
            // Check if setting as default
            $settingAsDefault = $request->has('is_default') && $request->boolean('is_default') && !$address->is_default;
            
            // Begin transaction
            \DB::beginTransaction();
            
            // If setting as default or changing address type, update other defaults
            if ($settingAsDefault || $addressTypeChanging) {
                $newAddressType = $request->address_type ?? $address->address_type;
                
                if ($newAddressType === 'both' || $newAddressType === 'billing') {
                    CustomerAddress::where('user_id', $userId)
                        ->where('id', '!=', $id)
                        ->where(function($query) {
                            $query->where('address_type', 'billing')
                                  ->orWhere('address_type', 'both');
                        })
                        ->update(['is_default' => false]);
                }
                
                if ($newAddressType === 'both' || $newAddressType === 'shipping') {
                    CustomerAddress::where('user_id', $userId)
                        ->where('id', '!=', $id)
                        ->where(function($query) {
                            $query->where('address_type', 'shipping')
                                  ->orWhere('address_type', 'both');
                        })
                        ->update(['is_default' => false]);
                }
            }
            
            // Update the address
            $address->fill($request->all());
            $address->save();
            
            \DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Address updated successfully',
                'data' => [
                    'address' => $address->fresh()
                ]
            ]);
        } catch (\Exception $e) {
            \DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating address',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Remove the specified address.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $userId = auth()->id();
            
            $address = CustomerAddress::where('id', $id)
                ->where('user_id', $userId)
                ->firstOrFail();
            
            // Check if address is in use as a billing address for payment methods
            $paymentMethodCount = $address->paymentMethods()->count();
            if ($paymentMethodCount > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete address that is used by payment methods',
                    'details' => "This address is used by {$paymentMethodCount} payment methods"
                ], 409);
            }
            
            // Delete the address
            $address->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Address deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting address',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Set an address as default for its type.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function setDefault($id)
    {
        try {
            $userId = auth()->id();
            
            $address = CustomerAddress::where('id', $id)
                ->where('user_id', $userId)
                ->firstOrFail();
            
            // Begin transaction
            \DB::beginTransaction();
            
            // Clear other defaults of the same type
            if ($address->address_type === 'both' || $address->address_type === 'billing') {
                CustomerAddress::where('user_id', $userId)
                    ->where('id', '!=', $id)
                    ->where(function($query) {
                        $query->where('address_type', 'billing')
                              ->orWhere('address_type', 'both');
                    })
                    ->update(['is_default' => false]);
            }
            
            if ($address->address_type === 'both' || $address->address_type === 'shipping') {
                CustomerAddress::where('user_id', $userId)
                    ->where('id', '!=', $id)
                    ->where(function($query) {
                        $query->where('address_type', 'shipping')
                              ->orWhere('address_type', 'both');
                    })
                    ->update(['is_default' => false]);
            }
            
            // Set this address as default
            $address->is_default = true;
            $address->save();
            
            \DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Address set as default successfully',
                'data' => [
                    'address' => $address->fresh()
                ]
            ]);
        } catch (\Exception $e) {
            \DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Error setting address as default',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get the default address for a specific type.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDefault(Request $request)
    {
        try {
            $userId = $request->user()->id;
            
            // Validate the request
            $validator = Validator::make($request->all(), [
                'type' => 'required|in:billing,shipping'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $addressType = $request->type;
            
            // Get the default address
            $address = CustomerAddress::where('user_id', $userId)
                ->where(function($query) use ($addressType) {
                    $query->where('address_type', $addressType)
                          ->orWhere('address_type', 'both');
                })
                ->where('is_default', true)
                ->first();

            if (!$address) {
                return response()->json([
                    'success' => false,
                    'message' => 'No default address found',
                    'details' => "No default {$addressType} address found"
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'address' => $address
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving default address',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}