<?php

namespace App\Http\Controllers;

use App\Models\Cart;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\CustomerAddress;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Services\StripeService;
use App\Services\OrderService;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Mail;

class CheckoutController extends Controller
{
    protected $stripeService;
    protected $orderService;

    public function __construct(StripeService $stripeService, OrderService $orderService)
    {
        $this->stripeService = $stripeService;
        $this->orderService = $orderService;
    }

    /**
     * Process checkout from direct order data
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function processOrder(Request $request)
    {
        try {
            // Validate required parameters
            $validated = $request->validate([
                'payment_intent_id' => 'required|string',
                'cart_id' => 'required|integer|exists:carts,id',
                'billing_address' => 'required|array',
                'billing_address.company_name' => 'nullable|string|max:255',
                'billing_address.contact_name' => 'required|string|max:255',
                'billing_address.contact_email' => 'nullable|email|max:255',
                'billing_address.contact_phone' => 'nullable|string|max:20',
                'billing_address.address_line1' => 'required|string|max:255',
                'billing_address.address_line2' => 'nullable|string|max:255',
                'billing_address.city' => 'required|string|max:100',
                'billing_address.state' => 'required|string|max:100',
                'billing_address.postal_code' => 'required|string|max:20',
                'billing_address.country' => 'required|string|max:100',
                'shipping_address' => 'required|array',
                'shipping_address.company_name' => 'nullable|string|max:255',
                'shipping_address.contact_name' => 'required|string|max:255',
                'shipping_address.contact_email' => 'nullable|email|max:255',
                'shipping_address.contact_phone' => 'nullable|string|max:20',
                'shipping_address.address_line1' => 'required|string|max:255',
                'shipping_address.address_line2' => 'nullable|string|max:255',
                'shipping_address.city' => 'required|string|max:100',
                'shipping_address.state' => 'required|string|max:100',
                'shipping_address.postal_code' => 'required|string|max:20',
                'shipping_address.country' => 'required|string|max:100',
                'shipping_address.delivery_instructions' => 'nullable|string',
                'payment_method' => 'required|string',
                'notes' => 'nullable|string',
            ]);

            Log::info('Starting order processing', [
                'payment_intent_id' => $request->payment_intent_id,
                'cart_id' => $request->cart_id,
                'user_id' => auth()->id()
            ]);

            // Verify the payment
            Log::info('Verifying payment intent', [
                'payment_intent_id' => $request->payment_intent_id
            ]);

            $paymentVerification = $this->stripeService->verifyPayment($request->payment_intent_id);
            
            Log::info('Payment verification result', [
                'success' => $paymentVerification['success'],
                'error' => $paymentVerification['error'] ?? null,
                'amount' => $paymentVerification['amount'] ?? null,
                'status' => $paymentVerification['status'] ?? null
            ]);
            
            if (!$paymentVerification['success']) {
                Log::error('Payment verification failed', [
                    'payment_intent_id' => $request->payment_intent_id,
                    'error' => $paymentVerification['error'] ?? 'Unknown error',
                    'status' => $paymentVerification['status'] ?? 'Unknown status'
                ]);

                // Try to send email but don't block the process
                try {
                    if (isset($request->shipping_address['contact_email'])) {
                        Log::info('Attempting to send payment failed email', [
                            'email' => $request->shipping_address['contact_email']
                        ]);

                        Mail::send('emails.payment-failed', [
                            'shippingAddress' => $request->shipping_address,
                            'orderNumber' => 'TEMP-' . time(),
                            'errorMessage' => $paymentVerification['error'] ?? 'Payment verification failed'
                        ], function($message) use ($request) {
                            $message->to($request->shipping_address['contact_email'])
                                   ->subject('Payment Failed - Diptouch');
                        });

                        Log::info('Payment failed email sent successfully');
                    }
                } catch (\Exception $e) {
                    Log::error('Failed to send payment failed email', [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString(),
                        'email' => $request->shipping_address['contact_email'] ?? 'no email set',
                        'mail_config' => [
                            'driver' => config('mail.default'),
                            'host' => config('mail.mailers.smtp.host'),
                            'port' => config('mail.mailers.smtp.port'),
                            'encryption' => config('mail.mailers.smtp.encryption'),
                            'username' => config('mail.mailers.smtp.username'),
                        ]
                    ]);
                }

                return response()->json([
                    'success' => false,
                    'message' => 'Payment verification failed',
                    'error' => $paymentVerification['error']
                ], 400);
            }

            // Get cart and verify amount
            Log::info('Retrieving cart', [
                'cart_id' => $request->cart_id
            ]);

            $cart = $this->orderService->getCart($request->cart_id);
            if (!$cart) {
                Log::error('Cart not found', [
                    'cart_id' => $request->cart_id
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Cart not found'
                ], 404);
            }

            // Ensure cart belongs to authenticated user
            if ($cart->user_id != auth()->id()) {
                Log::error('Cart does not belong to authenticated user', [
                    'cart_id' => $request->cart_id,
                    'cart_user_id' => $cart->user_id,
                    'authenticated_user_id' => auth()->id()
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to cart'
                ], 403);
            }

            // Ensure cart totals are calculated with role-based margins
            $user = auth()->user();
            $user->load(['role', 'customerProfile.membership']);
            $cart->calculateTotals($user);

            $finalPrice = round($this->orderService->calculateFinalPrice($cart), 2);
            $paymentVerification['amount'] = round($paymentVerification['amount'], 2);
            
            Log::info('Verifying payment amount', [
                'cart_final_price' => $finalPrice,
                'payment_amount' => $paymentVerification['amount']
            ]);
            
            // Verify amount matches
            if ($paymentVerification['amount'] != $finalPrice) {
                Log::error('Payment amount mismatch', [
                    'expected' => $finalPrice,
                    'received' => $paymentVerification['amount'],
                    'difference' => abs($finalPrice - $paymentVerification['amount'])
                ]);

                // Try to send email but don't block the process
                try {
                    if (isset($request->shipping_address['contact_email'])) {
                        Log::info('Attempting to send payment amount mismatch email', [
                            'email' => $request->shipping_address['contact_email']
                        ]);

                        Mail::send('emails.payment-failed', [
                            'shippingAddress' => $request->shipping_address,
                            'orderNumber' => 'TEMP-' . time(),
                            'errorMessage' => 'Payment amount mismatch'
                        ], function($message) use ($request) {
                            $message->to($request->shipping_address['contact_email'])
                                   ->subject('Payment Failed - Diptouch');
                        });

                        Log::info('Payment amount mismatch email sent successfully');
                    }
                } catch (\Exception $e) {
                    Log::error('Failed to send payment amount mismatch email', [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString(),
                        'email' => $request->shipping_address['contact_email'] ?? 'no email set'
                    ]);
                }

                return response()->json([
                    'success' => false,
                    'message' => 'Payment amount mismatch',
                    'expected' => $finalPrice,
                    'received' => $paymentVerification['amount']
                ], 400);
            }

            Log::info('Starting order creation', [
                'cart_id' => $cart->id,
                'user_id' => auth()->id()
            ]);

            // Create the order
            $order = new Order();
            $order->order_number = 'ORD-' . strtoupper(Str::random(8));
            $order->user_id = auth()->id();
            $order->cart_id = $cart->id;
            $order->status = 'pending';
            $order->payment_status = 'pending';
            $order->shipping_status = 'pending';
            
            // Set financial details using the same calculation logic as cart display
            // Calculate totals using the same logic as CartController
            $formattedItems = $cart->items->map(function ($item) use ($user) {
                $base_unit_price_with_margin = null;
                $special_unit_price_with_margin = null;
                $subtotal = null;
                $quantity = $item->quantity;
                
                // Get appropriate margin based on user role
                if ($user && ($user->hasRole('partner') || $user->hasRole('vendor'))) {
                    $margin = $item->partner_margin;
                    $marginType = $item->partner_margin_type;
                } else {
                    $margin = $item->customer_margin;
                    $marginType = $item->customer_margin_type;
                }

                // Use the actual unit_price that was set when the cart item was created
                $base_price = $item->unit_price;
                
                // Check for special pricing
                if ($item->is_bulk_pricing && $item->bulkPrice && $item->bulkPrice->per_pack_special_price) {
                    $special_price = $item->bulkPrice->per_pack_special_price;
                } elseif ($item->is_pack_pricing && $item->packPrice && $item->packPrice->per_pack_special_price) {
                    $special_price = $item->packPrice->per_pack_special_price;
                } else {
                    $special_price = null;
                }

                if ($marginType === 'percentage') {
                    $base_unit_price_with_margin = round($base_price * (1 + ($margin / 100)), 2);
                    if ($special_price && $special_price > 0) {
                        $special_unit_price_with_margin = round($special_price * (1 + ($margin / 100)), 2);
                    }
                } else {
                    $base_unit_price_with_margin = round($base_price + $margin, 2);
                    if ($special_price && $special_price > 0) {
                        $special_unit_price_with_margin = round($special_price + $margin, 2);
                    }
                }

                if ($special_unit_price_with_margin !== null) {
                    $subtotal = round($special_unit_price_with_margin * $quantity, 2);
                } else {
                    $subtotal = round($base_unit_price_with_margin * $quantity, 2);
                }

                return [
                    'product_id' => $item->product_id,
                    'quantity' => $quantity,
                    'pack_price_id' => $item->pack_price_id,
                    'bulk_price_id' => $item->bulk_price_id,
                    'subtotal' => $subtotal,
                    'delivery_fee' => $item->delivery_fee
                ];
            })->toArray();

            // Use the same fillCartItemPricingDetails logic as CartController
            list($formattedItems, $deliveryFees) = $this->orderService->fillCartItemPricingDetails($formattedItems);

            // Calculate totals using the same logic as CartController
            $cart_total_amount = array_sum(array_column($formattedItems, 'subtotal'));
            
            // Calculate tax per item based on product-specific tax rates
            $tax_amount = 0;
            foreach ($cart->items as $item) {
                $taxRate = $item->getTaxRate();
                $taxableAmount = $item->subtotal - $item->discount_amount;
                $tax_amount += round($taxableAmount * ($taxRate / 100), 2);
            }
            
            $total_delivery_fees = array_sum($deliveryFees);
            
            // Calculate membership discount on total_amount (subtotal)
            $membershipDiscount = $cart->calculateMembershipDiscount($user);
            $membership_discount_amount = $membershipDiscount['amount'];
            $membership_discount_percentage = $membershipDiscount['percentage'];
            
            // Apply membership discount to total_amount, then calculate final price
            $total_amount_after_membership_discount = $cart_total_amount - $membership_discount_amount;
            // Recalculate tax after membership discount (proportional to each item's tax rate)
            $tax_amount_after_membership_discount = 0;
            foreach ($cart->items as $item) {
                $taxRate = $item->getTaxRate();
                $itemSubtotal = $item->subtotal - $item->discount_amount;
                $itemMembershipDiscount = $this->calculateItemMembershipDiscount($itemSubtotal, $user)['amount'];
                $taxableAmountAfterDiscount = $itemSubtotal - $itemMembershipDiscount;
                $tax_amount_after_membership_discount += round($taxableAmountAfterDiscount * ($taxRate / 100), 2);
            }
            $total_amount_with_tax = round($total_amount_after_membership_discount + $tax_amount_after_membership_discount, 2);
            $final_price = round($total_amount_after_membership_discount + $tax_amount_after_membership_discount + $total_delivery_fees, 2);
            
            // Set financial details (rounded) - using the same calculation as cart display
            // subtotal should be post-membership subtotal for order display
            $order->subtotal = round($total_amount_after_membership_discount, 2);
            $order->discount_amount = round($cart->discount_amount + $membership_discount_amount, 2);
            $order->shipping_amount = round($total_delivery_fees, 2);
            $order->total_amount = round($final_price, 2);
            $order->tax_amount = round($tax_amount_after_membership_discount, 2);
            $order->currency_code = $cart->currency_code;
            
            // Set membership discount information
            $order->membership_discount_amount = round($cart->membership_discount_amount, 2);
            $order->membership_discount_percentage = round($cart->membership_discount_percentage, 2);
            $order->applied_membership_id = $cart->applied_membership_id;
            $order->membership_discount_description = $cart->membership_discount_description;
            
            // Set address info
            $order->billing_address = $this->formatAddress($request->billing_address);
            $order->shipping_address = $this->formatAddress($request->shipping_address);
            $order->billing_phone = $request->billing_address['contact_phone'];
            $order->shipping_phone = $request->shipping_address['contact_phone'];
            $order->billing_email = $request->billing_address['contact_email'] ?? auth()->user()->email;
            
            // Set payment info
            $order->payment_method = $request->payment_method;
            $order->notes = $request->notes ?? null;
            
            $order->save();

            Log::info('Order created successfully', [
                'order_id' => $order->id,
                'order_number' => $order->order_number
            ]);

            // Create order items
            Log::info('Creating order items', [
                'order_id' => $order->id,
                'item_count' => $cart->items->count()
            ]);

            foreach ($cart->items as $cartItem) {
                Log::info('Processing cart item', [
                    'cart_item_id' => $cartItem->id,
                    'product_id' => $cartItem->product_id,
                    'quantity' => $cartItem->quantity
                ]);

                // Recalculate membership discount for this item to ensure correct values
                $itemSubtotal = $cartItem->subtotal;
                $membershipDiscount = $this->calculateItemMembershipDiscount($itemSubtotal, $user);

                $orderItem = new OrderItem();
                $orderItem->order_id = $order->id;
                $orderItem->product_id = $cartItem->product_id;
                $orderItem->product_name = $cartItem->product->name;
                $orderItem->product_sku = $cartItem->product->sku;
                $orderItem->quantity = $cartItem->quantity;
                $orderItem->unit_price = round($cartItem->unit_price, 2);
                $orderItem->subtotal = round($cartItem->subtotal, 2);
                $orderItem->discount_amount = round($cartItem->discount_amount, 2);
                $orderItem->tax_amount = round($cartItem->tax_amount, 2);
                $orderItem->total_amount = round($cartItem->total_amount, 2);
                $orderItem->is_bulk_pricing = $cartItem->is_bulk_pricing;
                $orderItem->is_pack_pricing = $cartItem->is_pack_pricing;
                $orderItem->bulk_price_id = $cartItem->bulk_price_id;
                $orderItem->pack_price_id = $cartItem->pack_price_id;
                $orderItem->customer_margin = $cartItem->customer_margin;
                $orderItem->partner_margin = $cartItem->partner_margin;
                $orderItem->customer_margin_type = $cartItem->customer_margin_type;
                $orderItem->partner_margin_type = $cartItem->partner_margin_type;
                $orderItem->membership_discount_amount = round($membershipDiscount['amount'], 2);
                $orderItem->membership_discount_percentage = round($membershipDiscount['percentage'], 2);
                $orderItem->original_price_before_membership = round($cartItem->subtotal, 2);
                
                $orderItem->save();

                Log::info('Order item created', [
                    'order_item_id' => $orderItem->id,
                    'product_id' => $orderItem->product_id
                ]);
                
                // Decrement product inventory
                $product = $cartItem->product;
                $product->quantity -= $cartItem->quantity;
                $product->save();

                Log::info('Product inventory updated', [
                    'product_id' => $product->id,
                    'new_quantity' => $product->quantity
                ]);
            }

            // Try to send order confirmation email
            try {
                if (isset($order->shipping_address['contact_email'])) {
                    Log::info('Attempting to send order confirmation email', [
                        'email' => $order->shipping_address['contact_email'],
                        'order_id' => $order->id
                    ]);

                    Mail::send('emails.order-success', [
                        'order' => $order
                    ], function($message) use ($order) {
                        $message->to($order->shipping_address['contact_email'])
                               ->subject('Order Confirmation - Diptouch');
                    });

                    Log::info('Order confirmation email sent successfully');
                }
            } catch (\Exception $e) {
                Log::error('Failed to send order confirmation email', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                    'order_id' => $order->id,
                    'email' => $order->shipping_address['contact_email'] ?? 'no email set'
                ]);
            }

            // After order and order items are created, reload and round all monetary values
            Log::info('Finalizing order data', [
                'order_id' => $order->id
            ]);

            $order->refresh();
            $orderItems = $order->items;
            foreach ($orderItems as $item) {
                $item->unit_price = round($item->unit_price, 2);
                $item->subtotal = round($item->subtotal, 2);
                $item->discount_amount = round($item->discount_amount, 2);
                $item->tax_amount = round($item->tax_amount, 2);
                $item->total_amount = round($item->total_amount, 2);
            }
            $order->subtotal = round($order->subtotal, 2);
            $order->discount_amount = round($order->discount_amount, 2);
            $order->tax_amount = round($order->tax_amount, 2);
            $order->shipping_amount = round($order->shipping_amount, 2);
            $order->total_amount = round($order->total_amount, 2);

            Log::info('Order processing completed successfully', [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'total_amount' => $order->total_amount
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Order processed successfully',
                'data' => [
                    'order' => $order,
                    'items' => $orderItems
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Error processing order', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'payment_intent_id' => $request->payment_intent_id ?? 'not set',
                'cart_id' => $request->cart_id ?? 'not set',
                'user_id' => auth()->id() ?? 'not set'
            ]);

            // Try to send payment failed email
            try {
                if (isset($request->shipping_address['contact_email'])) {
                    Log::info('Attempting to send error notification email', [
                        'email' => $request->shipping_address['contact_email']
                    ]);

                    Mail::send('emails.payment-failed', [
                        'shippingAddress' => $request->shipping_address,
                        'orderNumber' => 'TEMP-' . time(),
                        'errorMessage' => 'An unexpected error occurred'
                    ], function($message) use ($request) {
                        $message->to($request->shipping_address['contact_email'])
                               ->subject('Payment Failed - Diptouch');
                    });

                    Log::info('Error notification email sent successfully');
                }
            } catch (\Exception $emailError) {
                Log::error('Failed to send error notification email', [
                    'error' => $emailError->getMessage(),
                    'trace' => $emailError->getTraceAsString(),
                    'email' => $request->shipping_address['contact_email'] ?? 'no email set'
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'Error processing order',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Format address array into a string
     *
     * @param array $address
     * @return string
     */
    private function formatAddress(array $address): string
    {
        $parts = [];
        
        if (!empty($address['company_name'])) {
            $parts[] = $address['company_name'];
        }
        
        if (!empty($address['contact_name'])) {
            $parts[] = $address['contact_name'];
        }
        
        if (!empty($address['address_line1'])) {
            $parts[] = $address['address_line1'];
        }
        
        if (!empty($address['address_line2'])) {
            $parts[] = $address['address_line2'];
        }
        
        $cityStatePostal = array_filter([
            $address['city'],
            $address['state'],
            $address['postal_code']
        ]);
        
        if (!empty($cityStatePostal)) {
            $parts[] = implode(', ', $cityStatePostal);
        }
        
        if (!empty($address['country'])) {
            $parts[] = $address['country'];
        }
        
        return implode("\n", $parts);
    }

    /**
     * Calculate membership discount for a specific item.
     */
    private function calculateItemMembershipDiscount($itemSubtotal, $user)
    {
        if (!$user) return ['amount' => 0, 'percentage' => 0];
        
        $membership = $user->getActiveMembership();
        if (!$membership) return ['amount' => 0, 'percentage' => 0];
        
        $discountPercentage = $membership->extra_discount;
        $discountAmount = ($itemSubtotal * $discountPercentage) / 100;
        
        return [
            'amount' => round($discountAmount, 2),
            'percentage' => $discountPercentage
        ];
    }

    /**
     * Initialize checkout process
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function initializeCheckout(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'cart_id' => 'required|exists:carts,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $cart = $this->orderService->getCart($request->cart_id);
            if (!$cart) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart not found'
                ], 404);
            }
            if ($cart->items->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart is empty',
                    'details' => 'Please add items to your cart before checkout'
                ], 400);
            }

            // Use OrderService's calculateFinalPrice method for consistent calculation
            // This ensures the same logic as used in cart display
            $amount = round($this->orderService->calculateFinalPrice($cart), 2);

            if ($amount <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid cart total'
                ], 400);
            }

            $paymentResult = $this->stripeService->createPaymentIntent(
                $amount,
                $cart->currency_code ?? 'AUD',
                [
                    'cart_id' => $request->cart_id,
                    'user_id' => auth()->id()
                ]
            );
            if (!$paymentResult['success']) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create payment intent',
                    'error' => $paymentResult['error']
                ], 500);
            }
            return response()->json([
                'success' => true,
                'data' => [
                    'client_secret' => $paymentResult['client_secret'],
                    'payment_intent_id' => $paymentResult['payment_intent_id'],
                    'amount' => $amount,
                    'currency' => $cart->currency_code ?? 'AUD'
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to initialize checkout',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Process checkout from cart (original method)
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function process(Request $request)
    {
        try {
            // Get the authenticated user
            $user = $request->user();
            
            // Validate request data
            $validated = $request->validate([
                'billing_address' => 'required|array',
                'billing_address.company_name' => 'nullable|string|max:255',
                'billing_address.contact_name' => 'required|string|max:255',
                'billing_address.contact_email' => 'nullable|email|max:255',
                'billing_address.contact_phone' => 'nullable|string|max:20',
                'billing_address.address_line1' => 'required|string|max:255',
                'billing_address.address_line2' => 'nullable|string|max:255',
                'billing_address.city' => 'required|string|max:100',
                'billing_address.state' => 'required|string|max:100',
                'billing_address.postal_code' => 'required|string|max:20',
                'billing_address.country' => 'required|string|max:100',
                'shipping_address' => 'required|array',
                'shipping_address.company_name' => 'nullable|string|max:255',
                'shipping_address.contact_name' => 'required|string|max:255',
                'shipping_address.contact_email' => 'nullable|email|max:255',
                'shipping_address.contact_phone' => 'nullable|string|max:20',
                'shipping_address.address_line1' => 'required|string|max:255',
                'shipping_address.address_line2' => 'nullable|string|max:255',
                'shipping_address.city' => 'required|string|max:100',
                'shipping_address.state' => 'required|string|max:100',
                'shipping_address.postal_code' => 'required|string|max:20',
                'shipping_address.country' => 'required|string|max:100',
                'shipping_address.delivery_instructions' => 'nullable|string',
                'payment_method' => 'required|string',
                'notes' => 'nullable|string',
            ]);
            
            // Get active cart with all relationships
            $cart = Cart::getOrCreateForUser($user->id);
            $cart->load(['items.product', 'items.packPrice', 'items.bulkPrice', 'currency']);
            
            // Ensure cart totals are calculated with role-based margins
            $user->load('role');
            $cart->calculateTotals($user);
            
            // Check if cart is empty
            if ($cart->items->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart is empty',
                    'details' => 'Please add items to your cart before checkout'
                ], 400);
            }
            
            // Check inventory
            foreach ($cart->items as $item) {
                if ($item->product->quantity < $item->quantity) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Insufficient inventory',
                        'details' => "Not enough inventory for product: {$item->product->name}",
                        'data' => [
                            'product_id' => $item->product_id,
                            'requested_quantity' => $item->quantity,
                            'available_quantity' => $item->product->quantity
                        ]
                    ], 400);
                }
            }
            
            // Start transaction
            DB::beginTransaction();
            
            // Create order
            $order = new Order();
            $order->order_number = 'ORD-' . strtoupper(Str::random(8));
            $order->user_id = $user->id;
            $order->cart_id = $cart->id;
            $order->status = 'pending';
            $order->payment_status = 'pending';
            $order->shipping_status = 'pending';
            
            // Set financial details - align with post-membership discount semantics
            $order->subtotal = $cart->total_amount; // post-membership subtotal
            $order->discount_amount = round(($cart->discount_amount ?? 0) + ($cart->membership_discount_amount ?? 0), 2);
            $order->shipping_amount = $cart->total_delivery_fees;
            $order->total_amount = $cart->final_price;
            $order->tax_amount = round($cart->total_amount_with_tax - $cart->total_amount, 2);
            $order->currency_code = $cart->currency_code;
            // Set membership discount information
            $order->membership_discount_amount = round($cart->membership_discount_amount ?? 0, 2);
            $order->membership_discount_percentage = round($cart->membership_discount_percentage ?? 0, 2);
            $order->applied_membership_id = $cart->applied_membership_id;
            $order->membership_discount_description = $cart->membership_discount_description;
            
            // Set address info
            $order->billing_address = $this->formatAddress($request->billing_address);
            $order->shipping_address = $this->formatAddress($request->shipping_address);
            $order->billing_phone = $request->billing_address['contact_phone'];
            $order->shipping_phone = $request->shipping_address['contact_phone'];
            $order->billing_email = $request->billing_address['contact_email'] ?? $user->email;
            
            // Set payment info
            $order->payment_method = $validated['payment_method'];
            $order->notes = $validated['notes'] ?? null;
            
            $order->save();
            
            // Create order items
            foreach ($cart->items as $cartItem) {
                $orderItem = new OrderItem();
                $orderItem->order_id = $order->id;
                $orderItem->product_id = $cartItem->product_id;
                $orderItem->product_name = $cartItem->product->name;
                $orderItem->product_sku = $cartItem->product->sku;
                $orderItem->quantity = $cartItem->quantity;
                $orderItem->unit_price = $cartItem->unit_price;
                $orderItem->subtotal = $cartItem->subtotal;
                $orderItem->discount_amount = $cartItem->discount_amount;
                $orderItem->tax_amount = $cartItem->tax_amount;
                $orderItem->total_amount = $cartItem->total_amount;
                $orderItem->is_bulk_pricing = $cartItem->is_bulk_pricing;
                $orderItem->is_pack_pricing = $cartItem->is_pack_pricing;
                $orderItem->bulk_price_id = $cartItem->bulk_price_id;
                $orderItem->pack_price_id = $cartItem->pack_price_id;
                $orderItem->customer_margin = $cartItem->customer_margin;
                $orderItem->partner_margin = $cartItem->partner_margin;
                $orderItem->customer_margin_type = $cartItem->customer_margin_type;
                $orderItem->partner_margin_type = $cartItem->partner_margin_type;
                
                $orderItem->save();
                
                // Decrement product inventory
                $product = $cartItem->product;
                $product->quantity -= $cartItem->quantity;
                $product->save();
            }
            
            // Create order history entry
            $order->history()->create([
                'status' => 'pending',
                'comment' => 'Order created',
                'user_id' => $user->id
            ]);
            
            // Create invoice
            $invoice = $this->orderService->createInvoice($order);
            
            // Update order with invoice ID
            $order->invoice_id = $invoice->id;
            $order->save();
            
            // Deactivate the cart
            $cart->is_active = false;
            $cart->save();
            
            // Commit transaction
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Order created successfully',
                'data' => [
                    'order' => $order->load(['items'])
                ]
            ], 201);
            
        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollBack();
            
            Log::error('Checkout error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Checkout failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}