<?php

namespace App\Http\Controllers;

use App\Models\Cart;
use App\Models\CartItem;
use App\Models\Product;
use App\Models\ProductBulkPrice;
use App\Models\ProductPackPrice;
use App\Models\Currency;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class CartController extends Controller
{
    /**
     * Helper method to determine which margin to use based on user role
     */
    private function getMarginForUser($cartItem, $user)
    {
        // Debug logging
        Log::info('getMarginForUser called', [
            'user_id' => $user ? $user->id : null,
            'user_role' => $user && $user->role ? $user->role->slug : 'no_role',
            'has_role_partner' => $user ? $user->hasRole('partner') : false,
            'customer_margin' => $cartItem->customer_margin,
            'partner_margin' => $cartItem->partner_margin
        ]);
        
        if ($user && ($user->hasRole('partner') || $user->hasRole('vendor'))) {
            Log::info('Using partner/vendor margin', [
                'margin' => $cartItem->partner_margin,
                'type' => $cartItem->partner_margin_type
            ]);
            return [
                'value' => $cartItem->partner_margin,
                'type' => $cartItem->partner_margin_type
            ];
        }
        
        Log::info('Using customer margin', [
            'margin' => $cartItem->customer_margin,
            'type' => $cartItem->customer_margin_type
        ]);
        return [
            'value' => $cartItem->customer_margin,
            'type' => $cartItem->customer_margin_type
        ];
    }

    /**
     * Get the current user's active cart.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $user = $request->user();
            $user->load(['role', 'customerProfile.membership']); // Load membership
            $userId = $user->id;
            
            // Get active cart or create one
            $cart = Cart::getOrCreateForUser($userId);
            // Load relationships, including bulkPrice and packPrice for delivery_fee
            $cart->load([
                'items.product.brand',
                'items.product.images',
                'items.bulkPrice',
                'items.packPrice',
                'currency',
                'appliedMembership' // Load applied membership
            ]);

            // Force refresh of relationships to ensure they're loaded
            foreach ($cart->items as $item) {
                if ($item->is_bulk_pricing && $item->bulk_price_id) {
                    $item->load('bulkPrice');
                }
                if ($item->is_pack_pricing && $item->pack_price_id) {
                    $item->load('packPrice');
                }
            }

            // Ensure cart totals are calculated with current user's membership and role-based pricing
            $cart->calculateTotals($user);

            // Format the items array
            $formattedItems = $cart->items->map(function ($item) use ($user) {
                $base_unit_price_with_margin = null;
                $special_unit_price_with_margin = null;
                $subtotal = null;
                $quantity = $item->quantity;
                
                // Get appropriate margin based on user role
                $marginData = $this->getMarginForUser($item, $user);
                $margin = $marginData['value'];
                $marginType = $marginData['type'];

                // Use the actual unit_price that was set when the cart item was created
                $base_price = $item->unit_price;
                
                // Check for special pricing
                if ($item->is_bulk_pricing && $item->bulkPrice && $item->bulkPrice->per_pack_special_price) {
                    $special_price = $item->bulkPrice->per_pack_special_price;
                } elseif ($item->is_pack_pricing && $item->packPrice && $item->packPrice->per_pack_special_price) {
                    $special_price = $item->packPrice->per_pack_special_price;
                } else {
                    $special_price = null;
                }

                if ($marginType === 'percentage') {
                    $base_unit_price_with_margin = round($base_price * (1 + ($margin / 100)), 2);
                    if ($special_price && $special_price > 0) {
                        $special_unit_price_with_margin = round($special_price * (1 + ($margin / 100)), 2);
                    }
                } else {
                    $base_unit_price_with_margin = round($base_price + $margin, 2);
                    if ($special_price && $special_price > 0) {
                        $special_unit_price_with_margin = round($special_price + $margin, 2);
                    }
                }

                if ($special_unit_price_with_margin !== null) {
                    $subtotal = round($special_unit_price_with_margin * $quantity, 2);
                } else {
                    $subtotal = round($base_unit_price_with_margin * $quantity, 2);
                }

                // Calculate membership discount for this item
                $membershipDiscount = $this->calculateItemMembershipDiscount($subtotal, $user);

                // Tax rates visibility
                $effective_tax_rate = $item->getTaxRate();
                $product_tax_rate = $item->product ? $item->product->getTaxRate() : null;
                $bulk_tax_rate = ($item->is_bulk_pricing && $item->bulkPrice) ? $item->bulkPrice->tax_rate : null;
                $pack_tax_rate = ($item->is_pack_pricing && $item->packPrice) ? $item->packPrice->tax_rate : null;

                return [
                    'id' => $item->id,
                    'cart_id' => $item->cart_id,
                    'product_id' => $item->product_id,
                    'quantity' => $quantity,
                    'unit_price' => $item->unit_price,
                    'subtotal' => $subtotal,
                    'discount_amount' => $item->discount_amount,
                    'membership_discount_amount' => $membershipDiscount['amount'],
                    'membership_discount_percentage' => $membershipDiscount['percentage'],
                    'price_after_membership_discount' => $subtotal - $membershipDiscount['amount'],
                    'effective_tax_rate' => $effective_tax_rate,
                    'product_tax_rate' => $product_tax_rate,
                    'bulk_tax_rate' => $bulk_tax_rate,
                    'pack_tax_rate' => $pack_tax_rate,
                    'is_bulk_pricing' => $item->is_bulk_pricing,
                    'is_pack_pricing' => $item->is_pack_pricing,
                    'notes' => $item->notes,
                    'customer_margin' => $item->customer_margin,
                    'partner_margin' => $item->partner_margin,
                    'customer_margin_type' => $item->customer_margin_type,
                    'partner_margin_type' => $item->partner_margin_type,
                    'applied_margin' => $margin,
                    'applied_margin_type' => $marginType,
                    'created_at' => $item->created_at,
                    'updated_at' => $item->updated_at,
                    'formatted_quantity' => $item->formatted_quantity,
                    'base_unit_price_with_margin' => $base_unit_price_with_margin,
                    'special_unit_price_with_margin' => $special_unit_price_with_margin,
                    'bulk_price' => $item->bulk_price_id ? \App\Models\ProductBulkPrice::find($item->bulk_price_id) : null,
                    'pack_price' => $item->pack_price_id ? \App\Models\ProductPackPrice::find($item->pack_price_id) : null,
                    'delivery_fee' => $item->delivery_fee,
                    'product' => [
                        'id' => $item->product->id,
                        'name' => $item->product->name,
                        'title' => $item->product->title,
                        'sku' => $item->product->sku,
                        'tax_rate' => $product_tax_rate,
                        'quantity' => $item->product->quantity,
                        'stock_status' => $item->product->stock_status,
                        'images' => $item->product->images
                    ]
                ];
            });

            // Calculate delivery fees from actual cart items
            $total_delivery_fees = $cart->items->sum(function ($item) {
                // Debug logging
                \Log::info('CartItem delivery fee calculation', [
                    'item_id' => $item->id,
                    'is_bulk_pricing' => $item->is_bulk_pricing,
                    'is_pack_pricing' => $item->is_pack_pricing,
                    'bulk_price_id' => $item->bulk_price_id,
                    'pack_price_id' => $item->pack_price_id,
                    'bulk_price_loaded' => $item->relationLoaded('bulkPrice'),
                    'pack_price_loaded' => $item->relationLoaded('packPrice'),
                    'bulk_price' => $item->bulkPrice ? $item->bulkPrice->toArray() : null,
                    'pack_price' => $item->packPrice ? $item->packPrice->toArray() : null,
                ]);
                
                if ($item->is_pack_pricing && $item->packPrice) {
                    $fee = $item->packPrice->delivery_fee ?? 0;
                    \Log::info('Pack pricing delivery fee', ['fee' => $fee]);
                    return $fee;
                } elseif ($item->is_bulk_pricing && $item->bulkPrice) {
                    $fee = $item->bulkPrice->delivery_fee ?? 0;
                    \Log::info('Bulk pricing delivery fee', ['fee' => $fee]);
                    return $fee;
                }
                \Log::info('No delivery fee found', ['item_id' => $item->id]);
                return 0;
            });

            $cart_total_amount = array_sum(array_column($formattedItems->toArray(), 'subtotal'));
            
            // Calculate tax per item based on product-specific tax rates
            $tax_amount = 0;
            foreach ($cart->items as $item) {
                $taxRate = $item->getTaxRate();
                $taxableAmount = $item->subtotal - $item->discount_amount;
                $tax_amount += round($taxableAmount * ($taxRate / 100), 2);
            }
            
            // Delivery fees already calculated above
            
            // Calculate membership discount on total_amount (subtotal)
            $membershipDiscount = $cart->calculateMembershipDiscount($user);
            $membership_discount_amount = $membershipDiscount['amount'];
            $membership_discount_percentage = $membershipDiscount['percentage'];
            
            // Apply membership discount to total_amount, then calculate final price
            $total_amount_after_membership_discount = $cart_total_amount - $membership_discount_amount;
            // Recalculate tax after membership discount using displayed item subtotals
            $tax_amount_after_membership_discount = 0;
            foreach ($formattedItems as $fi) {
                $taxRate = (float) ($fi['effective_tax_rate'] ?? 0);
                $itemSubtotal = (float) $fi['subtotal'];
                $itemDiscount = (float) ($fi['discount_amount'] ?? 0);
                $itemMembershipDiscount = (float) ($fi['membership_discount_amount'] ?? 0);
                $taxableAmountAfterDiscount = $itemSubtotal - $itemDiscount - $itemMembershipDiscount;
                if ($taxableAmountAfterDiscount < 0) { $taxableAmountAfterDiscount = 0; }
                $tax_amount_after_membership_discount += round($taxableAmountAfterDiscount * ($taxRate / 100), 2);
            }
            $total_amount_with_tax = round($total_amount_after_membership_discount + $tax_amount_after_membership_discount, 2);
            $final_price = round($total_amount_after_membership_discount + $tax_amount_after_membership_discount + $total_delivery_fees, 2);

            return response()->json([
                'success' => true,
                'data' => [
                    'cart' => [
                        'id' => $cart->id,
                        'user_id' => $cart->user_id,
                        'session_id' => $cart->session_id,
                        'notes' => $cart->notes,
                        'discount_amount' => round($cart->discount_amount + $membership_discount_amount, 2),
                        'tax_amount' => $tax_amount_after_membership_discount,
                        'total_amount' => $total_amount_after_membership_discount,
                        'currency_code' => $cart->currency_code,
                        'is_active' => $cart->is_active,
                        'expires_at' => $cart->expires_at,
                        'created_at' => $cart->created_at,
                        'updated_at' => $cart->updated_at,
                        'total_amount_with_tax' => $total_amount_with_tax,
                        'total_delivery_fees' => $total_delivery_fees,
                        'final_price' => $final_price,
                        'membership_info' => $user->getMembershipDiscountInfo(),
                        'membership_discount_amount' => $membership_discount_amount,
                        'membership_discount_percentage' => $membership_discount_percentage,
                        'items' => $formattedItems,
                        'currency' => $cart->currency
                    ]
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving cart', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving cart',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get the current user's cart.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCart(Request $request)
    {
        try {
            $user = $request->user();
            $user->load(['role', 'customerProfile.membership']); // Load membership
            $userId = $user->id;
            
            // Get active cart or create one - call the static method directly on the Cart model
            $cart = Cart::getOrCreateForUser($userId);
            
            // Then load the relationships on the retrieved cart
            $cart->load(['items.product.brand', 'items.product.images', 'currency']);
            
            // Ensure cart totals are calculated with current user's membership and role-based pricing
            $cart->calculateTotals($user);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'cart' => $cart
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving cart', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving cart',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Add an item to the cart.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $cartId
     * @return \Illuminate\Http\JsonResponse
     */
    public function addItem(Request $request, $cartId)
    {
        try {
            $validator = Validator::make($request->all(), [
                'product_id' => 'required|exists:products,id',
                'quantity' => 'required|integer|min:1',
                'is_bulk_pricing' => 'nullable|boolean',
                'bulk_price_id' => 'nullable|exists:product_bulk_prices,id',
                'notes' => 'nullable|string',
                'currency_code' => 'nullable|exists:currencies,code'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            $user = $request->user();
            $user->load(['role', 'customerProfile.membership']); // Load membership
            $userId = $user->id;
            $productId = $request->input('product_id');
            $quantity = $request->input('quantity');
            $isBulkPricing = $request->boolean('is_bulk_pricing', false);
            $bulkPriceId = $request->input('bulk_price_id');
            $notes = $request->input('notes');
            $currencyCode = $request->input('currency_code', Currency::getDefaultCode());
            
            // Make sure product exists and is approved
            $product = Product::approved()->active()->find($productId);
            if (!$product) {
                return response()->json([
                    'success' => false,
                    'message' => 'Product not found or not available'
                ], 404);
            }
            
            // Check inventory
            if ($product->quantity < $quantity) {
                return response()->json([
                    'success' => false,
                    'message' => 'Not enough inventory available',
                    'available_quantity' => $product->quantity
                ], 400);
            }

            // Check minimum order quantity
            if ($product->min_order_quantity && $quantity < $product->min_order_quantity) {
                return response()->json([
                    'success' => false,
                    'message' => 'Minimum order quantity not met',
                    'min_order_quantity' => $product->min_order_quantity
                ], 400);
            }
            
            // Get the specified cart
            $cart = Cart::where('id', $cartId)
                ->where('user_id', $userId)
                ->where('is_active', true)
                ->first();
                
            if (!$cart) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart not found or not active'
                ], 404);
            }
            
            // If user is trying to add a product with different currency, return an error
            if ($cart->currency_code !== $currencyCode) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot mix currencies in the same cart. Please complete or empty your existing cart first.'
                ], 400);
            }

            // Check if item already exists in cart with same pricing configuration
            $existingItem = CartItem::where('cart_id', $cart->id)
                ->where('product_id', $product->id)
                ->where('is_bulk_pricing', $isBulkPricing)
                ->where('is_pack_pricing', !$isBulkPricing)
                ->when($isBulkPricing && $bulkPriceId, function($query) use ($bulkPriceId) {
                    return $query->where('bulk_price_id', $bulkPriceId);
                })
                ->first();

            if ($existingItem) {
                // Update existing item
                $existingItem->quantity += $quantity;
                $existingItem->notes = $notes;
                $existingItem->calculatePricing();
                $existingItem->save();
                $cartItem = $existingItem;
            } else {
                // Determine which margin to apply based on user role
                $customerMargin = $product->customer_margin;
                $partnerMargin = $product->partner_margin;
                $customerMarginType = $product->customer_margin_type;
                $partnerMarginType = $product->partner_margin_type;
                
                // If user is a partner, use partner margins; otherwise use customer margins
                if ($user && $user->hasRole('partner')) {
                    $appliedMargin = $partnerMargin;
                    $appliedMarginType = $partnerMarginType;
                } else {
                    $appliedMargin = $customerMargin;
                    $appliedMarginType = $customerMarginType;
                }
                
                // Create new cart item
                $cartItem = new CartItem([
                    'cart_id' => $cart->id,
                    'product_id' => $product->id,
                    'quantity' => $quantity,
                    'notes' => $notes,
                    'is_bulk_pricing' => $isBulkPricing,
                    'is_pack_pricing' => !$isBulkPricing,
                    'bulk_price_id' => $isBulkPricing ? $bulkPriceId : null,
                    'customer_margin' => $customerMargin,
                    'partner_margin' => $partnerMargin,
                    'customer_margin_type' => $customerMarginType,
                    'partner_margin_type' => $partnerMarginType
                ]);

                // Set pricing based on type
                if ($isBulkPricing && $bulkPriceId) {
                    $bulkPrice = ProductBulkPrice::where('id', $bulkPriceId)
                        ->where('product_id', $product->id)
                        ->first();
                    
                    if (!$bulkPrice) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Invalid bulk price for this product'
                        ], 400);
                    }
                    $cartItem->unit_price = $bulkPrice->per_pack_price;
                } elseif (!$isBulkPricing) {
                    $packPrice = $product->packPrice;
                    if (!$packPrice) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Product does not have pack pricing'
                        ], 400);
                    }
                    $cartItem->unit_price = $packPrice->per_pack_price;
                } else {
                    $cartItem->unit_price = $product->price;
                }

                $cartItem->calculatePricing();
                $cartItem->save();
            }

            // After saving the cart item
            $cartItem->refresh();
            $cartItem->load(['bulkPrice', 'packPrice']);

            // Debug logging
            Log::info('CartItem debug', [
                'cart_item_id' => $cartItem->id,
                'bulk_price_id' => $cartItem->bulk_price_id,
                'bulk_price' => $cartItem->bulkPrice,
                'pack_price_id' => $cartItem->pack_price_id,
                'pack_price' => $cartItem->packPrice,
                'delivery_fee' => $cartItem->delivery_fee,
            ]);

            // Recalculate pricing after update
            $cart->calculateTotals($user);
            $cart->load([
                'items.product.brand',
                'items.product.images',
                'items.bulkPrice',
                'items.packPrice',
                'currency'
            ]);

            // Add pricing info to cart_item
            $cartItem->load(['product.packPrice', 'product.bulkPrices', 'product.currency']);

            // Refresh cart with relationships
            $cart->load([
                'items.product.brand',
                'items.product.images',
                'items.bulkPrice',
                'items.packPrice',
                'currency'
            ]);

            // Format the response
            $formattedItems = $cart->items->map(function ($item) use ($user) {
                $base_unit_price_with_margin = null;
                $special_unit_price_with_margin = null;
                $subtotal = null;
                $quantity = $item->quantity;
                
                // Get appropriate margin based on user role
                $marginData = $this->getMarginForUser($item, $user);
                $margin = $marginData['value'];
                $marginType = $marginData['type'];

                // Use the actual unit_price that was set when the cart item was created
                $basePrice = $item->unit_price;
                
                if ($marginType === 'percentage') {
                    $base_unit_price_with_margin = round($basePrice * (1 + ($margin / 100)), 2);
                    // For special pricing, we need to check if there's a special price available
                    if ($item->is_bulk_pricing && $item->bulkPrice && $item->bulkPrice->per_pack_special_price) {
                        $special_unit_price_with_margin = round($item->bulkPrice->per_pack_special_price * (1 + ($margin / 100)), 2);
                    } elseif ($item->is_pack_pricing && $item->packPrice && $item->packPrice->per_pack_special_price) {
                        $special_unit_price_with_margin = round($item->packPrice->per_pack_special_price * (1 + ($margin / 100)), 2);
                    } else {
                        $special_unit_price_with_margin = null;
                    }
                } else {
                    $base_unit_price_with_margin = round($basePrice + $margin, 2);
                    // For special pricing, we need to check if there's a special price available
                    if ($item->is_bulk_pricing && $item->bulkPrice && $item->bulkPrice->per_pack_special_price) {
                        $special_unit_price_with_margin = round($item->bulkPrice->per_pack_special_price + $margin, 2);
                    } elseif ($item->is_pack_pricing && $item->packPrice && $item->packPrice->per_pack_special_price) {
                        $special_unit_price_with_margin = round($item->packPrice->per_pack_special_price + $margin, 2);
                    } else {
                        $special_unit_price_with_margin = null;
                    }
                }

                if ($special_unit_price_with_margin !== null) {
                    $subtotal = round($special_unit_price_with_margin * $quantity, 2);
                } else {
                    $subtotal = round($base_unit_price_with_margin * $quantity, 2);
                }

                // Calculate membership discount for this item
                $membershipDiscount = $this->calculateItemMembershipDiscount($subtotal, $user);
                
                // Tax rates visibility
                $effective_tax_rate = $item->getTaxRate();
                $product_tax_rate = $item->product ? $item->product->getTaxRate() : null;
                $bulk_tax_rate = ($item->is_bulk_pricing && $item->bulkPrice) ? $item->bulkPrice->tax_rate : null;
                $pack_tax_rate = ($item->is_pack_pricing && $item->packPrice) ? $item->packPrice->tax_rate : null;

                return [
                    'id' => $item->id,
                    'cart_id' => $item->cart_id,
                    'product_id' => $item->product_id,
                    'quantity' => $quantity,
                    'unit_price' => $item->unit_price,
                    'subtotal' => $subtotal,
                    'discount_amount' => $item->discount_amount,
                    'membership_discount_amount' => $membershipDiscount['amount'],
                    'membership_discount_percentage' => $membershipDiscount['percentage'],
                    'price_after_membership_discount' => $subtotal - $membershipDiscount['amount'],
                    'effective_tax_rate' => $effective_tax_rate,
                    'product_tax_rate' => $product_tax_rate,
                    'bulk_tax_rate' => $bulk_tax_rate,
                    'pack_tax_rate' => $pack_tax_rate,
                    'is_bulk_pricing' => $item->is_bulk_pricing,
                    'is_pack_pricing' => $item->is_pack_pricing,
                    'notes' => $item->notes,
                    'customer_margin' => $item->customer_margin,
                    'partner_margin' => $item->partner_margin,
                    'customer_margin_type' => $item->customer_margin_type,
                    'partner_margin_type' => $item->partner_margin_type,
                    'applied_margin' => $margin,
                    'applied_margin_type' => $marginType,
                    'created_at' => $item->created_at,
                    'updated_at' => $item->updated_at,
                    'formatted_quantity' => $item->formatted_quantity,
                    'base_unit_price_with_margin' => $base_unit_price_with_margin,
                    'special_unit_price_with_margin' => $special_unit_price_with_margin,
                    'bulk_price' => $item->bulk_price_id ? \App\Models\ProductBulkPrice::find($item->bulk_price_id) : null,
                    'pack_price' => $item->pack_price_id ? \App\Models\ProductPackPrice::find($item->pack_price_id) : null,
                    'delivery_fee' => $item->delivery_fee,
                    'product' => [
                        'id' => $item->product->id,
                        'name' => $item->product->name,
                        'title' => $item->product->title,
                        'sku' => $item->product->sku,
                        'tax_rate' => $item->product->tax_rate,
                        'quantity' => $item->product->quantity,
                        'stock_status' => $item->product->stock_status,
                        'images' => $item->product->images
                    ]
                ];
            });

            // Calculate total_amount as the sum of all item subtotals
            $cart_total_amount = $formattedItems->sum('subtotal');
            // Sum all item delivery fees (treat null as 0)
            $total_delivery_fees = $formattedItems->sum(function ($item) {
                return isset($item['delivery_fee']) ? $item['delivery_fee'] : 0;
            });
            
            // Calculate membership discount on total_amount (subtotal)
            $membershipDiscount = $cart->calculateMembershipDiscount($user);
            $membership_discount_amount = $membershipDiscount['amount'];
            $membership_discount_percentage = $membershipDiscount['percentage'];
            
            // Apply membership discount to total_amount
            $total_amount_after_membership_discount = $cart_total_amount - $membership_discount_amount;
            
            // Recalculate tax after membership discount using displayed item subtotals
            $tax_amount_after_membership_discount = 0;
            foreach ($formattedItems as $fi) {
                $taxRate = (float) ($fi['effective_tax_rate'] ?? 0);
                $itemSubtotal = (float) $fi['subtotal'];
                $itemDiscount = (float) ($fi['discount_amount'] ?? 0);
                $itemMembershipDiscount = (float) ($fi['membership_discount_amount'] ?? 0);
                $taxableAmountAfterDiscount = $itemSubtotal - $itemDiscount - $itemMembershipDiscount;
                if ($taxableAmountAfterDiscount < 0) { $taxableAmountAfterDiscount = 0; }
                $tax_amount_after_membership_discount += round($taxableAmountAfterDiscount * ($taxRate / 100), 2);
            }
            $total_amount_with_tax = round($total_amount_after_membership_discount + $tax_amount_after_membership_discount, 2);
            $final_price = round($total_amount_after_membership_discount + $tax_amount_after_membership_discount + $total_delivery_fees, 2);

            list($formattedItems, $deliveryFees) = $this->fillCartItemPricingDetails($formattedItems->toArray());

            return response()->json([
                'success' => true,
                'message' => 'Item added to cart',
                'data' => [
                    'cart' => [
                        'id' => $cart->id,
                        'user_id' => $cart->user_id,
                        'session_id' => $cart->session_id,
                        'notes' => $cart->notes,
                        'discount_amount' => round($cart->discount_amount + $membership_discount_amount, 2),
                        'tax_amount' => $tax_amount_after_membership_discount,
                        'total_amount' => $total_amount_after_membership_discount,
                        'currency_code' => $cart->currency_code,
                        'is_active' => $cart->is_active,
                        'expires_at' => $cart->expires_at,
                        'created_at' => $cart->created_at,
                        'updated_at' => $cart->updated_at,
                        'total_amount_with_tax' => $total_amount_with_tax,
                        'total_delivery_fees' => $total_delivery_fees,
                        'final_price' => $final_price,
                        'membership_info' => $user->getMembershipDiscountInfo(),
                        'membership_discount_amount' => $membership_discount_amount,
                        'membership_discount_percentage' => $membership_discount_percentage,
                        'currency_code' => $cart->currency_code,
                        'currency' => $cart->currency
                    ],
                    'items' => $formattedItems
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error adding item to cart', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error adding item to cart',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Update an item in the cart.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $cartId
     * @param  int  $itemId
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateItem(Request $request, $cartId, $itemId)
    {
        try {
            $validator = Validator::make($request->all(), [
                'quantity' => 'required|integer|min:1',
                'notes' => 'nullable|string'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            $userId = $request->user()->id;
            $quantity = $request->input('quantity');
            $notes = $request->input('notes');
            
            // First check if cart exists and belongs to user
            $cart = Cart::where('id', $cartId)
                ->where('user_id', $userId)
                ->first();
                
            if (!$cart) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart not found'
                ], 404);
            }
            
            // Then find the cart item
            $cartItem = CartItem::where('cart_id', $cartId)
                ->where('id', $itemId)
                ->first();
                
            if (!$cartItem) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart item not found',
                    'details' => [
                        'cart_id' => $cartId,
                        'item_id' => $itemId
                    ]
                ], 404);
            }
            
            // Check inventory
            $product = $cartItem->product;
            if ($product->quantity < $quantity) {
                return response()->json([
                    'success' => false,
                    'message' => 'Not enough inventory available',
                    'available_quantity' => $product->quantity
                ], 400);
            }

            // Check minimum order quantity
            if ($product->min_order_quantity && $quantity < $product->min_order_quantity) {
                return response()->json([
                    'success' => false,
                    'message' => 'Minimum order quantity not met',
                    'min_order_quantity' => $product->min_order_quantity
                ], 400);
            }
            
            // Update cart item
            $cartItem->quantity = $quantity;
            if ($notes !== null) {
                $cartItem->notes = $notes;
            }
            $cartItem->calculatePricing();
            
            // Recalculate cart totals with membership discount
            $user = $request->user();
            $user->load(['role', 'customerProfile.membership']);
            $cart->calculateTotals($user);
            
            $cart = $cart->fresh(['items.product.brand', 'items.product.images', 'currency']);

            // Add pricing info to cart_item
            $cartItem->load(['product.packPrice', 'product.bulkPrices', 'product.currency']);
            $cartItemData = $cartItem->toArray();
            $user = $cart->user ?? $request->user();
            if ($user && $user->hasRole('partner')) {
                $marginAmount = $cartItem->partner_margin_amount;
                $marginType = $cartItem->partner_margin_type;
            } else {
                $marginAmount = $cartItem->customer_margin_amount;
                $marginType = $cartItem->customer_margin_type;
            }
            $cartItemData['unit_price'] = (float) $cartItem->unit_price;
            $cartItemData['margin_amount'] = (float) $marginAmount;
            $cartItemData['margin_type'] = $marginType;
            $cartItemData['total_price'] = ($cartItem->unit_price + $marginAmount) * $cartItem->quantity;
            $cartItemData['pack_price'] = $cartItem->product->packPrice;
            $cartItemData['bulk_prices'] = $cartItem->product->bulkPrices;
            $cartItemData['currency'] = $cartItem->product->currency;

            return response()->json([
                'success' => true,
                'message' => 'Cart item updated',
                'data' => [
                    'cart' => $cart,
                    'cart_item' => $cartItemData
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error updating cart item', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'cart_id' => $cartId,
                'item_id' => $itemId
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating cart item',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Remove an item from the cart.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $cartId
     * @param  int  $itemId
     * @return \Illuminate\Http\JsonResponse
     */
    public function removeItem(Request $request, $cartId, $itemId)
    {
        try {
            $userId = $request->user()->id;
            Log::info('Attempting to delete cart item', [
                'cart_id' => $cartId,
                'item_id' => $itemId,
                'user_id' => $userId
            ]);

            // Find the cart item by cart and item id, and ensure cart belongs to user
            $cartItem = CartItem::where('id', $itemId)
                ->where('cart_id', $cartId)
                ->whereHas('cart', function($query) use ($userId) {
                    $query->where('user_id', $userId);
                })
                ->first();

            Log::info('Cart item query result', [
                'found' => $cartItem ? true : false,
                'cart_item_id' => $cartItem ? $cartItem->id : null,
                'cart_id' => $cartItem ? $cartItem->cart_id : null
            ]);

            if (!$cartItem) {
                Log::warning('Cart item not found for deletion', [
                    'cart_id' => $cartId,
                    'item_id' => $itemId,
                    'user_id' => $userId
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Cart item not found'
                ], 404);
            }

            $cart = $cartItem->cart;

            // Delete the cart item
            $cartItem->delete();

            // Recalculate cart totals
            $user = $request->user();
            $user->load(['role', 'customerProfile.membership']); // Load membership
            $cart->calculateTotals($user);
            $cart->load(['items.product.brand', 'items.product.images', 'currency']);

            Log::info('Cart item deleted successfully', [
                'cart_id' => $cartId,
                'item_id' => $itemId,
                'user_id' => $userId
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Item removed from cart',
                'data' => [
                    'cart' => $cart
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error removing item from cart', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'cart_id' => $cartId,
                'item_id' => $itemId,
                'user_id' => $request->user()->id
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error removing item from cart',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Clear the cart.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function clearCart(Request $request)
    {
        try {
            $userId = $request->user()->id;
            
            // Find the cart
            $cart = Cart::forUser($userId)->active()->notExpired()->first();
            
            if (!$cart) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active cart found'
                ], 404);
            }
            
            // Delete all cart items
            CartItem::where('cart_id', $cart->id)->delete();
            
            // Reset cart totals
            $cart->subtotal = 0;
            $cart->discount_amount = 0;
            $cart->tax_amount = 0;
            $cart->shipping_amount = 0;
            $cart->total_amount = 0;
            $cart->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Cart cleared successfully',
                'data' => [
                    'cart' => $cart
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error clearing cart', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error clearing cart',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Update cart notes.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateNotes(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'notes' => 'nullable|string'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            $userId = $request->user()->id;
            $notes = $request->input('notes');
            
            // Find the cart
            $cart = Cart::forUser($userId)->active()->notExpired()->first();
            
            if (!$cart) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active cart found'
                ], 404);
            }
            
            // Update notes
            $cart->notes = $notes;
            $cart->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Cart notes updated',
                'data' => [
                    'cart' => $cart
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error updating cart notes', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating cart notes',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Apply shipping amount to cart.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateShipping(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'shipping_amount' => 'required|numeric|min:0'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            $userId = $request->user()->id;
            $shippingAmount = $request->input('shipping_amount');
            
            // Find the cart
            $cart = Cart::forUser($userId)->active()->notExpired()->first();
            
            if (!$cart) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active cart found'
                ], 404);
            }
            
            // Update shipping
            $cart->shipping_amount = $shippingAmount;
            $user = $request->user();
            $user->load('role'); // Ensure role is loaded
            $cart->calculateTotals($user);
            
            $cart->load(['items.product.brand', 'items.product.images', 'currency']);
            
            return response()->json([
                'success' => true,
                'message' => 'Shipping updated',
                'data' => [
                    'cart' => $cart
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error updating shipping', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating shipping',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Create a new empty cart for the authenticated user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function create(Request $request)
    {
        try {
            $user = $request->user();
            
            // Check if user already has an active cart
            $existingCart = Cart::where('user_id', $user->id)
                ->where('is_active', true)
                ->first();
                
            if ($existingCart) {
                return response()->json([
                    'success' => true,
                    'message' => 'Active cart already exists',
                    'data' => [
                        'cart' => $existingCart->load(['items.product', 'currency'])
                    ]
                ]);
            }
            
            // Create new cart
            $cart = new Cart();
            $cart->user_id = $user->id;
            $cart->is_active = true;
            $cart->currency_code = 'AUD'; // Default currency
            $cart->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Cart created successfully',
                'data' => [
                    'cart' => $cart->load(['items.product', 'currency'])
                ]
            ], 201);
            
        } catch (\Exception $e) {
            Log::error('Cart creation error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to create cart',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete the specified cart.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $cartId
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(Request $request, $cartId)
    {
        try {
            $userId = $request->user()->id;
            
            // Find the cart
            $cart = Cart::where('user_id', $userId)
                ->where('id', $cartId)
                ->first();
                
            if (!$cart) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart not found'
                ], 404);
            }
            
            // Delete all cart items first
            CartItem::where('cart_id', $cart->id)->delete();
            
            // Delete the cart
            $cart->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Cart deleted successfully'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error deleting cart', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error deleting cart',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get a specific cart.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $cartId
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Request $request, $id)
    {
        $cart = Cart::with(['items.product.images', 'currency'])
            ->findOrFail($id);

        $user = $request->user();
        $user->load('role'); // Ensure role is loaded

        // Format the response to use only the correct fields from CartItem model
        $formattedItems = $cart->items->map(function ($item) use ($user) {
            // Get appropriate margin based on user role
            $marginData = $this->getMarginForUser($item, $user);
            $margin = $marginData['value'];
            $marginType = $marginData['type'];
            
            // Calculate base unit price with margin
            $basePrice = $item->unit_price;
            if ($marginType === 'percentage') {
                $base_unit_price_with_margin = round($basePrice * (1 + ($margin / 100)), 2);
            } else {
                $base_unit_price_with_margin = round($basePrice + $margin, 2);
            }
            
            return [
                'id' => $item->id,
                'cart_id' => $item->cart_id,
                'product_id' => $item->product_id,
                'quantity' => $item->quantity,
                'unit_price' => $item->unit_price,
                'subtotal' => $item->subtotal,
                'discount_amount' => $item->discount_amount,
                'is_bulk_pricing' => $item->is_bulk_pricing,
                'is_pack_pricing' => $item->is_pack_pricing,
                'notes' => $item->notes,
                                    'customer_margin' => $item->customer_margin,
                    'partner_margin' => $item->partner_margin,
                    'customer_margin_type' => $item->customer_margin_type,
                    'partner_margin_type' => $item->partner_margin_type,
                    'applied_margin' => $margin,
                    'applied_margin_type' => $marginType,
                    'created_at' => $item->created_at,
                'updated_at' => $item->updated_at,
                'formatted_quantity' => $item->formatted_quantity,
                'base_unit_price_with_margin' => $base_unit_price_with_margin,
                'special_unit_price_with_margin' => null,
                'product' => [
                    'id' => $item->product->id,
                    'name' => $item->product->name,
                    'title' => $item->product->title,
                    'sku' => $item->product->sku,
                    'quantity' => $item->product->quantity,
                    'stock_status' => $item->product->stock_status,
                    'images' => $item->product->images->map(function ($image) {
                        return [
                            'id' => $image->id,
                            'product_id' => $image->product_id,
                            'image_url' => $image->image_url,
                            'alt_text' => $image->alt_text,
                            'sort_order' => $image->sort_order,
                            'created_at' => $image->created_at,
                            'updated_at' => $image->updated_at,
                            'full_image_url' => $image->full_image_url
                        ];
                    })
                ],
                'pack_price' => $item->is_pack_pricing ? $item->pack_price : null,
                'bulk_price' => $item->is_bulk_pricing ? $item->bulk_price : null
            ];
        });

        // Calculate cart totals with role-based margins
        $total_amount = 0;
        foreach ($cart->items as $item) {
            $marginData = $this->getMarginForUser($item, $user);
            $margin = $marginData['value'];
            $marginType = $marginData['type'];
            
            $basePrice = $item->unit_price;
            if ($marginType === 'percentage') {
                $priceWithMargin = $basePrice * (1 + ($margin / 100));
            } else {
                $priceWithMargin = $basePrice + $margin;
            }
            
            $total_amount += $priceWithMargin * $item->quantity;
        }
        $total_delivery_fees = $cart->items->sum(function ($item) {
            if ($item->is_pack_pricing && $item->pack_price) {
                return $item->pack_price->delivery_fee ?? 0;
            } elseif ($item->is_bulk_pricing && $item->bulk_price) {
                return $item->bulk_price->delivery_fee ?? 0;
            }
            return 0;
        });
        // Apply membership discount then recompute tax for show endpoint
        $user->load(['customerProfile.membership']);
        $membership = $user->getActiveMembership();
        $membershipPercentage = $membership ? $membership->extra_discount : 0;
        $subtotalAfterMembership = round($total_amount - (($total_amount * $membershipPercentage) / 100), 2);
        // Approximate tax: recompute proportionally by items (fallback here uses 10% if item rates unavailable)
        $taxAfterMembership = 0;
        foreach ($cart->items as $item) {
            $taxRate = method_exists($item, 'getTaxRate') ? (float) $item->getTaxRate() : 10.0;
            $itemSubtotal = (float) $item->subtotal;
            $itemMembershipDiscount = round($itemSubtotal * ($membershipPercentage / 100), 2);
            $taxable = $itemSubtotal - $item->discount_amount - $itemMembershipDiscount;
            if ($taxable < 0) { $taxable = 0; }
            $taxAfterMembership += round($taxable * ($taxRate / 100), 2);
        }
        $total_amount_with_tax = round($subtotalAfterMembership + $taxAfterMembership, 2);
        $final_price = round($total_amount_with_tax + $total_delivery_fees, 2);

        return response()->json([
            'success' => true,
            'data' => [
                'cart' => [
                    'id' => $cart->id,
                    'user_id' => $cart->user_id,
                    'session_id' => $cart->session_id,
                    'notes' => $cart->notes,
                    'discount_amount' => $cart->discount_amount,
                    'tax_amount' => $cart->tax_amount,
                    'total_amount' => $subtotalAfterMembership,
                    'currency_code' => $cart->currency_code,
                    'is_active' => $cart->is_active,
                    'expires_at' => $cart->expires_at,
                    'created_at' => $cart->created_at,
                    'updated_at' => $cart->updated_at,
                    'total_amount_with_tax' => $total_amount_with_tax,
                    'total_delivery_fees' => $total_delivery_fees,
                    'final_price' => $final_price,
                    'currency' => $cart->currency,
                    'items' => $formattedItems
                ]
            ]
        ]);
    }

    private function fillCartItemPricingDetails($items)
    {
        // This method is no longer needed as we calculate delivery fees directly from cart items
        // Keeping for backward compatibility but it's not used in the main flow
        return [$items, []];
    }

    /**
     * Calculate membership discount for a specific item.
     */
    private function calculateItemMembershipDiscount($itemSubtotal, $user)
    {
        if (!$user) return ['amount' => 0, 'percentage' => 0];
        
        $membership = $user->getActiveMembership();
        if (!$membership) return ['amount' => 0, 'percentage' => 0];
        
        $discountPercentage = $membership->extra_discount;
        $discountAmount = ($itemSubtotal * $discountPercentage) / 100;
        
        return [
            'amount' => round($discountAmount, 2),
            'percentage' => $discountPercentage
        ];
    }
}