<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Role;
use App\Models\SocialLogin;
use App\Models\User;
use App\Models\VendorProfile;
use App\Services\TokenService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Laravel\Socialite\Facades\Socialite;

class AuthController extends Controller
{
    protected $tokenService;

    public function __construct(TokenService $tokenService)
    {
        $this->tokenService = $tokenService;
    }
    /**
     * Register a new user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'username' => 'nullable|string|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'user_type' => 'required|in:buyer,vendor', // Only allow buyer or vendor in registration
            'company_name' => 'nullable|string|max:255',
            'company_description' => 'nullable|string',
            'business_registration_number' => 'nullable|string|max:100',
            'company_website' => 'nullable|url|max:255',
            'tax_identification_number' => 'nullable|string|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Get the role based on user type
        $role = Role::where('slug', $request->user_type)->first();
        
        if (!$role) {
            return response()->json(['message' => 'Invalid user type'], 400);
        }

        // Create the user
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'username' => $request->username,
            'password' => Hash::make($request->password),
            'phone' => $request->phone,
            'address' => $request->address,
            'role_id' => $role->id,
        ]);

        // If user is a vendor, create vendor profile
        if ($request->user_type === 'vendor') {
            VendorProfile::create([
                'user_id' => $user->id,
                'company_name' => $request->company_name ?? $request->name . '\'s Company',
                'company_description' => $request->company_description ?? null,
                'business_registration_number' => $request->business_registration_number ?? null,
                'company_website' => $request->company_website ?? null,
                'tax_identification_number' => $request->tax_identification_number ?? null,
                'is_approved' => true, // Auto-approve vendor
            ]);
        }

        try {
            // Extract device information
            $deviceInfo = $this->tokenService->extractDeviceInfo($request);
            
            // Create tokens using TokenService
            $tokens = $this->tokenService->createTokens($user, $deviceInfo, false); // Don't revoke previous tokens for registration

            // Notify all admins and superadmins
            $adminUsers = User::whereHas('role', function($q) {
                $q->whereIn('slug', ['admin', 'superadmin']);
            })->get();
            foreach ($adminUsers as $admin) {
                $admin->notify(new \App\Notifications\UserSignedUpNotification($user));
            }

            return response()->json([
                'message' => 'User registered successfully',
                'data' => $tokens
            ]);

        } catch (\Exception $e) {
            Log::error('Registration failed', [
                'email' => $request->email,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'message' => 'Registration failed. Please try again.'
            ], 500);
        }
    }

    /**
     * Login user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required_without:username|string|email|max:255',
            'username' => 'required_without:email|string|max:255',
            'password' => 'required|string|min:8',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Check if login is using email or username
        $credentials = $request->has('email') 
            ? ['email' => $request->email, 'password' => $request->password]
            : ['username' => $request->username, 'password' => $request->password];

        if (!Auth::attempt($credentials)) {
            return response()->json(['message' => 'Invalid credentials'], 401);
        }

        $user = User::where(array_key_first($credentials), current($credentials))->first();

        // Check if user is active
        if (!$user->is_active) {
            return response()->json(['message' => 'Your account is inactive. Please contact admin.'], 403);
        }

        // Check if vendor is approved
        if ($user->isVendor() && !$user->vendorProfile->is_approved) {
            return response()->json(['message' => 'Your vendor account is pending approval.'], 403);
        }

        try {
            // Extract device information
            $deviceInfo = $this->tokenService->extractDeviceInfo($request);
            
            // Create tokens using TokenService
            $tokens = $this->tokenService->createTokens($user, $deviceInfo);

            return response()->json([
                'message' => 'Login successful',
                'data' => $tokens
            ]);

        } catch (\Exception $e) {
            Log::error('Login failed', [
                'user_id' => $user->id,
                'email' => $user->email,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'message' => 'Login failed. Please try again.'
            ], 500);
        }
    }

    /**
     * Logout user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {
        try {
            $user = $request->user();
            $revokeAll = $request->boolean('revoke_all', false);
            
            $result = $this->tokenService->revokeTokens(
                $user, 
                $request->refresh_token, 
                $revokeAll
            );

            return response()->json($result);

        } catch (\Exception $e) {
            Log::error('Logout failed', [
                'user_id' => $request->user()->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'message' => 'Logout failed. Please try again.'
            ], 500);
        }
    }

    /**
     * Refresh access token using refresh token.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function refresh(Request $request)
    {
        try {
            $request->validate([
                'refresh_token' => 'required|string'
            ]);

            $tokens = $this->tokenService->refreshAccessToken($request->refresh_token);

            return response()->json([
                'message' => 'Token refreshed successfully',
                'data' => $tokens
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'message' => 'Invalid refresh token',
                'errors' => $e->errors()
            ], 401);
        } catch (\Exception $e) {
            Log::error('Token refresh failed', [
                'error' => $e->getMessage(),
                'ip_address' => $request->ip()
            ]);

            return response()->json([
                'message' => 'Token refresh failed. Please try again.'
            ], 500);
        }
    }

    /**
     * Get user's active sessions.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function sessions(Request $request)
    {
        try {
            $user = $request->user();
            $sessions = $this->tokenService->getUserSessions($user);

            return response()->json($sessions);

        } catch (\Exception $e) {
            Log::error('Failed to get user sessions', [
                'user_id' => $request->user()->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'message' => 'Failed to retrieve sessions.'
            ], 500);
        }
    }

    /**
     * Revoke a specific session.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function revokeSession(Request $request)
    {
        try {
            $request->validate([
                'session_id' => 'required|integer'
            ]);

            $user = $request->user();
            $result = $this->tokenService->revokeSession($user, $request->session_id);

            return response()->json($result);

        } catch (ValidationException $e) {
            return response()->json([
                'message' => 'Invalid session ID',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Failed to revoke session', [
                'user_id' => $request->user()->id,
                'session_id' => $request->session_id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'message' => 'Failed to revoke session.'
            ], 500);
        }
    }

    /**
     * Get token statistics for the authenticated user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function tokenStats(Request $request)
    {
        try {
            $user = $request->user();
            $stats = $this->tokenService->getTokenStats($user);

            return response()->json($stats);

        } catch (\Exception $e) {
            Log::error('Failed to get token stats', [
                'user_id' => $request->user()->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'message' => 'Failed to retrieve token statistics.'
            ], 500);
        }
    }

    /**
     * Change user password.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function changePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'current_password' => 'required|string',
            'password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = $request->user();

        // Verify current password
        if (!Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect'
            ], 401);
        }

        // Update password
        $user->password = Hash::make($request->password);
        $user->save();

        // Revoke all existing tokens for security
        $user->tokens()->delete();

        // Create new token
        $token = $user->createToken('auth_token')->plainTextToken;

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully',
            'token' => $token // Return new token since we revoked the old one
        ]);
    }

    /**
     * Get the authenticated user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function me(Request $request)
    {
        $user = $request->user();
        
        // Load relationships based on user type
        if ($user->isVendor()) {
            $user->load('vendorProfile');
        }
        
        $user->load('role');

        return response()->json(['user' => $user]);
    }

    /**
     * Redirect the user to the provider for authentication.
     *
     * @param  string  $provider
     * @return \Illuminate\Http\Response
     */
    public function redirectToProvider($provider)
    {
        // Validate provider
        if (!in_array($provider, ['facebook', 'google'])) {
            return response()->json(['message' => 'Invalid provider'], 400);
        }

        return response()->json([
            'url' => Socialite::driver($provider)->stateless()->redirect()->getTargetUrl(),
        ]);
    }

    /**
     * Handle provider callback.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  string  $provider
     * @return \Illuminate\Http\Response
     */
    public function handleProviderCallback(Request $request, $provider)
    {
        // Validate provider
        if (!in_array($provider, ['facebook', 'google'])) {
            return response()->json(['message' => 'Invalid provider'], 400);
        }

        try {
            $providerUser = Socialite::driver($provider)->stateless()->user();
            
            // Find existing social login or create a new one
            $socialLogin = SocialLogin::where('provider', $provider)
                ->where('provider_id', $providerUser->getId())
                ->first();
            
            if ($socialLogin) {
                // Login existing user
                $user = $socialLogin->user;
                
                // Update token
                $socialLogin->update([
                    'provider_token' => $providerUser->token,
                    'provider_refresh_token' => $providerUser->refreshToken,
                ]);
            } else {
                // Check if user exists with the email
                $user = User::where('email', $providerUser->getEmail())->first();
                
                if (!$user) {
                    // Get buyer role
                    $role = Role::where('slug', 'buyer')->first();
                    
                    // Create new user
                    $user = User::create([
                        'name' => $providerUser->getName(),
                        'email' => $providerUser->getEmail(),
                        'role_id' => $role->id,
                        'is_active' => true,
                    ]);
                }
                
                // Create social login
                $user->socialLogins()->create([
                    'provider' => $provider,
                    'provider_id' => $providerUser->getId(),
                    'provider_token' => $providerUser->token,
                    'provider_refresh_token' => $providerUser->refreshToken,
                ]);
            }
            
            // Check if user is active
            if (!$user->is_active) {
                return response()->json(['message' => 'Your account is inactive. Please contact admin.'], 403);
            }
            
            // Generate token
            $token = $user->createToken('auth_token')->plainTextToken;
            
            return response()->json([
                'message' => 'Login successful',
                'user' => $user,
                'token' => $token,
            ]);
            
        } catch (\Exception $e) {
            return response()->json(['message' => 'Authentication failed: ' . $e->getMessage()], 500);
        }
    }
}